/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.resources.database.internal.InternalDatabase;
/**
 * This class updates the database with the given list of updates.
 */
public class UpdateCommand extends DBCommand {
	protected InternalDatabase database;
	protected List updates, delayed;
	protected Set addedObjects;
	protected WeakObjectCache cache;
	public UpdateCommand(InternalDatabase database, List updates, WeakObjectCache cache) {
		super(database.getJDBCHelper(), database.getDBMap());
		this.database = database;
		this.updates = updates;
		this.cache = cache;
		delayed = new ArrayList();
	}
	public Object execute() throws Exception {
		if (updates == null || updates.isEmpty())
			return null;
		try {
			PerfUtil p = new PerfUtil("UpdateCommand.execute() 1",true);
			addObjects();
			p.stopAndPrintStatus();
			p.setMessageAndStart("UpdateCommand.execute() 2");
			setReferences();
			p.stopAndPrintStatus();
			p.setMessageAndStart("UpdateCommand.execute() 3");
			setAttributes();
			p.stopAndPrintStatus();
			updates.clear();
			updates.addAll(delayed);
			helper.commitTransaction();
		} catch (Exception e) {
			helper.rollbackTransaction();
			throw e;
		}
		
		return null;
	}
	protected void addObjects() throws Exception {
		addedObjects = new HashSet();
		PerfUtil p = new PerfUtil("UpdateCommand.addObjects() 1 size="+updates.size(),true);
		for (int i = 0, l = updates.size(); i < l; i++) {
			Notification notification = (Notification) updates.get(i);
			if (addObject(notification)) {
				EObject addedObject = (EObject) notification.getNewValue();
				Integer id = null;
				if (cache != null && !addedObjects.contains(addedObject))
					id = cache.getId(addedObject);
				if (id == null) {
					addedObjects.add(addedObject);
				}
			}
		}
		p.stopAndPrintStatus();
		if (!addedObjects.isEmpty())
		{
			p.setMessageAndStart("UpdateCommand.addObjects() 2 size="+updates.size());
			database.add(new ArrayList(addedObjects));
			p.stopAndPrintStatus();
		}
	}
	protected boolean addObject(Notification notification) {
		if (notification.getEventType() != Notification.ADD)
			return false;
		EStructuralFeature feature = (EStructuralFeature) notification.getFeature();
		if (!(feature instanceof EReference))
			return false;
		return ((EReference) feature).isContainment();
	}
	protected void setReferences() throws Exception {
		setReferencesFromAddedObjects();
		setReferencesFromUpdates();
	}
	/**
	 * Set references from added objects to objects that have already been
	 * added; otherwise, these references will not be set.
	 */
	protected void setReferencesFromAddedObjects() throws Exception {
		PerfUtil p = new PerfUtil("UpdateCommand.setReferencesFromAddedObjects 1 addedObjects.size="+addedObjects.size(),true);

		Iterator i = addedObjects.iterator();
		while (i.hasNext()) {
			EObject object = (EObject) i.next();
			List setReferences = getSetReferences(object);
			setReferencesIfNeeded(object, setReferences);
		}
		p.stopAndPrintStatus();
	}
	protected List getSetReferences(EObject object) {
		List setReferences = new ArrayList();
		List allReferences = object.eClass().getEAllReferences();
		for (int i = 0, l = allReferences.size(); i < l; i++) {
			EReference reference = (EReference) allReferences.get(i);
			if (!reference.isContainment() && !reference.isContainer() && object.eIsSet(reference))
				setReferences.add(reference);
		}
		return setReferences;
	}
	protected void setReferencesIfNeeded(EObject object, List references) throws Exception {
		for (int i = 0, l = references.size(); i < l; i++) {
			EReference reference = (EReference) references.get(i);
			if (reference.isMany())
				setManyValuedReference(object, reference);
			else
				setSingleValuedReference(object, reference);
		}
	}
	protected void setManyValuedReference(EObject object, EReference reference) throws Exception {
		Integer sourceId = getObjectId(object);
		if(sourceId == null)
			return;
		List values = (List) object.eGet(reference, false);
		List targets = new ArrayList();
		List indexes = new ArrayList();
		PerfUtil p = new PerfUtil("UpdateCommand.setManyValuedReference() 1 size="+values.size(),true);
		for (int j = 0, l2 = values.size(); j < l2; j++) {
			EObject value = (EObject) values.get(j);
			if (!addedObjects.contains(value)) {
				if(usePreparedStatement)
				{
					Integer targetId = getObjectId(value);
					if (targetId == null) 
						continue;
					targets.add(targetId);
					indexes.add(new Integer(j));
				}
				else
				{
					database.setReference(object, reference,value,j);
				}
			}
		}
		p.stopAndPrintStatus();
		if(usePreparedStatement)
		{
			if (indexes.size() > 0) {
				p.setMessageAndStart("UpdateCommand.setManyValuedReference() 2 size="+indexes.size());
				database.setReference(sourceId, reference, targets, indexes);
				p.stopAndPrintStatus();
			}
		}
	}
	protected void setSingleValuedReference(EObject object, EReference reference) throws Exception {
		EObject value = (EObject) object.eGet(reference, false);
		if (!addedObjects.contains(value))
		{
			PerfUtil p = new PerfUtil("UpdateCommand.setSingleValuedReference()",true);		
			database.setReference(object, reference, value, -1);
			p.stopAndPrintStatus();
		}
	}
	protected Integer getObjectId(EObject object) throws Exception {
		if (cache == null)
			return null;
		else
			return cache.getId(object);
	}
	protected void setReferencesFromUpdates() throws Exception {
		Map objToReferences = new HashMap();
		Map referenceToValues;
		PerfUtil p = new PerfUtil("UpdateCommand.setReferencesFromUpdates() 1 size="+updates.size(),true);
		for (int i = 0, l = updates.size(); i < l; i++) {
			Notification notification = (Notification) updates.get(i);
			if (!(notification.getFeature() instanceof EReference))
				continue;
			if (notification.getEventType() != Notification.ADD && notification.getEventType() != Notification.SET)
				continue;
			EObject source = (EObject) notification.getNotifier();
			EReference reference = (EReference) notification.getFeature();
			EObject target = (EObject) notification.getNewValue();
			int position = notification.getPosition();
			if(!usePreparedStatement)
			{
				boolean set = database.setReference(source, reference, target, position);
				if (!set)
					delayed.add(notification);
			}
			else
			{
				Integer sourceId = getObjectId(source);
				Integer targetId = getObjectId(target);
				if (sourceId == null || targetId == null) {
					delayed.add(notification);
					continue;
				}
				referenceToValues = (Map) objToReferences.get(sourceId);
				if (referenceToValues == null) {
					referenceToValues = new HashMap();
					objToReferences.put(sourceId, referenceToValues);
				}
				List[] targetInfos = (List[]) referenceToValues.get(reference);
				if (targetInfos == null) {
					targetInfos = new List[2];
					targetInfos[0] = new ArrayList(); // target
					targetInfos[1] = new ArrayList(); // position
					referenceToValues.put(reference, targetInfos);
				}
				if (!targetInfos[0].contains(targetId)) {
					targetInfos[0].add(targetId);
					targetInfos[1].add(new Integer(position));
				}
			}
		}
		p.stopAndPrintStatus();
		if(usePreparedStatement)
		{
			if (!objToReferences.isEmpty()) {
				
				p.setMessageAndStart("UpdateCommand.setReferencesFromUpdates() 2 size="+objToReferences.size());
				try {
					Integer sourceId;
					List targets;
					EReference reference;
					List positions;
					for (Iterator iter = objToReferences.entrySet().iterator(); iter.hasNext();) {
						Map.Entry entry = (Map.Entry) iter.next();
						sourceId = (Integer) entry.getKey();
						for (Iterator iterator = ((Map) entry.getValue()).entrySet().iterator(); iterator.hasNext();) {
							Map.Entry refs = (Map.Entry) iterator.next();
							reference = (EReference) refs.getKey();
							targets = ((List[]) refs.getValue())[0];
							positions = ((List[]) refs.getValue())[1];
//							if(sourceId==null)
//							{
//								System.out.println("--------------- "+reference+" - "+targets+" - "+positions);
//							}
//							else
//							{
								database.setReference(sourceId, reference, targets, positions);
//							}
						}
					}
					
				} catch (Exception e) {
					p.stopAndPrintStatus(e.getLocalizedMessage());
					throw e;
				}
				p.stopAndPrintStatus();
			}
		}
		
	}
	protected void setAttributes() throws Exception {
		for (int i = 0, l = updates.size(); i < l; i++) {
			Notification notification = (Notification) updates.get(i);
			if (notification.getFeature() == null)
				continue;
			if (!(notification.getFeature() instanceof EAttribute))
				continue;
			EObject owner = (EObject) notification.getNotifier();
			EAttribute attribute = (EAttribute) notification.getFeature();
			database.set(owner, attribute);
		}
	}
} // UpdateCommand
