/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.sample;

import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.hyades.ui.internal.util.CoreUtil;
import org.eclipse.hyades.ui.internal.util.ResourceBundleManager;
import org.eclipse.ui.plugin.AbstractUIPlugin;

/**
 * Log Analyzer Samples Plugin class.  This class should not be used 
 * outside the context of Eclipse.
 * 
 * @author sleeloy
 * @since 0.2.0
 */
public class HyadesSamplePlugin 
extends AbstractUIPlugin	
{
	private static HyadesSamplePlugin instance;
	private ResourceBundleManager resourceBundleManager;

	/**
	 * Constructor for HyadesUIPlugin
	 * @param descriptor
	 */
	public HyadesSamplePlugin(IPluginDescriptor descriptor)
	{
		super(descriptor);
		instance = this;
	}
	
	/**
	 * Returns the instance of this class created by the eclipse framework.
	 * @return HyadesUIPlugin
	 */
	public static HyadesSamplePlugin getInstance()
	{
		return instance;
	}
	
	/**
	 * @see org.eclipse.core.runtime.Plugin#shutdown()
	 */
	public void shutdown()
	throws CoreException
	{
		resourceBundleManager.dispose();
		instance = null;
		super.shutdown();
	}
	
	/**
	 * Returns this plugin's id.
	 * @return String
	 */
	public static String getID()
	{
		return instance.getDescriptor().getUniqueIdentifier();
	}

	/**
	 * @see org.eclipse.core.runtime.Plugin#startup()
	 */
	public void startup()
	throws CoreException
	{
		resourceBundleManager = new ResourceBundleManager(); 
		resourceBundleManager.add(getDescriptor().getResourceBundle());

		super.startup();		
	}


	/**
	 * Logs an error described by a throwable.
	 * 
	 * <p>This method should be used whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param throwable
	 */
	public static void logError(Throwable throwable)
	{
		logError(CoreUtil.getStackTrace(throwable));
	}
	
	/**
	 * Logs an error described by a text.
	 * 
	 * <p>This method should be whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 */
	public static void logError(String text)
	{
		System.err.println("***** HyadesUIPlugin.logError");
		System.err.println(text);
	}
	
	/**
	 * Logs an information described by a text.
	 * 
	 * <p>This method should be whenever a class in this plugin
	 * has to log an information since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 */
	public static void logInfo(String text)
	{
		System.out.println("***** HyadesUIPlugin.logInfo");
		System.out.println(text);
	}
	

	/**
	 * Returns the resource bundle used by this plugin.
	 * 
	 * <p>IMPORTANT: Don't use this method to retrieve values from the
	 * resource bundle.  For this purpose use the static <code>getString()</code> 
	 * defined in this class.
	 * 
	 * <p>This method is provided so this resource bundle can
	 * be used as the parent of another resource bundle.
	 * 
	 * @return ResourceBundle
	 */
	public ResourceBundle getResourceBundle()
	{
		return getDescriptor().getResourceBundle();
	}

	/**
	 * Returns the "plugin.properties" file's value associate to a given key.
	 * @param key
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key);
	}
	
	/**
	 * Returns the "plugin.properties" file's value associate to a given key.
	 * 
	 * <p>The strings defined in <code>args</code> replaces the %n (where n>=1) 
	 * variables defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and args is <code>{"v1", null, "v3"}</code>,
	 * the return of this method is <code>"%0 v1  v3 %4"</code>.
	 * 
	 * @param key
	 * @param args
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key, String[]args)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key, args);
	}
	
}
