/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.editor.form.util;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.Widget;

import org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.base.SelectableFormLabel;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * This class allows the client to create some widgets.
 * @author marcelop, bcormier
 * @since 3.0.0
 */
public class WidgetFactory 
extends FormWidgetFactory implements IDisposable
{
	/**
	 * This class is used to paint the borders for all the borderizables widgets of a control.
	 * @author marcelop, bcormier
	 * @since 3.0.0
	 */
	protected class BorderPainter 
	implements PaintListener
	{
		public void paintControl(PaintEvent event)
		{
			Composite composite = (Composite)event.widget;
			Control[] children = composite.getChildren();
			for (int i = 0; i < children.length; i++)
			{
				Control c = children[i];
				boolean inactiveBorder = false;
				if (c.getEnabled() == false && !(c instanceof CCombo))
					continue;
				if (c instanceof SelectableFormLabel)
					continue;
					
				Object flag = c.getData(KEY_DRAW_BORDER);
				if (flag != null)
				{
					if (flag.equals(Boolean.FALSE))
						continue;
					if (flag.equals(TREE_BORDER))
						inactiveBorder = true;
				}

				if (!inactiveBorder && (c instanceof Text || c instanceof Canvas || c instanceof CCombo))
				{
					Rectangle b = c.getBounds();
					GC gc = event.gc;
					gc.setForeground(c.getBackground());
					gc.drawRectangle(b.x - 1, b.y - 1, b.width + 1, b.height + 1);
					gc.setForeground(getForegroundColor());
					if (c instanceof CCombo)
						gc.drawRectangle(b.x - 1, b.y - 1, b.width + 1, b.height + 1);
					else if (c instanceof StyledText)
						gc.drawRectangle(b.x - 2, b.y - 2, b.width + 2, b.height + 3);
					else 
						gc.drawRectangle(b.x - 1, b.y - 2, b.width + 1, b.height + 3);
				}
				else if (inactiveBorder || c instanceof Table || c instanceof Tree || c instanceof TableTree)
				{
					Rectangle b = c.getBounds();
					GC gc = event.gc;
					gc.setForeground(getBorderColor());
					gc.drawRectangle(b.x - 1, b.y - 1, b.width + 2, b.height + 2);
				}
			}
		}
	}
	
	/**
	 * This class handles the visibility of a widget.
	 * @author marcelop, bcormier
	 * @since 3.0.0
	 */
	protected static class VisibilityHandler 
	extends FocusAdapter
	{
		/**
		 * This method is called when the control which raised a focus event gets focus.
		 * @param e the focus event
		 */
		public void focusGained(FocusEvent e)
		{
			Widget w = e.widget;
			if (w instanceof Control)
			{
				ensureVisible((Control) w);
			}
		}
	}

	private VisibilityHandler visibilityHandler;
	private BorderPainter borderPainter;

	/**
	 * Constructor for WidgetFactory
	 * 
	 */
	public WidgetFactory()
	{
		this(Display.getCurrent());
	}

	/**
	 * Constructor for WidgetFactory
	 * @param display
	 */
	public WidgetFactory(Display display)
	{
		super(display);
		visibilityHandler = new VisibilityHandler();
		borderPainter = new BorderPainter();
	}

	/**
	 * Disposes the widget factory.
	 */
	public void dispose()
	{
		visibilityHandler = null;
		borderPainter = null;
		super.dispose();
	}
	
	/**
	 * Returns the visibilty handler used as the focus listeners
	 * of some of the created widgets.
	 * @return VisibilityHandler
	 */
	protected VisibilityHandler getVisibilityHandler()
	{
		return visibilityHandler;
	}
	
	/**
	 * @return the painter used to paint the borders
	 */
	protected BorderPainter getBorderPainter()
	{
		return borderPainter;
	}
	
	/**
	 * Creates a styled text.
	 * @param parent
	 * @param style
	 * @return StyledText
	 */
	public StyledText createStyledText(Composite parent, int style)
	{
		StyledText styledText = new StyledText(parent, style);

		styledText.setBackground(getBackgroundColor());
		styledText.setForeground(getForegroundColor());
		styledText.addFocusListener(getVisibilityHandler());
		return styledText;
	}
	
	/**
	 * Creates a ccombo.
	 * @param parent
	 * @param style
	 * @return CCombo
	 */
	public CCombo createCCombo(Composite parent, int style)
	{
		CCombo combo = new CCombo(parent, style);

		combo.setBackground(getBackgroundColor());
		combo.setForeground(getForegroundColor());
		combo.addFocusListener(getVisibilityHandler());
		return combo;
	}

	/**
	 * Paints the borders for all the borderizable widgets of the composite.
	 * @param composite the composite. 
	 */
	public void paintBordersFor(Composite composite)
	{
		if (BORDER_STYLE == SWT.BORDER)
			return;
		composite.addPaintListener(getBorderPainter());
	}
}
