/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.impl.TPFBehaviorImpl;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;

/**
 * This class provides a central place for utility functions used to handle 
 * {@link org.eclipse.hyades.models.common.configuration.CFGArtifact} resources.
 * 
 * @author bjiang
 * @since  3.0
 */
public class ArtifactUtil 
{
	public static final String PROPERTY_NAME_CLASSPATH = Common_ConfigurationFactory.eINSTANCE.createHyadesClasspathCategory().getName(); //"CLASSPATH"
	
	public static void associateTestAsset(CFGClass testAsset, CFGArtifact artifact)
	{
		if(!ConfigurationUtil.containsById(artifact.getDeployableInstances(), testAsset))
		{
			artifact.getDeployableInstances().add(testAsset);
			//testAsset.getArtifacts().add(artifact);
		}
		
		if(testAsset instanceof TPFTestSuite)
			calculateClasspath((TPFTestSuite)testAsset, artifact);
	}
	
	public static void calculateClasspath(TPFTestSuite testSuite, CFGArtifact artifact)
	{		
		CFGPropertyGroup propertyGroup = ConfigurationUtil.searchPropertyGroupById(artifact.getPropertyGroups(), ConfigurationUtil.ATTRS_PROP_GROUP_ID_ARTIFACT);
		if(propertyGroup == null)
		{
		    propertyGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
		    propertyGroup.setPropertyGroupID(ConfigurationUtil.ATTRS_PROP_GROUP_ID_ARTIFACT);
		    artifact.getPropertyGroups().add(propertyGroup);
		}
		
		//CFGComparableProperty[] properties = findPropertiesByName(propertyGroup.getProperties(), PROPERTY_NAME_CLASSPATH, false);
		//for (int i = 0, maxi = properties.length; i < maxi; i++)
			//propertyGroup.getProperties().remove(properties[i]);
			
		if((testSuite.getBehavior() == null) || (((TPFBehaviorImpl)testSuite.getBehavior()).getProject() == null))
		{
			// Test execution should detect this upon test suite selection.
			// This is not the right place to do this.
			//MessageDialog.openError(Display.getCurrent().getActiveShell(), 
			//						TestUIPlugin.getString("DLG_GENERATE_TEST"),  //$NON-NLS-1$
			//						TestUIPlugin.getString("DLG_GENERATE_TEST_DSC")); //$NON-NLS-1$
			return;
		}
			
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(((TPFBehaviorImpl)testSuite.getBehavior()).getProject());
		if((project == null) || (!project.exists()))
		    return; 
		
		String[] classpathEntries = getProjectClasspath(project, false);
		for(int i = 0, maxi = classpathEntries.length; i < maxi; i++)
		{
			if(addClasspathEntry(classpathEntries[i]))
			{
			    BVRProperty[] existingProps = ConfigurationUtil.searchPropertiesByName(propertyGroup.getProperties(), PROPERTY_NAME_CLASSPATH, false);
				boolean existed = false;
			    for(int propIndex = 0; propIndex < existingProps.length; propIndex++)
			    {
			        if(classpathEntries[i].equals(existingProps[propIndex].getValue()))
			        {
			            existed = true;
			            break;
			        }
			    }
			    
			    if(existed == false)
			    {
				    CFGComparableProperty property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
					property.setName(PROPERTY_NAME_CLASSPATH);
					property.setOperator("=");
					property.setValue(classpathEntries[i]);
					propertyGroup.getProperties().add(property);
			    }
			}
		}
	}
	
	protected static boolean addClasspathEntry(String cp)
	{
		if(cp.indexOf("junit.jar") > 0)
			return false;

		if(cp.indexOf("xercesImpl.jar") > 0)
			return false;

		if(cp.indexOf("xmlParserAPIs.jar") > 0)
			return false;

		if(cp.indexOf("hexr.jar") > 0)
			return false;

		if(cp.indexOf("hexcore.jar") > 0)
			return false;

		if(cp.indexOf("common.runner.jar") > 0)
			return false;

		if(cp.indexOf("java.runner.jar") > 0)
			return false;

		if(cp.indexOf("http.runner.jar") > 0)
			return false;

		if(cp.indexOf("manual.runner.jar") > 0)
			return false;
					
		return true;
	}
	
	/**
	 * Returns a Project's classpath.  If the project has a dependency on another 
	 * project the exported entries of the last one will be added to the classpath
	 * of the former. 
	 * 
	 * @param project
	 * @param exportedOnly
	 * @return String[]
	 */
	public static String[] getProjectClasspath(IProject project, boolean exportedOnly)
	{
		if(project == null)
			return new String[0];
			
		IJavaProject javaProject = JavaCore.create(project);
		if(javaProject == null)
			return new String[0];

		List cpEntries = null;
		try
		{
			cpEntries = new ArrayList(Arrays.asList(javaProject.getRawClasspath()));
		}
		catch(JavaModelException e)
		{
		}
		if((cpEntries == null) || cpEntries.isEmpty())
			return new String[0];
		
		Set classpath = new HashSet(cpEntries.size());
		try
		{	
			IPath path = javaProject.getOutputLocation();
			if(path != null)
				classpath.add(getFullPath(path));
		}
		catch(JavaModelException e)
		{
		}
				
		for (Iterator i = cpEntries.iterator(); i.hasNext();)
		{
			IClasspathEntry entry = JavaCore.getResolvedClasspathEntry((IClasspathEntry)i.next());
			if(exportedOnly && (!entry.isExported()))
				continue;
			
			String strcp = null;
			switch(entry.getEntryKind())
			{
				case IClasspathEntry.CPE_LIBRARY:
				case IClasspathEntry.CPE_VARIABLE:
					strcp = getFullPath(entry.getPath());
					if(strcp != null)
						classpath.add(strcp);
					break;

				case IClasspathEntry.CPE_PROJECT:
					IPath projectPath = entry.getPath().makeAbsolute();
					IProject parentProject = ResourcesPlugin.getWorkspace().getRoot().getProject(projectPath.toString());				
					classpath.addAll(Arrays.asList(getProjectClasspath(parentProject, true)));
					break;
				
				case IClasspathEntry.CPE_CONTAINER:
					break;
					
				case IClasspathEntry.CPE_SOURCE:
					strcp = getFullPath(entry.getOutputLocation());
					if(strcp != null)
						classpath.add(strcp);
			}
		}
				
		return (String[])classpath.toArray(new String[classpath.size()]);
	}
	
	private static String getFullPath(IPath path)
	{
		if(path == null)
			return null;
			
		IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path.makeAbsolute());
		if(res != null)
			path = res.getLocation().makeAbsolute();
			
		return path.toString();
	}
	

    public static void adjustDefaultLocation(TPFDeployment depl, TPFTest test)
    {
		CFGLocation location = ConfigurationUtil.getDefaultLocation(depl);

		if ( location != null )
    	{
        	TPFTestSuite testSuite = null;
        	TPFTest subTest = null;
        	if (test instanceof TPFTestCase) {
        		subTest = test;
        		testSuite = ((TPFTestCase)subTest).getTestSuite();
        	} else if (test instanceof TPFTestSuite) {
        		testSuite = (TPFTestSuite)test;
        		subTest = test;
        	}
			Resource resource = depl.eResource();

        	CFGArtifact artifact = ConfigurationUtil.createArtifact(null, null);
    		ArtifactUtil.associateTestAsset(testSuite, artifact);
    		CFGArtifactLocationPair pair = ConfigurationUtil.createArtifactLocationPair(artifact, location);
			depl.getArtifactLocations().add(pair);
			resource.setModified(true);
   		}
    }
	
}
