/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.util;

import java.io.File;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.ui.PartInitException;

/**
 * This class provides a central place for utility functions used to create or handle 
 * {@link org.eclipse.hyades.models.common.configuration.CFGLocation} resources.
 * 
 * @author bjiang
 * @since  1.3
 */
public class LocationUtil 
{
	public static final String PROPERTY_NAME_ROOTDIR = Common_ConfigurationFactory.eINSTANCE.createHyadesRootDirectroyCategory().getName(); //"ROOTDIR"
	
	/**
	 * This method creates a new location EObject and persists the EObject in a resource
	 * sepecified by <i>file</i>.
	 *  
	 * @param file
	 * @param name
	 * @param description
	 * @return
	 * @throws Exception
	 */
	public static void createNew(IFile file, String name, String description, String hostname)
	throws Exception
	{
		Resource resource = createResource(file);
		CFGLocation location = createLocation(name, description, hostname);
		resource.getContents().add(location);
		EMFUtil.save(resource);
	}
	
	/**
	 * Creates the CFGLocation setting the attributes based on the specified information and
	 * initialize this new location object with basic properties.
	 * @return a not null EObject
	 */
	public static CFGLocation createLocation(String name, String description, String hostname)
	{
		CFGLocation location = ConfigurationUtil.createLocation(name, description, hostname);
		setLocationRootDir(location);
		return location;
	}

	/**
	 * Creates the EMF resource in which the object is added to.
	 * @return a not null Resource
	 */
	public static Resource createResource(IFile file)
	{
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString());
		Resource.Factory factory = EMFUtil.getResourceFactory(ICommonConstants.LOCATION_FILE_EXTENSION);
		return factory.createResource(uri);
	}

	public static String getFileExtension()
	{
		return ICommonConstants.LOCATION_FILE_EXTENSION;
	}
	
	/**
	 * Opens the editor for the specified location resource file. This method does nothing
	 * if the specified file does not represent a location resource.
	 */
	public static void openEditor(IFile file) throws PartInitException
	{
		if(!file.getFileExtension().equals(getFileExtension()))
			return;
		
		if(TestUIPlugin.getInstance().getPreferenceStore().getBoolean(TestUI.OPEN_EDITOR))
			UIUtil.openEditor(file, TestUIExtension.LOCATION_EDITOR_PART_ID, false);
	}
	
	public static void setLocationRootDir(CFGLocation location)
	{
	    CFGPropertyGroup propGroup = ConfigurationUtil.searchPropertyGroupById(location.getPropertyGroups(), ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
	    if(propGroup == null)
	    {
	        propGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
	        propGroup.setPropertyGroupID(ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
	        location.getPropertyGroups().add(propGroup);
	    }
	    
	    BVRProperty[] properties = ConfigurationUtil.searchPropertiesByName(propGroup.getProperties(), PROPERTY_NAME_ROOTDIR, false);
	    CFGComparableProperty property = null;
	    if(properties == null || properties.length < 1)
	    {
	        property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
	        property.setName(PROPERTY_NAME_ROOTDIR);
	        property.setOperator("=");
	        propGroup.getProperties().add(property);	
	    }
	    else
	        property = (CFGComparableProperty)properties[0];
	    
	    if(property.getValue() == null)
	    {
	        String workspaceRootDir = getWorkspaceRootDir();
	        property.setValue(workspaceRootDir);        
	    }
	}
	
	protected static String getWorkspaceRootDir()
    {
        String workspaceRoot = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString();  
        if(workspaceRoot.endsWith(File.separator))
            workspaceRoot = workspaceRoot.substring(0, (workspaceRoot.length() - File.separator.length()));
        
        return workspaceRoot;
    }

}
