/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.dialog;

import java.text.MessageFormat;
import java.text.NumberFormat;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.test.ui.datapool.DatapoolPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.util.DatapoolUtil;
import org.eclipse.hyades.test.ui.datapool.internal.util.GridDataUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;

/**
 * Dialog to collect information about a equivalenceClass/row group that is 
 * being inserted or edited by the datapool table.  
 * 
 * @author psun
 *
 */
public class DatapoolRowGroupDialog extends Dialog implements KeyListener, SelectionListener 
{

	private static final String TAG_EQUIVALENCE_CLASSES = "equivalenceClasses"; //$NON-NLS-1$

	private static final int HEIGHT = 80;
	private static final int WIDTH = 300;
	private Shell shell = null;
	private IDatapool datapool = null;
	private String title = null;
	private boolean insertionMode = false;

	private Table table = null;
	private IDatapoolEquivalenceClass selectedEquivalenceClass = null;
	private IDatapoolEquivalenceClass previousEquivalenceClass = null;

	private Text nameField = null;
	private Color nameFieldForeground = null;
	private Combo equivalenceClassCombo = null;
	private Label errorLabel = null;

	private String name = null;
	private int insertionIndex = -1;
	private int defaultNameCounter = 1;

	public DatapoolRowGroupDialog(Shell parentShell, 
								  IDatapool datapool,
								  Table table,
								  IDatapoolEquivalenceClass selectedEquivalenceClassItem,								  
								  IDatapoolEquivalenceClass previousEquivalenceClassItem,
								  String title)
	{
		super(parentShell);
		this.shell = parentShell;
		this.datapool = datapool;
		this.title = title; 
		
		this.table = table;
		this.previousEquivalenceClass = previousEquivalenceClassItem;
		this.selectedEquivalenceClass = selectedEquivalenceClassItem;
		if(title.equals(DatapoolPlugin.getResourceString("DATA_ROW_GRP_DLG_TITLE_INS"))) //$NON-NLS-1$
			this.insertionMode = true;

	}

	/**
	 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createContents(Composite parent)
	{
		Control control = super.createContents(parent);
		enableOK();
		return control;
	}

	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent)
	{
		getShell().setText(title);
		Composite superComposite = (Composite)super.createDialogArea(parent);
		GridData gridData = GridDataUtil.createFill();
		//gridData.heightHint = HEIGHT;
		//gridData.widthHint = WIDTH;
		superComposite.setLayoutData(gridData);	

		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		superComposite.setLayout(gridLayout);

		Label nameLabel = new Label(superComposite, SWT.NONE);
		nameLabel.setText(DatapoolPlugin.getResourceString("DATA_ROW_GRP_DLG_NAME")); //$NON-NLS-1$

		nameField = new Text(superComposite, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 1;
		nameField.setLayoutData(gridData);
		nameField.addKeyListener(this);

		if(selectedEquivalenceClass != null)
		{
			name = selectedEquivalenceClass.getName();
			nameField.setText(name);
			nameField.setSelection(name.length() + 1);
		}

		Label equivalenceClassComboLabel = new Label(superComposite, SWT.NONE);
		if(insertionMode)
		{
			equivalenceClassComboLabel.setText(DatapoolPlugin.getResourceString("DATA_DLG_INS")); //$NON-NLS-1$
			name = createDefaultName();
			while(!DatapoolUtil.getInstance().isEquivalenceClassNameUnique(datapool, name, null))
			{
				name = createDefaultName();
			}
			nameField.setText(name);
			nameField.setSelection(name.length() + 1);
		}
		else
			equivalenceClassComboLabel.setText(DatapoolPlugin.getResourceString("DATA_DLG_MOVE")); //$NON-NLS-1$

		
		equivalenceClassCombo = new Combo(superComposite, SWT.READ_ONLY | SWT.DROP_DOWN);
		IDatapoolEquivalenceClass[] equivalenceClasses = null;
		equivalenceClasses = DatapoolUtil.getInstance().getEquivalenceClasses(datapool, null);
		String[] equivalenceClassLabels = getEquivalenceClassLabels(equivalenceClasses);

		equivalenceClassCombo.setItems(equivalenceClassLabels);
		equivalenceClassCombo.setData(TAG_EQUIVALENCE_CLASSES, equivalenceClasses);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 1;
		gridData.widthHint = 250;
		equivalenceClassCombo.setLayoutData(gridData);
		equivalenceClassCombo.addSelectionListener(this);

		if(previousEquivalenceClass != null)
		{
			int previousEquivalenceClassIndex = datapool.getEquivalenceClassIndex(previousEquivalenceClass.getName());
			if(previousEquivalenceClassIndex != -1)
			{
				equivalenceClassCombo.select(previousEquivalenceClassIndex + 1);
				insertionIndex = previousEquivalenceClassIndex;
			}
		}
		else
		{
			equivalenceClassCombo.select(0);
			insertionIndex = -1;			
		}

		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL|GridData.GRAB_HORIZONTAL);
		gridData.horizontalSpan = 2;
		errorLabel = new Label(superComposite, SWT.NONE);
		errorLabel.setLayoutData(gridData);
		
		superComposite.pack();
		return superComposite;
	}

	private boolean enableOK()
	{
		boolean enableButton = true;
		String nameErrorText = new String();
		Display display= Display.getDefault();
		Color color= display.getSystemColor(SWT.COLOR_RED);
		String name = nameField.getText();

		if(name == null || name.length() == 0)
		{
			getButton(IDialogConstants.OK_ID).setEnabled(false);
			return false;
		}

		if(name != null && name.length() > 0)
		{
			boolean isNameUnique = DatapoolUtil.getInstance().isEquivalenceClassNameUnique(datapool, name, selectedEquivalenceClass);
			boolean isNameValid = DatapoolUtil.getInstance().isEquivalenceClassNameValid(name);
			if(isNameUnique && isNameValid)
			{		
				nameField.setForeground(nameFieldForeground);
			}
			else
			{
				nameField.setForeground(color);
				if(!isNameUnique)
				{
					nameErrorText = DatapoolPlugin.getResourceString("DATA_ROW_GRP_DLG_ERROR_NAME_NOT_UNIQUE"); //$NON-NLS-1$
				}
				else if(!isNameValid)
				{
					nameErrorText = DatapoolPlugin.getResourceString("DATA_ROW_GRP_DLG_ERROR_NAME_NOT_VALID"); //$NON-NLS-1$
				}
				enableButton = false;
			}
		}
		
		if(enableButton)
		{		
			getButton(IDialogConstants.OK_ID).setEnabled(true);
			errorLabel.setText(new String());
			return true;
		}
		else
		{
			String errorText = DatapoolPlugin.getResourceString("DATA_DLG_ERROR_INDICATOR") + nameErrorText; //$NON-NLS-1$
			errorLabel.setText(errorText);
			errorLabel.setForeground(color);
			getButton(IDialogConstants.OK_ID).setEnabled(false);		
			return false;
		}
	}

	private String[] getEquivalenceClassLabels(IDatapoolEquivalenceClass[] equivalenceClasses)
	{
		int listSize = 0;
		if(equivalenceClasses != null)
			listSize = equivalenceClasses.length;
		if(equivalenceClasses.length == 1 && equivalenceClasses[0].getRecordCount() == 0)
			listSize = 0;
		String[] returnList = new String[listSize + 1];
		String insertFormat = DatapoolPlugin.getResourceString("DATA_DLG_INSERT_VARIABLE_FORMAT");
		for(int i = 0; i < listSize; i++)
		{
			IDatapoolEquivalenceClass equivalenceClass = equivalenceClasses[i];
			String name = equivalenceClass.getName();
			if(i == 0)
			{
				Object[] messageElements = {DatapoolPlugin.getResourceString("DATA_DLG_BEFORE"), name}; //$NON-NLS-1$
				returnList[i] = MessageFormat.format(insertFormat, messageElements); //$NON-NLS-1$				
			}			
			Object[] messageElements = {DatapoolPlugin.getResourceString("DATA_DLG_AFTER"), name}; //$NON-NLS-1$
			returnList[i + 1] = MessageFormat.format(insertFormat, messageElements); //$NON-NLS-1$
		}
		if(listSize == 0)
		{
			returnList[0] = DatapoolPlugin.getResourceString("DATA_DLG_FIRST_ELEMENT"); //$NON-NLS-1$
		}
		return returnList;
	}

	private String[] getRecordIndexLabels(IDatapoolEquivalenceClass equivalenceClass)
	{
		if(equivalenceClass == null)
			return null;
		int listSize = equivalenceClass.getRecordCount();
		String[] returnList = new String[listSize];
		for(int i = 0; i < listSize; i++)
			returnList[i] = String.valueOf(i);
		return returnList;
	}


	private int findEquivalenceClassIndexInCombo(IDatapoolEquivalenceClass[] equivalenceClasses, IDatapoolEquivalenceClass equivalenceClass)
	{
		if(equivalenceClasses == null || equivalenceClass == null)
			return -1;
		for(int i = 0; i < equivalenceClasses.length; i++)
		{
			if(equivalenceClasses[i].equals(equivalenceClass))
				return i;	
		}
		return -1;
	}

	public String getName()
	{
		return name;
	}

	public int getInsertionIndex()
	{
		return insertionIndex;
	}
	
	// ISelectionListener
	
	/*
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) 
	{
		IDatapoolEquivalenceClass equivalenceClass = null; 
		Combo combo = (Combo)e.widget;
		if(combo.getSelectionIndex() != 0)
		{
			int selectionIndex = combo.getSelectionIndex();
			IDatapoolEquivalenceClass[] equivalenceClasses = (IDatapoolEquivalenceClass[])combo.getData(TAG_EQUIVALENCE_CLASSES);
			equivalenceClass = equivalenceClasses[selectionIndex - 1];
			insertionIndex = datapool.getEquivalenceClassIndex(equivalenceClass.getName());
		}
		else
			insertionIndex = -1;
		
	}

	/*
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
	}

	// IKeyListener

	/*
	 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyPressed(KeyEvent e) 
	{
		return;
	}

	/*
	 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyReleased(KeyEvent e) 
	{
		if(nameField != null)
			name = nameField.getText();
		enableOK();		
	}	
	
	private String createDefaultName()
	{
		Object[] messageElements = {DatapoolPlugin.getResourceString("DATA_EC_NAME"), NumberFormat.getInstance().format(defaultNameCounter)}; //$NON-NLS-1$
		defaultNameCounter++;
		return MessageFormat.format("{0}{1}", messageElements); //$NON-NLS-1$
	}
}
