/**********************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.recorder.remote;

import java.io.CharArrayWriter;
import java.io.PrintWriter;

import org.eclipse.hyades.internal.execution.remote.AgentControllerUnavailableException;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;



/**
 * This is the controller which is instantiated by the RAC and which starts all RecorderAgents.
 * If the application to be recorded is started by the recorder, this is the object which handles that startup.
 * A good part of its behaviour occurs as a result of commands being handled by the RecorderCommandHandler
 * @see org.eclipse.hyades.internal.execution.recorder.remote.RecorderCommandHandler
 * @author Ernest Jessee
 */
public class RecorderAgentController implements IRecorderMessageTypes
{
		
		
	/**
	 * boolean indicating whether or not we have received the "Start" message from the RAC
	 */
	private boolean okToStart = false;

	/**
	 * our reference to the client side object
	 */
	private RemoteComponentSkeleton skeleton = null;

	/**
	 * the agent to be controlled
	 */
	private RecorderAgent recordHarnessAgent = null;
	
	/**
	 * the full path of the agent to be controlled
	 */
	private static String agentClassPath;
	
	/**
	 * The process which represents the application that we will be recording.  If the user has chosend to not automatically start the app, this 
	 * will remain null.
	 */
	private Process process=null;
	
	/**
	 * the thread in which the RecorderAgent will perform its actual recording
	 */
	private Thread recorderThread;


	

	//private RecorderDeathWatch deathWatch;

	/**
	 * thread in which the RecorderAgent resides and performs
	 * @author Ernest Jessee
	 */
	public class RecorderThread extends Thread
	{
		private RecorderAgent runnable = null;
			
		private Thread workerThread=null;

		public RecorderThread(RecorderAgent agent)
		{
			this.runnable = agent;
			setName("Recorder Thread"); //$NON-NLS-1$
		}
	
	
		public void run()
		{
			workerThread = runnable.run();
		}	

		/**
		 * @return
		 */
		public Thread getWorkerThread() {
			return workerThread;
		}

	}
	
	/**
	 * Thread which monitors the recorder thread
	 * when the recorder dies this threads shuts down everything else 
	 * in a friendly fashion. 
	 * @author Ernest Jessee
	 */
	protected class RecorderDeathWatch extends Thread
	{

		private Process process;
		private Thread thread;
		private RecorderAgent agent;

		public RecorderDeathWatch(Process process, Thread thread, RecorderAgent agent)
		{
			this.process = process;
			this.thread = thread;
			this.agent = agent;
			String name="Recorder Death Watch"; //$NON-NLS-1$
			sendDebugMessageToDataProcessor("recorder death watch started started with name: "+name); //$NON-NLS-1$
			setName(name);
		}

		public void run()
		{
			try
			{	
				thread.join();
				stopRecording();
			}
			catch (InterruptedException e)
			{
				//this is an expected occurance when the user selects the stop button.
				
			} 
		 			
		}

	}
	/**
	 * the entry point for the controller.  The first arg will be the full path of the agent to be controlled.
	 * @param args
	 * void
	 */
	public static void main(String[] args)
	{
		agentClassPath = args[0];
		
		RecorderAgentController agentController = new RecorderAgentController();
						
		agentController.setupCommandHandler();
		
		agentController.waitForInitialization();
						
	}
	

	/**
	 * Waits for the start command which will be intercepted by the RecorderCommandHandler.  After the Start command has been received,it waits 
	 * for the agent to be instantiated
	 * 
	 * void
	 */
	private void waitForInitialization()
	{		
		while(!okToStart)
		{
			delay(100);
		}
		while(recordHarnessAgent == null)
		{
			delay(100);
		}
		
	}

	
	/**
	 * delay method
	 * @param int miliseconds
	 */
	private void delay(int miliseconds)
	{
		try
		{
			Thread.sleep(miliseconds);
		}
		catch (InterruptedException e)
		{
			
			e.printStackTrace();
		}
	}

	/**
	 * sets up the command handler so that control messages can be dealth with	 
	 */
	private void setupCommandHandler()
	{
		skeleton = new RemoteComponentSkeleton("racRunner","tester"); //$NON-NLS-1$ //$NON-NLS-2$

	  	skeleton.addCommandListener(new RecorderCommandHandler(this));
	  		  	  	
	  	try
		{
			skeleton.initialize();
		}	
		catch (AgentControllerUnavailableException e3)
		{
			e3.printStackTrace();
		}
				
	}
	

	/**
	 * called by the RecorderCommandHandler when the Start message is received.
	 * @param b
	 * void
	 */
	public void setOKToStart(boolean b)
	{
		okToStart = b;
	}

	
	/**
	 * called by the RecorderAgent to send data across the RAC data channel
	 * @param byte[] data
	 * @param int iOffset
	 * @param int dataLength
	  */	
	public synchronized void sendByteDataToDataProcessor(byte[] data, int iOffset, int dataLength)
	{
		sendByteDataToDataProcessor(data, iOffset, dataLength,MESSAGE_TYPE_DATA);
	}
	
	/**
	 * called by the RecorderAgent to send debug info across the RAC data channel
	 * @param message
	 */
	public synchronized void sendDebugMessageToDataProcessor(String message)
	{
		sendByteDataToDataProcessor(message.getBytes(), 0, message.getBytes().length,MESSAGE_TYPE_DEBUG_STRING);
//		skeleton.sendMessageToAttachedClient(message,0);	
	}
	
	/**
	 * called by the RecorderAgent to send control messages across the RAC data channel
	 * @param message
	 */
	public synchronized void sendControlMessageToDataProcessor(String message)	
	{
		sendByteDataToDataProcessor(message.getBytes(), 0, message.getBytes().length,MESSAGE_TYPE_CONTROL_STRING);	
	}
	
	/**
	 * called to report that the recording is complet
	 */
	public synchronized void sendCompleteNotification()
	{
		String message = "finished"; //$NON-NLS-1$
		sendByteDataToDataProcessor(message.getBytes(), 0, message.getBytes().length,MESSAGE_TYPE_COMPLETE_NOTIFICATION);
	}

	
	/**
	 * sends byte data with a type data byte at the beginning of the byte array across the RAC data channel
	 * @param byte[] msg
	 * @param int iOffset
	 * @param int bufLength
	 * @param int messageType
	 */
	protected synchronized void sendByteDataToDataProcessor(byte[] msg, int iOffset, int bufLength, int messageType)
	{
		byte[] byteBuf = createByteBuf(msg, iOffset, bufLength, messageType);
		skeleton.logMessage(byteBuf,0,byteBuf.length);
	}

	/**
	 * creates a data byte array with a control byte in the beginning indicating what type the data is.
	 * @param byte[] msg
	 * @param int iOffset
	 * @param int bufLength
	 * @param int messageType
	 */
	private byte[] createByteBuf(byte[] msg, int iOffset, int bufLength, int messageType) 
	{
		byte[] byteBuf = new byte[bufLength + 1];
	
		byteBuf[0] = new Integer(messageType).byteValue();
		
		for(int i=1; i<= bufLength; i++)
			byteBuf[i]=msg[i + iOffset -1];
			
		return byteBuf;
	}

	
	/**
	 * called by the RecorderAgent to report an exception to the IRecorderDataProcessor on the other side.
	 * @param explanation
	 * @param exception
	 */
	public void reportException(String explanation, Throwable exception)
	{
		sendDebugMessageToDataProcessor("\n\n*****************************Exception Stack********************************\n"+explanation); //$NON-NLS-1$
		
		StringBuffer stackTrace = new StringBuffer();
		if ( exception != null ) 
		{

			CharArrayWriter caw = new CharArrayWriter();
			PrintWriter pw = new PrintWriter(caw);
			exception.printStackTrace(pw);
			pw.flush();
			pw.close();
			stackTrace.append(caw.toString()+"\n"); //$NON-NLS-1$

		}	
		sendDebugMessageToDataProcessor(stackTrace.toString());			
		
		sendDebugMessageToDataProcessor("****************************************************************************\n\n"); //$NON-NLS-1$
	}

	/**
	 * called by the RecorderCommandHandler to handle the "Init" command. When called, the Recorder Agent is loaded and initialized.
	 * @param String initParams - semi-colon delimited collection of comma delimited name-value pairs
	 */
	public void initAgent(String initParams)
	{
				
		sendDebugMessageToDataProcessor("about to load recorder agent: "+agentClassPath); //$NON-NLS-1$
		
		try
		{
			recordHarnessAgent = (RecorderAgent)Class.forName(agentClassPath).newInstance();
			
		}
		catch (InstantiationException e)
		{
			reportException("unable to load recorder agent: "+agentClassPath,e); //$NON-NLS-1$
		}
		catch (IllegalAccessException e)
		{
			reportException("unable to load recorder agent: "+agentClassPath,e); //$NON-NLS-1$
		}
		catch (ClassNotFoundException e)
		{
			reportException("unable to load recorder agent: "+agentClassPath,e); //$NON-NLS-1$
		}
		sendDebugMessageToDataProcessor("load successful"); //$NON-NLS-1$
		
		sendDebugMessageToDataProcessor("about to initialize recorder agent: "+agentClassPath); //$NON-NLS-1$
		
		recordHarnessAgent.setController(this);
		
		if(recordHarnessAgent.init(initParams))
			sendDebugMessageToDataProcessor("initialization successful"); //$NON-NLS-1$
		else
			sendDebugMessageToDataProcessor("initialization failed"); //$NON-NLS-1$
		
	}
	
	/**
	 * called by the RecorderCommandHandler to handle the "StartRecording" command
	 */
	public void startRecording()
	{
		RecorderThread starterThread = new RecorderThread(recordHarnessAgent);
		starterThread.start();
		recorderThread = starterThread.getWorkerThread();

		//we want to wait for the agent to be ready before we return;
		while(!recordHarnessAgent.isReady())
		{
			//TODO handle timeout
			delay(100);
		}			
	}
	
	/**
	 * called to stop the recording.  This is called by the RecorderCommandHandler when the user selects the stop recording button and is 
	 * called byt the RecorderStopper Thread when the application to record is closed.
	 *
	 */
	public void stopRecording()
	{
		try
		{
			recordHarnessAgent.handleCommand(RecorderAgent.STOP_RECORDING,null);				
		}
		catch (RecorderHandleCommandException e)
		{
			reportException("unable to stop recording",e); //$NON-NLS-1$
		} 		
		sendCompleteNotification();
		//System.exit(0);  //removed bugzilla_72456 
	}
	
	

}
