/**********************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.testgen;

import java.util.HashMap;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;

/**
 * Factory for managing Test Generators
 * manages objects defined by extension point org.eclipse.hyades.execution.testgen.TestGenerator"
 * @author Ernest Jessee
 */
public class TestGeneratorFactory
{

	/**
	 * extension point ID for objects managed by this factory.
	 */
	private String extensionPointID = "org.eclipse.hyades.execution.testgen.TestGenerator"; //$NON-NLS-1$
	
	/**
	 * the default instance
	 */
	private static TestGeneratorFactory instance = null;
	
	/**
	 * HashMap for generators by ID
	 */
	private HashMap elementsByID = new HashMap();
		
		
	/**
	 * private constructor (singleton)
	 */
	private TestGeneratorFactory(){}
	
	
	/**
	 * called to get the default instance of this factory.  At this initial call, the factory loads all system metadata
	 * related to Test Generators
	 */
	public static TestGeneratorFactory getInstance()
	{
		if(instance == null)
		{
			instance = new TestGeneratorFactory();
			instance.loadMetadata();
			
		}
		
		return instance;
	}
	
	/**
	 * loads metadata related to Test Generators
	 *
	 */
	private void loadMetadata()
	{
		IConfigurationElement[] adapterElements = Platform.getPluginRegistry().getConfigurationElementsFor(extensionPointID);
		for(int i=0; i<adapterElements.length; i++)
		{
			IConfigurationElement element = adapterElements[i];
			elementsByID.put(element.getAttribute("id"),element); //$NON-NLS-1$
		}
		
	}
	
	/**
	 * returns the name of the generator with the specified ID
	 * @param adapterID
	 * @return
	 */
	public String getGeneratorName(String adapterID)
	{
		return ((IConfigurationElement)elementsByID.get(adapterID)).getAttribute("name"); //$NON-NLS-1$
	}
	
	/**
	 * returns the full classpath of the generator with the specified ID
	 * @param adapterID
	 * @return
	 */
	public String getGeneratorClassPath(String adapterID)
	{
		return ((IConfigurationElement)elementsByID.get(adapterID)).getAttribute("class"); //$NON-NLS-1$
	}
	
	public int getGeneratorPriority(String adapterID)
	{
		String strPriority = ((IConfigurationElement)elementsByID.get(adapterID)).getAttribute("priority");
		if (strPriority == null)
			return 0;
		try 
		{
			int pr = Integer.parseInt(strPriority);
			if (pr < -1 || pr > 99)
				return -1;
			return pr;
		}
		catch (NumberFormatException e)
		{
			return -1; 
		}
		
	}
	/**
	 * returns a String array of all registered generator ID's
	 * @return
	 */
	public String[] getGeneratorIDs()
	{
		Object[] array = elementsByID.keySet().toArray();
		String retArray[] = new String[array.length];
		for(int i=0; i<array.length; i++)
		{
			retArray[i]=(String)array[i];
		}
		return retArray;
	}
	
	/**
	 * returns a generator ID by index
	 * @param int index
	 * @return String
	 */
	public String getGeneratorIDByIndex(int index)
	{
		Object[] values = elementsByID.values().toArray();
		
		if(index>values.length-1)
			return null;
		
		IConfigurationElement element = (IConfigurationElement)values[index];
			
		return element.getAttribute("id"); //$NON-NLS-1$
	}
	
	
	/**
	 * returns the IConfigurationElement for the generator with the specified ID
	 * @param ID
	 * @return
	 */
	public IConfigurationElement getGeneratorConfigElement(String ID)
	{
		return (IConfigurationElement)elementsByID.get(ID);
	}
	

}
