/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.extension;

import java.util.Map;
import java.util.TreeMap;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.util.ListenerList;
import org.eclipse.ui.IPropertyListener;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationDescriptorFilter;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.extension.IAssociationMappingRegistry;
import org.eclipse.hyades.ui.internal.util.CoreUtil;
import org.eclipse.hyades.ui.internal.util.XMLUtil;

/**
 * Implementation of the {@link org.eclipse.hyades.ui.extension.IAssociationMappingRegistry}
 * interface.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class AssociationMappingRegistry 
implements IAssociationMappingRegistry
{
	private static final String PREFERENCE_PREFIX_KEY = HyadesUIPlugin.getID() + ".AssociationMappingRegistry."; 
	
	private Map associationMappingByExtensionPoint;
	
	private IAssociationDescriptorFilter filter;
	private String objectCategory;
	private IPreferenceStore preferenceStore;
	private ImageRegistry imageRegistry;
	private ListenerList commitChangeListeners;
	private String xmlBuffer; 
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#initialize(String, IAssociationDescriptorFilter, IPreferenceStore)
	 */
	public void initialize(String objectCategory, IAssociationDescriptorFilter filter, IPreferenceStore preferenceStore)
	throws IllegalArgumentException
	{
		if(objectCategory == null)
			throw new IllegalArgumentException(HyadesUIPlugin.getString("_ERROR_ARG_NULL", "objectCategory"));
			
		this.objectCategory = objectCategory;
		this.filter = filter;
		this.preferenceStore = preferenceStore;
		
		load(preferenceStore);
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		CoreUtil.dispose(associationMappingByExtensionPoint);
		if(commitChangeListeners != null)
			commitChangeListeners.clear();
		
		filter = null;
		preferenceStore = null;
		xmlBuffer = null;
		imageRegistry = null;		
	}

	/**
	 * Returns the image registry that is used to store the image
	 * of the mappings.
	 * @return
	 */
	public ImageRegistry getImageRegistry()
	{
		if(imageRegistry == null)
		{
			try
			{
				imageRegistry = new ImageRegistry();
			}
			catch(RuntimeException e)
			{
				HyadesUIPlugin.logError(e);
			}
		}
			
		return imageRegistry;
	}
	
	/**
	 * Sets the image registry that is used to store the mappings'
	 * images.
	 * @param imageRegistry
	 */
	public void setImageRegistry(ImageRegistry imageRegistry)
	{
		this.imageRegistry = imageRegistry;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#getObjectCategory()
	 */
	public String getObjectCategory()
	{
		return objectCategory;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#getPreferenceStore()
	 */
	public IPreferenceStore getPreferenceStore()
	{
		return preferenceStore;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#getFilter()
	 */
	public IAssociationDescriptorFilter getFilter()
	{
		return filter;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#retrieveAssociationMapping(String, String)
	 */
	public synchronized IAssociationMapping retrieveAssociationMapping(String extensionPoint, String pluginId)
	throws IllegalArgumentException
	{
		if(extensionPoint == null)
			throw new IllegalArgumentException(HyadesUIPlugin.getString("_ERROR_ARG_NULL", "extensionPoint"));
			
		AssociationMapping associationMapping = null;
		if(associationMappingByExtensionPoint == null)
			associationMappingByExtensionPoint = new TreeMap();
		else
			associationMapping = (AssociationMapping)associationMappingByExtensionPoint.get(extensionPoint);

		if(associationMapping == null)
		{
			associationMapping = new AssociationMapping(this, extensionPoint, pluginId);
			associationMappingByExtensionPoint.put(extensionPoint, associationMapping);
		}
		
		return associationMapping;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#getAssociationMapping(java.lang.String)
	 */
	public IAssociationMapping getAssociationMapping(String extensionPoint)
	{
		if((extensionPoint == null) || (associationMappingByExtensionPoint == null))
			return null;
			
		return (AssociationMapping)associationMappingByExtensionPoint.get(extensionPoint);
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#getAssociationMappings()
	 */
	public IAssociationMapping[] getAssociationMappings()
	{
		if(associationMappingByExtensionPoint == null)
			return new IAssociationMapping[0];
			
		return (AssociationMapping[])associationMappingByExtensionPoint.values().toArray(new AssociationMapping[associationMappingByExtensionPoint.values().size()]);
	}
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#checkpoint()
	 */
	public synchronized boolean checkpoint()
	{
		xmlBuffer = write();
		return true;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#clearCheckpoint()
	 */
	public synchronized void clearCheckpoint()
	{
		xmlBuffer = null;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#commit()
	 */
	public synchronized boolean commit()
	{
		checkpoint();
		if(write(preferenceStore))
			notifyCommitChangeListeners();
			
		return true;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#rollback()
	 */
	public synchronized boolean rollback()
	{
		if(xmlBuffer != null)
			load(xmlBuffer);
		return true;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#addCommitChangeListener(org.eclipse.ui.IPropertyListener)
	 */
	public void addCommitChangeListener(IPropertyListener propertyListener)
	{
		if(commitChangeListeners == null)
			commitChangeListeners = new ListenerList();
		commitChangeListeners.add(propertyListener);
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#removeCommitChangeListener(org.eclipse.ui.IPropertyListener)
	 */
	public void removeCommitChangeListener(IPropertyListener propertyListener)
	{
		if(commitChangeListeners != null)
			commitChangeListeners.remove(propertyListener);
	}
	
	/**
	 * Notify all the commit listeners.  This implementation ensures that
	 * an exception thrown by a listener doesn't affect the notification
	 * process.
	 */
	protected void notifyCommitChangeListeners()
	{
		if(commitChangeListeners == null)
			return;
			
		Object[] listeners = commitChangeListeners.getListeners();
		for(int i = 0, maxi = listeners.length; i < maxi; i++)
		{
			IPropertyListener propertyListener = (IPropertyListener)listeners[i];
			try
			{
				propertyListener.propertyChanged(this, PROP_CONTENTS);	
			}
			catch(RuntimeException e)
			{
			}
		}
	}

	/**
	 * Returns the preference key to retrieve this registry settings.
	 * @return String
	 */
	protected String getPreferenceKey()
	{
		StringBuffer key = new StringBuffer(PREFERENCE_PREFIX_KEY);
		key.append(objectCategory);
		return key.toString();
	}
	
	/**
	 * Reads this registry from a preference store.
	 * @param preferenceStore
	 * @return <code>true</code> if the registry was read or 
	 * <code>false</code> otherwise.
	 */
	protected boolean load(IPreferenceStore preferenceStore)
	{
		if(preferenceStore == null)
			return false;
			
		String value = preferenceStore.getString(getPreferenceKey());
		if((value == null) || (value.length() == 0))
			return false;
			
		return load(value);
	}
	
	/**
	 * Reads this registry from a xml string.
	 * @param xml
	 * @return <code>true</code> if the registry was read or 
	 * <code>false</code> otherwise.
	 */
	protected boolean load(String xml)
	{
		XMLUtil.setLogger(HyadesUIPlugin.getLogger());
		Element registryElement = XMLUtil.loadDom(xml, "AssociationMappingRegistry");
		if(registryElement == null)
			return false;
			
		NodeList nodeList = XMLUtil.getChildrenByName(registryElement, "AssociationMapping");
		for(int i=0, maxi=nodeList.getLength(); i<maxi; i++)
		{
			if(nodeList.item(i) instanceof Element)
				loadMapping((Element)nodeList.item(i));
		}
		
		return true;
	}
	
	/**
	 * Writes this registry to a preference store.
	 * @param preferenceStore
	 * @return <code>true</code> if the registry was written to the prefernce or 
	 * <code>false</code> otherwise.  If the information to be stored is already
	 * in the preference store this method returns false.
	 */
	protected boolean write(IPreferenceStore preferenceStore)
	{
		if(preferenceStore == null)
			return false;
		
		String xmlString = write();
		if(xmlString.equals(preferenceStore.getString(getPreferenceKey())))
			return false;
			
		preferenceStore.setValue(getPreferenceKey(), xmlString);
		return true;
	}
	
	/**
	 * Writes this registry to a xml string.
	 * @return <code>true</code> if the registry was written or <code>false</code> 
	 * otherwise.
	 */
	protected String write()
	{
		StringBuffer xml = new StringBuffer();
		
		//Header
		xml.append("<?xml");
		xml.append(XMLUtil.createXMLAttribute("version","1.0", false)).append(XMLUtil.createXMLAttribute("encoding","UTF-8", false));
		xml.append("?>");
		
		//Body
		xml.append("<AssociationMappingRegistry>");
		IAssociationMapping[] mappings = getAssociationMappings();
		for(int i = 0, maxi = mappings.length; i < maxi; i++)
			writeMapping(xml, mappings[i]);
		xml.append("</AssociationMappingRegistry>");
		
		return xml.toString();
	}
	
	/**
	 * Loads the preference setting located in in XML element.
	 * @param mappingElement
	 */
	protected void loadMapping(Element mappingElement)
	{
		String extensionPoint = XMLUtil.getValue(mappingElement, "extensionPoint");
		if(extensionPoint == null)
			return;
		String pluginId = XMLUtil.getValue(mappingElement, "pluginId");
			
		IAssociationMapping mapping = retrieveAssociationMapping(extensionPoint, pluginId);
		
		String defaultId = XMLUtil.getValue(mappingElement, "defaultId");
		if(defaultId != null)
			mapping.setDefaultAssociationDescriptor(mapping.getAssociationDescriptor(defaultId));
			
		NodeList typeDetailNodeList = XMLUtil.getChildrenByName(mappingElement, "TypeDetail");
		for(int i=0, maxi=typeDetailNodeList.getLength(); i<maxi; i++)
		{
			if(!(typeDetailNodeList.item(i) instanceof Element))
				continue;
				
			Element typeDetailElement = (Element)typeDetailNodeList.item(i);
			String type = XMLUtil.getValue(typeDetailElement, "type");
			if(type == null)
				continue;
			
			defaultId = XMLUtil.getValue(typeDetailElement, "defaultId");
			if(defaultId != null)
				mapping.setDefaultAssociationDescriptor(type, mapping.getAssociationDescriptor(type, defaultId));
			
			mapping.removeAllFromAvoidSet(type);
			NodeList avoidedDescriptorNodeList = XMLUtil.getChildrenByName(typeDetailElement, "AvoidedDescriptor");
			for(int j=0, maxj=avoidedDescriptorNodeList.getLength(); j<maxj; j++)
			{
				if(!(avoidedDescriptorNodeList.item(j) instanceof Element))
					continue;
				
				Element avoidedDescriptorElement = (Element)avoidedDescriptorNodeList.item(j);
				String id = XMLUtil.getValue(avoidedDescriptorElement, "id");
				if(id != null)
					mapping.addToAvoidedSet(type, mapping.getAssociationDescriptor(type, id));
			}
		}
	}

	/**
	 * Writes the preferences of the specified mapping in an XML format.  This method
	 * is <b>NOT</b> writing the mapping itself.  It is concerned only with the 
	 * preference settings.
	 * @param xml
	 * @param mapping
	 */
	protected void writeMapping(StringBuffer xml, IAssociationMapping mapping)
	{
		StringBuffer tempXML = new StringBuffer();
		tempXML.append("<AssociationMapping");
		tempXML.append(XMLUtil.createXMLAttribute("extensionPoint",mapping.getExtensionPoint(), false));
		tempXML.append(XMLUtil.createXMLAttribute("pluginId",mapping.getPluginId(), false));

		IAssociationDescriptor defaultDescriptor = mapping.getDefaultAssociationDescriptor();
		if(defaultDescriptor != null)
			tempXML.append(XMLUtil.createXMLAttribute("defaultId", defaultDescriptor.getId(), false));

		tempXML.append(">");
			
			
		boolean hasContent = false;
		String[] types = mapping.getTypes();
		for(int i = 0, maxi = types.length; i < maxi; i++)
		{
			defaultDescriptor = mapping.getDefaultAssociationDescriptor(types[i]);
			IAssociationDescriptor[] avoidedDescriptors = mapping.getAvoidedAssociationDescriptors(types[i]);
			
			if(((defaultDescriptor == null) || (defaultDescriptor.getId() == null)) && (avoidedDescriptors.length == 0))
				continue;
				
			hasContent = true;
			tempXML.append("<TypeDetail");
			tempXML.append(XMLUtil.createXMLAttribute("type", types[i], false));
			if(defaultDescriptor != null)
				tempXML.append(XMLUtil.createXMLAttribute("defaultId", defaultDescriptor.getId(), false));
			tempXML.append(">");
			for(int j = 0, maxj = avoidedDescriptors.length; j < maxj; j++)
			{
				if(avoidedDescriptors[j].getId() == null)
					continue;
					
				tempXML.append("<AvoidedDescriptor");
				tempXML.append(XMLUtil.createXMLAttribute("id", avoidedDescriptors[j].getId(), false));
				tempXML.append("/>");
			}
			tempXML.append("</TypeDetail>");					
		}
		tempXML.append("</AssociationMapping>");
		
		if(hasContent)
			xml.append(tempXML);
	}
}
