/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.navigator.action;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.actions.ActionGroup;

import org.eclipse.hyades.ui.extension.INavigatorContribution;
import org.eclipse.hyades.ui.internal.action.CommonNavigatorFilterSetAction;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.hyades.ui.internal.extension.NavigatorFilterSet;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * Superclass of all actions to be added to the navigator.
 * 
 * <p>This implementation is based on the 
 * {@link org.eclipse.ui.views.navigator.ResourceNavigatorActionGroup} class.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public abstract class NavigatorActionGroup 
extends ActionGroup implements IDisposable
{
	public static final String GROUP_FILTERS = "group.filters";
	
	private Separator fAdditionsGroup; 
	private Separator fNewGroup;
	private Separator fOpenGroup; 
	private Separator fDeleteGroup; 
	private Separator fSaveGroup; 
	private Separator fImportGroup; 
	private Separator fRefreshGroup; 
	private Separator fWorkbenchAdditionsGroup; 
	private Separator fPropertiesGroup; 
	
	private Separator fFiltersGroup;
	private IAction fFiltersAction;
	private IAction[] fCommonFilters;
	
	/**
	 * The navigator.
	 */
	private INavigator navigator;
	
	/**
	 * Constructs a new navigator action group and creates its actions and 
	 * action groups.
	 * @param navigator the resource navigator
	 */
	public NavigatorActionGroup(INavigator navigator)
	{
		this.navigator = navigator;
		createActions();
		createActionGroups();
	}
	
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#dispose()
	 */
	public void dispose()
	{
		navigator = null;
		disposeActionGroups();
		super.dispose();
	}

	/**
	 * Returns the navigator.
	 */
	public INavigator getNavigator()
	{
		return navigator;
	}
	
	/**
	 * Handles a key pressed event by invoking the appropriate action.
	 * Does nothing by default.
	 */
	public void handleKeyPressed(KeyEvent event)
	{
	}

	/**
	 * Creates the actions contained in this action group.
	 */
	protected void createActions() {
		fFiltersAction = new FiltersAction(navigator);
		
		List common = new ArrayList();
		NavigatorFilterSet[] filters = NavigatorExtensionUtil.getFilterSetElements(navigator.getID());
		for (int i=0;i<filters.length;++i) {
			if (filters[i].isCommon())
				common.add(new CommonNavigatorFilterSetAction(filters[i], navigator));
		}
		fCommonFilters = new IAction[common.size()];
		common.toArray(fCommonFilters);
	}
	
	/**
	 * Creates the actions groups contained in this action group.
	 */
	 protected void createActionGroups()
	 {
		fAdditionsGroup = new Separator(INavigatorContribution.GROUP_ADDITIONS);
		fNewGroup = new Separator(INavigatorContribution.GROUP_NEW); 
		fOpenGroup = new Separator(INavigatorContribution.GROUP_OPEN);
		fDeleteGroup = new Separator(INavigatorContribution.GROUP_DELETE);
		fSaveGroup = new Separator(INavigatorContribution.GROUP_SAVE);
		fImportGroup = new Separator(INavigatorContribution.GROUP_IMPORT);
		fRefreshGroup = new Separator(INavigatorContribution.GROUP_REFRESH);
		fWorkbenchAdditionsGroup =new Separator(IWorkbenchActionConstants.MB_ADDITIONS);
		fPropertiesGroup = new Separator(INavigatorContribution.GROUP_PROPERTIES);
		fFiltersGroup = new Separator(GROUP_FILTERS);
	 }
	
	protected void disposeActionGroups() {
		if (fAdditionsGroup != null) {
			fAdditionsGroup.dispose();
			fAdditionsGroup = null;
		}
		if (fNewGroup != null) {
			fNewGroup.dispose();
			fNewGroup = null;
		}
		if (fOpenGroup != null) {
			fOpenGroup.dispose();
			fOpenGroup = null;
		}
		if (fDeleteGroup != null) {
			fDeleteGroup.dispose();
			fDeleteGroup = null;
		}
		if (fSaveGroup != null) {
			fSaveGroup.dispose();
			fSaveGroup = null;
		}
		if (fImportGroup != null) {
			fImportGroup.dispose();
			fImportGroup = null;
		}
		if (fRefreshGroup != null) {
			fRefreshGroup.dispose();
			fRefreshGroup = null;
		}
		if (fWorkbenchAdditionsGroup != null) {
			fWorkbenchAdditionsGroup.dispose();
			fWorkbenchAdditionsGroup = null;
		}
		if (fPropertiesGroup != null) {
			fPropertiesGroup.dispose();
			fPropertiesGroup = null;
		}
	}
	
	/**
	 * Runs the default action in the group.
	 * Does nothing by default.
	 * @param selection the current selection
	 */
	public void runDefaultAction(IStructuredSelection selection)
	{
	}
	
	public void fillContextMenu(IMenuManager menu) {
		super.fillContextMenu(menu);
		menu.add(fAdditionsGroup);
		menu.add(fNewGroup);
		menu.add(fOpenGroup);
		menu.add(fDeleteGroup);
		menu.add(fSaveGroup);
		menu.add(fImportGroup);
		menu.add(fRefreshGroup);
		menu.add(fWorkbenchAdditionsGroup);
		menu.add(fPropertiesGroup);
		NavigatorExtensionUtil.fillContextMenu(menu, navigator.getStructuredSelection(), navigator.getID());
	}
	
	public void fillActionBars(IActionBars actionBars) {
		super.fillActionBars(actionBars);
		
		if (fCommonFilters != null && fFiltersAction != null && 
				NavigatorExtensionUtil.getFilterSetElements(navigator.getID()).length > 0) {
			IMenuManager menu = actionBars.getMenuManager();
			menu.add(fFiltersGroup);
			for (int i=0;i<fCommonFilters.length;++i)
				menu.appendToGroup(GROUP_FILTERS, fCommonFilters[i]);
			menu.appendToGroup(GROUP_FILTERS, fFiltersAction);
		}
	}
}
