/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;

/**
 * Contains String utility methods.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class StringUtil
{
	/**
	 * Returns a new string resulting from replacing all occurrences of
	 * <code>oldString</code> in the "in" string with <code>newString</code>.
	 * 
	 * <p>If <code>in</code> or <code>oldString</code> are null then this method
	 * returns null.  If <code>newString</code> is null then <code>oldString</code> is
	 * replaced by "".
	 * 
	 * @param in
	 * @param oldString
	 * @param newString
	 * @return String
	 */	
	public static String replace(String in, String oldString, String newString)
	{
		if((in == null) || (oldString == null))
			return in;
			
		int oldStringLength = oldString.length();
		if(oldStringLength == 0)
			return in;
		
		if(newString == null)
			newString = "";
		int newStringLength = newString.length();
			
		int index = 0-newStringLength;
		StringBuffer sbIn = new StringBuffer(in);
		while((index=indexOf(sbIn, oldString, index+newStringLength)) >= 0)
			sbIn.replace(index, index+oldStringLength, newString);
		
		return sbIn.toString();
	}
	
	/**
	 * Implements the {@link java.lang.String#indexOf(java.lang.String, int)} method but
	 * looking for a String in a StringBuffer.
	 * 
	 * <p>This method can be really handful when implement algorithm using StringBuffers 
	 * since it helps to eliminate the use or creation of Strings.
	 * 
	 * <p>Any argument can be equals <code>null</code>.
	 * 
	 * @param in
	 * @param str
	 * @param fromIndex
	 * @return
	 */
	public static int indexOf(StringBuffer in, String str, int fromIndex)
	{
		if(in == null)
			return -1;
		
		if(str == null)
			str = "";
			
		int lengthIn = in.length();
		int lengthStr = str.length();

		if(lengthIn < lengthStr)
			return -1;

		if(fromIndex > lengthIn)
		{
			if (lengthIn == 0 && fromIndex == 0 && lengthStr == 0)
				return 0;
			return -1;
		}
		
		if(fromIndex < 0)
			fromIndex = 0;
			
		if(lengthStr == 0)
			return fromIndex;
			
		int strPos = 0;
		for(int i=fromIndex; i<lengthIn; i++)
		{
			if(in.charAt(i) == str.charAt(strPos))
			{
				strPos++;
				if(strPos == lengthStr)
					return i-lengthStr+1;
			}
			else
				strPos = 0;	
		}
		
		return -1;
	}
	
	/**
	 * Light version of the tokenizer facilities provided by 
	 * {@link java.util.StringTokenizer}. 
	 * @param text to be tokenized
	 * @param token
	 * @param includeToken If <code>true</code> the tokens are included in the array.
	 * @return String[]
	 */
	public static String[] tokenizer(String text, String token, boolean includeToken)
	{
		if((text == null) || (token == null))
			return null;

		StringBuffer sb = new StringBuffer(text);
		List ret = new ArrayList();
		int index = 0;
		do
		{
			index = indexOf(sb, token, 0);
			if(index >= 0)
			{
				if(index > 0) ret.add(sb.substring(0, index));
				index += token.length();
				sb.delete(0, index);
				
				if(includeToken)
					ret.add(token);
			}
		}	
		while(index >= 0);
		if(sb.length() > 0)
			ret.add(sb.toString());

		return (String[]) ret.toArray(new String[ret.size()]);
	}
	
	/**
	 * Encodes a String from unicode to the specified encoding.
	 * This method returns null if the encoding is not supported
	 * @param encoding
	 * @param text
	 * @return byte[]
	 */
	public static byte[] encode(String encoding, String text)
	{
		if(text != null)
		{
			if(encoding == null)
				return text.getBytes();
				
			try
			{
				return text.getBytes(encoding);
			}
			catch(UnsupportedEncodingException e)
			{
			}
		}
		
		return null;
	}
	
	/**
	 * Decodes a byte array from specified encoding to a unicode String.
	 * This method returns null if the encoding is not supported
	 * @param encoding
	 * @param text
	 * @return byte[]
	 */
	public static String decode(String encoding, byte[] data)
	{
		if(data != null)
		{
			if(encoding == null)
				return new String(data);
				
			try
			{
				return new String(data, encoding);
			}
			catch(UnsupportedEncodingException e)
			{
			}
		}
		
		return null;
	}
	
	/**
	 * Returns <code>true</code> if all chars are in '0' - '9'
	 * or <code>false</code> otherwise.
	 * @param string
	 * @return boolean
	 */
	public static boolean isDigits(String string)
	{
		if(string == null)
			return false;
			
		for(int i=0, max=string.length(); i<max; i++)
		{
			switch(string.charAt(i))
			{
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					break;
				default:
					return false;
			}
		}
		return true;
	}
	
	/**
	 * Returns the proper case of a string.
	 * 
	 * Examples:
	 * 		this is a text => This Is A Text
	 * 		myClass	=> MyClass
	 */
	public static String toProperCase(String text)
	{
		if((text == null) || (text.length() == 0))
			return null;
			
		if(text.length() == 1)
			return text.toUpperCase();
			
		StringBuffer sb = new StringBuffer();
		String[] pieces = tokenizer(text, " ", true);
		for(int i = 0; i < pieces.length; i++)
		{
			sb.append(pieces[i].substring(0,1).toUpperCase());
			if(pieces[i].length() > 1)
				sb.append(pieces[i].substring(1));
		}
		
		return sb.toString();		
	}	
}
