/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.core;


/**
 * A SyncMessage is a synchronous message which appear at the 
 * same event occurence on both lifeline ends (sender and receiver).<br>
 * A Syn message is usually drawn horizontally.<br>
 * <br>
 * <br>
 * Usage example:
 * <pre>
 *  Frame frame;
 *  Lifeline lifeLine1;
 *  Lifeline lifeLine2;
 * 
 *	SyncMessage message = new SyncMessage();
 *	//Create a new event occurrence on each lifeline
 *	lifeline1.getNewOccurrenceIndex();
 *	lifeline2.getNewOccurrenceIndex();
 *  //Set the message sender and receiver
 *	message.setStartLifeline(lifeLine1);
 *	message.setEndLifline(lifeline2);
 *	message.setName("Message label");
 *  //add the message to the frame
 *	frame.addMessage(message);
 * </pre>
 * 
 * @see org.eclipse.hyades.uml2sd.ui.core.Lifeline Lifeline for more event occurence details 
 * @author sveyrier
 *
 */
public class SyncMessage extends BaseMessage implements ITimeRange{
	
	/**
	 * The associated message return
	 */
	protected SyncMessageReturn	messageReturn;
	
	/**
	 * The time when the message occurs
	 */
	protected double eventTime = 0;
	
	protected boolean hasTime = false;
	
	
	/**
	 * Ensure both lifelines have the same event occurence (the greater found on each lifeline)
	 */
	protected void syncLifelinesEventOccurrence()
	{
		if ((getStartLifeline()!=null)&&(getEndLifeline()!=null))
		{
			int newIndex = 0;
			if (getStartLifeline().getEventOccurrence()>getEndLifeline().getEventOccurrence())
				newIndex = getStartLifeline().getEventOccurrence();
			else newIndex = getEndLifeline().getEventOccurrence();
			getStartLifeline().setCurrentEventOccurrence(newIndex);
			getEndLifeline().setCurrentEventOccurrence(newIndex);
			setEventOccurrence(getStartLifeline().getEventOccurrence());
		}
	}
	
	/**
	 * Set the lifeLine from which the message has been sent.<br>
	 * A new event occurence will be created on this lifeLine.<br>
	 * SyncMessage must occur at the same event occurence on both lifeline, this method
	 * is reponsible to synchonise the event occurence on each lifeline (the greater value will be used).<br>
	 * This synchronization is only done if the end lifeline has already been set.
	 * @param lifeline the message sender
	 */
	public void autoSetStartLifeline(Lifeline lifeline)
	{
		lifeline.getNewEventOccurrence();
		setStartLifeline(lifeline);
	}
	
	/**
	 * Set the lifeLine which has receiver the message.<br>
	 * A new EventOccurence will be create on this lifeLine.<br>
	 * SyncMessage must occur at the same event occurence on both lifeline, this method
	 * is reponsible to synchonise the event occurence on each lifeline (the greater value will be used).<br>
	 * This synchronization is only done if the start lifeline has already been set.
	 * @param lifeline the message receiver
	 */
	public void autoSetEndLifeline(Lifeline lifeline)
	{
		lifeline.getNewEventOccurrence();
		setEndLifeline(lifeline);
	}

	/**
	 * Set the lifeLine which has receiver the message.<br>
	 * SyncMessage must occur at the same event occurence on both lifeline, this method
	 * is reponsible to synchonise the event occurence on each lifeline (the greater value will be used).<br>
	 * This synchronization is only done if the start lifeline has already been set.
	 * @param lifeline the message receiver
	 */
	public void setStartLifeline(Lifeline lifeline)
	{
		super.setStartLifeline(lifeline);
		if ((getEndLifeline()==null))
		{
			setEventOccurrence(getStartLifeline().getEventOccurrence());
		}
		else
			syncLifelinesEventOccurrence();
	}
	
	/**
	 * Set the lifeLine which has receiver the message.<br>
	 * SyncMessage must occur at the same event occurence on both lifelines, this method
	 * is reponsible to synchonise the event occurence on each lifeline (the greater value will be used).<br>
	 * This synchronization is only done if the start lifeline has already been set.
	 * @param lifeline the message receiver
	 */
	public void setEndLifeline(Lifeline lifeline)
	{
		super.setEndLifeline(lifeline);
		if ((getStartLifeline()==null))
		{
			setEventOccurrence(getEndLifeline().getEventOccurrence());
		}
		else
			syncLifelinesEventOccurrence();
	}
	
	/**
	 * Set the message return associated with this message.
	 * @param message the message return to associate
	 */
	protected void setMessageReturn(SyncMessageReturn message)
	{
		messageReturn = message;
	}
	
	/**
	 * Returns the syncMessageReturn associated to this syncMessage
	 * @return the message return
	 */
	public SyncMessageReturn getMessageReturn()
	{
		return messageReturn;
	}
	
	/**
	 * Set the time when the message occurs
	 * @param time the time when the message occurs
	 */
	public void setTime(double time)
	{
		eventTime = time;
		hasTime = true;
		if (getStartLifeline() != null&& getStartLifeline().getFrame() !=null)
		  getStartLifeline().getFrame().setHasTimeInfo(true);
		else if (getEndLifeline() != null&& getEndLifeline().getFrame() !=null)
		  getEndLifeline().getFrame().setHasTimeInfo(true);
	 }
	
	/**
	 * Returns the time when the message begin
	 * @return the time
	 */	
	 public double getLastTime()
	 {
		return eventTime;
	 }
	
	/**
	 * Returns the time when the message end
	 * @return the time
	 */	
	 public double getFirstTime()
	 {	
		return eventTime;
	 }
	 
	 public boolean hasTimeInfo()
	 {
	 	return hasTime;
	 }

}
