package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.StringTokenizer;

import org.eclipse.hyades.internal.logging.core.Constants;

public final class ParserUtilities {

	private static ResourceBundle resourceBundle = null;

	public static String getResourceString(String key) {

	    if(key != null){
		//Load ResourceBundle:
		if (resourceBundle == null) {

			//Workbench ResourceBundle:
			try {
				resourceBundle = LoggingParsersPlugin.getResourceBundle();
			}
			catch (Throwable e) {
                    //Ignore and try the hparse.jar ResourceBundle.
			}

	            //hparse.jar ResourceBundle:
			if (resourceBundle == null) {
				try {
					resourceBundle = ResourceBundle.getBundle("org.eclipse.hyades.logging.parsers.properties.plugin", Locale.getDefault());
				}
				catch (MissingResourceException m) {
					return key;
				}
			}
		}

		try {
			return (resourceBundle.getString(key.trim()).trim());
		}
		catch (Exception e) {
		}
	    }
		return key;
	}

	public static String getResourceString(String key, String argument) {
		return (getResourceString(key, new String[] { argument }));
	}

	public static String getResourceString(String key, String argumentA, String argumentB) {
		return (getResourceString(key, new String[] { argumentA, argumentB }));
	}

	public static String getResourceString(String key, String argumentA, String argumentB, String argumentC) {
		return (getResourceString(key, new String[] { argumentA, argumentB, argumentC }));
	}

	public static String getResourceString(String key, String[] arguments) {
		try {
			return (MessageFormat.format(getResourceString(key), arguments));
		}
		catch (Exception e) {
			return key;
		}
	}

	/**
	 * Displays a detailed error message in the case of an exception while parsing
	 * a log file.  The error message dispalys the number and the content of the faulty
	 * line number as well as additional messages indicating as to why the exception 
	 * occurred.
	 * 
	 * @param 	throwable		- The throwable that occurred while parsing the file
	 * 			faultyLineNum	- The faulty line number causing the exception
	 * 			faultyLine		- The content of the faulty line causing the exception
	 * 			message			- A generic message printed to indicated the log parser being used
	 * 
	 * @throws	LogParserException	- Used for displaying the error message
	 */
	public static void exceptionHandler(Throwable throwable, long faultyLineNum, String faultyLine, String message) throws LogParserException {
		StringBuffer finalMsg = new StringBuffer(ParserUtilities.getResourceString("INVALID_LINE_NUMBER_ERROR_", String.valueOf(faultyLineNum)));
		String lineSep = Constants.LINE_SEPARATOR;
		finalMsg.append(':').append(lineSep).append(lineSep).append(faultyLine).append(lineSep);
		String errorMsg = null;

		if (throwable != null)
			errorMsg = throwable.getMessage();
		if (errorMsg != null && errorMsg.length() > 0)
			finalMsg.append(errorMsg).append(lineSep);

		finalMsg.append(message);

		LogParserException lpe = new LogParserException(finalMsg.toString(), throwable);
		lpe.fillInStackTrace();
		throw lpe;
	}

	public static void exceptionHandler(Throwable throwable, String message) throws LogParserException {
		String lineSep = Constants.LINE_SEPARATOR;
		StringBuffer finalMsg = new StringBuffer();

		if (throwable != null && throwable.getMessage() != null && throwable.getMessage().length() > 0)
			finalMsg.append(throwable.getMessage()).append(lineSep);

		finalMsg.append(message);

		LogParserException lpe = new LogParserException(finalMsg.toString(), throwable);
		lpe.fillInStackTrace();
		throw lpe;
	}

	/**
	 * Checks the parameter IP address for proper IPv4 structure.  That is:
	 * 
	 * <n>.<n>.<n>.<n>
	 * 
	 * where
	 * <n> = An integer between 0 and 255  
	 * 
	 * NOTE:  This API does NOT confirm if the parameter IP address is active (e.g. accessible on the network).
	 * NOTE:  This API does NOT support IPv6 addresses.
	 * 
	 * @param ipAddress IPv4 address to check for proper IPv4 structure.  
	 * @return true is the parameter IP address is of a proper IPv4 structure, otherwise false.  
	 */
	public static boolean isValidIPAddress(String ipAddress) {

		//Running example: 9.26.157.24

		//Split the parameter IP address into period and number tokens (e.g. 9|.|26|.|157|.|24)::
		StringTokenizer tokens = new StringTokenizer(ipAddress, ".", true);

		//We must have 7 tokens (e.g. 9|.|26|.|157|.|24):
		if (tokens.countTokens() == 7) {

			try {

				//Retrieve and convert the first number token to an integer(e.g. 9):
				int number = Integer.parseInt(tokens.nextToken());

				//Check if the first number is between 0 and 255 and the second token is a period (e.g. 9.):
				if ((number >= 0) && (number <= 255) && (tokens.nextToken().equals("."))) {

					//Retrieve and convert the second number token to an integer(e.g. 26):
					number = Integer.parseInt(tokens.nextToken());

					//Check if the second number is between 0 and 255 and the fourth token is a period (e.g. 26.):
					if ((number >= 0) && (number <= 255) && (tokens.nextToken().equals("."))) {

						//Retrieve and convert the third number token to an integer(e.g. 157):
						number = Integer.parseInt(tokens.nextToken());

						//Check if the third number is between 0 and 255 and the sixth token is a period (e.g. 157.):
						if ((number >= 0) && (number <= 255) && (tokens.nextToken().equals("."))) {

							//Retrieve and convert the fourth number token to an integer(e.g. 24):
							number = Integer.parseInt(tokens.nextToken());

							//Check if the fourth number is between 0 and 255 (e.g. 24):
							if ((number >= 0) && (number <= 255))
								return true;
						}
					}
				}
			}
			catch (NumberFormatException n) {
			}
		}

		return false;
	}
}
