/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.facades.behavioral.impl;

import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.common.impl.CMNAnnotationImpl;
import org.eclipse.hyades.models.common.common.impl.CommonFactoryImpl;
import org.eclipse.hyades.models.common.configuration.impl.CFGClassImpl;
import org.eclipse.hyades.models.common.configuration.impl.CFGInstanceImpl;
import org.eclipse.hyades.models.common.configuration.impl.CFGOperationImpl;
import org.eclipse.hyades.models.common.configuration.impl.Common_ConfigurationFactoryImpl;
import org.eclipse.hyades.models.common.facades.behavioral.IAnnotation;
import org.eclipse.hyades.models.common.facades.behavioral.IDecision;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ILoop;
import org.eclipse.hyades.models.common.facades.behavioral.IMethod;
import org.eclipse.hyades.models.common.facades.behavioral.IProperty;
import org.eclipse.hyades.models.common.facades.behavioral.ISystemUnderTest;
import org.eclipse.hyades.models.common.facades.behavioral.ITargetInvocation;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestComponent;
import org.eclipse.hyades.models.common.facades.behavioral.ITestComponentInvocation;
import org.eclipse.hyades.models.common.facades.behavioral.ITestInvocation;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.facades.behavioral.IVariable;
import org.eclipse.hyades.models.common.facades.behavioral.IVerificationPoint;
import org.eclipse.hyades.models.common.fragments.BVRInteractionOperator;
import org.eclipse.hyades.models.common.fragments.Common_Behavior_FragmentsFactory;
import org.eclipse.hyades.models.common.fragments.impl.BVRCombinedFragmentImpl;
import org.eclipse.hyades.models.common.fragments.impl.BVRInteractionConstraintImpl;
import org.eclipse.hyades.models.common.fragments.impl.BVRInteractionImpl;
import org.eclipse.hyades.models.common.fragments.impl.BVRInteractionOperandImpl;
import org.eclipse.hyades.models.common.fragments.impl.Common_Behavior_FragmentsFactoryImpl;
import org.eclipse.hyades.models.common.interactions.impl.BVRExecutionOccurrenceImpl;
import org.eclipse.hyades.models.common.interactions.impl.BVRLifelineImpl;
import org.eclipse.hyades.models.common.interactions.impl.BVRMessageEndImpl;
import org.eclipse.hyades.models.common.interactions.impl.BVRPropertyImpl;
import org.eclipse.hyades.models.common.interactions.impl.Common_Behavior_InteractionsFactoryImpl;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofileFactoryImpl;
import org.eclipse.hyades.models.common.testprofile.impl.TPFBehaviorImpl;
import org.eclipse.hyades.models.common.testprofile.impl.TPFSUTImpl;
import org.eclipse.hyades.models.common.testprofile.impl.TPFTestCaseImpl;
import org.eclipse.hyades.models.common.testprofile.impl.TPFTestComponentImpl;
import org.eclipse.hyades.models.common.testprofile.impl.TPFTestSuiteImpl;
import org.eclipse.hyades.models.common.testprofile.impl.TPFValidationActionImpl;

/**
 * Factory that creates the Hyades implementation of the simple facade. 
 * 
 * @author marcelop
 * @author psun
 * @author jtoomey
 * @since 0.0.1
 */
public class HyadesFactory
{
	public static final HyadesFactory INSTANCE = new HyadesFactory();
	
	/**
	 * Creates a test suite.
	 * 
	 * @param resource 
	 * @return ITestSuite
	 */	
	public ITestSuite createTestSuite(Resource resource)
	{
		TPFTestSuiteImpl testSuiteImpl = (TPFTestSuiteImpl)Common_TestprofileFactoryImpl.eINSTANCE.createTPFTestSuite();
		resource.getContents().add(testSuiteImpl);
		testSuiteImpl.setPersistenceId(FacadeResourceFactoryImpl.PERSISTENCE_ID);
		return testSuiteImpl;
	}

	/**
	 * Creates a test case.
	 * 
	 * @return ITestCase
	 */	
	public ITestCase createTestCase()
	{
		TPFTestCaseImpl testCaseImpl = (TPFTestCaseImpl)Common_TestprofileFactoryImpl.eINSTANCE.createTPFTestCase();
		return testCaseImpl;	
	}

	/**
	 * Creates an implementor for a test suite or a test case.  
	 * 
	 * @param implementedTest
	 * @param isExternalImplementor
	 * @return IImplementor
	 */	
	public IImplementor createImplementor(ITest implementedTest, boolean isExternalImplementor)
	{
		TPFBehaviorImpl testBehavior = null;

		String implementedTestName = "";
		if(implementedTest instanceof TPFTestSuiteImpl)
		{
			testBehavior = (TPFBehaviorImpl)((TPFTestSuiteImpl)implementedTest).getBehavior();
		}
		else if(implementedTest instanceof TPFTestCaseImpl)
		{
			testBehavior = (TPFBehaviorImpl)((TPFTestCaseImpl)implementedTest).getBehavior();
		}
		
		if(testBehavior == null)
		{
			testBehavior = (TPFBehaviorImpl) Common_TestprofileFactoryImpl.eINSTANCE.createTPFBehavior();
			testBehavior.setName(implementedTest.getName() + "_behavior");
		}

		if ( implementedTest != null )
		{
			associateBehavior(testBehavior, implementedTest);
			implementedTest.getName();
		}

		TPFTestSuiteImpl testSuite;
		if ( implementedTest instanceof ITestSuite )
		{
			testSuite = (TPFTestSuiteImpl) implementedTest;
		}
		else if ( implementedTest instanceof ITestCase )
		{
			testSuite = (TPFTestSuiteImpl)((ITestCase)implementedTest).getOwner();
			if(testSuite == null)
			{
				throw new IllegalArgumentException("Test cases must be first added to test suites before implementors can be associated.");
			}

		}
		else
		{
			throw new IllegalArgumentException("ITest must be either a ITestCase or a ITestSuite.");
		}
		
		CFGClassImpl theClass = (CFGClassImpl)testSuite;
		
		return configureImplementor(
			isExternalImplementor,
			testBehavior,
			implementedTestName,
			theClass);
	}

	protected IImplementor configureImplementor(
		boolean isExternalImplementor,
		TPFBehaviorImpl testBehavior,
		String implementedTestName,
		CFGClassImpl theClass) {
		
		// Find or create an instance
		CFGInstanceImpl instance = null;
		
/*		List instances = theClass.getInstances();
		Iterator iter = instances.iterator();
		while(iter.hasNext())
		{
			CFGInstanceImpl tempInstance = (CFGInstanceImpl)iter.next();
			if(tempInstance.getClassType() == theClass)
			instance = tempInstance;
		}
*/		if(instance == null)
		{
			// Create instance
			instance = (CFGInstanceImpl) Common_ConfigurationFactoryImpl.eINSTANCE.createCFGInstance();
		
			// Associate instance with Deployable (testsuite)		
			instance.setClassType(theClass);
			theClass.getInstances().add(instance);
		}
		
		if(!isExternalImplementor)
		{
			// Create lifeline
			BVRLifelineImpl selfLifeline = (BVRLifelineImpl)Common_Behavior_InteractionsFactoryImpl.eINSTANCE.createBVRLifeline();
			selfLifeline.setName(implementedTestName + "_selfLifeline");
		
			// Since this behavior is modeled, create the mandatory Interaction.
			BVRInteractionImpl testInteraction = (BVRInteractionImpl) Common_Behavior_FragmentsFactoryImpl.eINSTANCE.createBVRInteraction();
			testInteraction.setName(implementedTestName + "_interaction");
			testBehavior.setInteraction(testInteraction);
							
			// Add lifeline to interaction
			testInteraction.getLifelines().add(selfLifeline);
		
			// Associate the lifeline to the instance
			instance.setLifeline(selfLifeline);
		}
		
		 		
		return (TPFBehaviorImpl)testBehavior;
	}

	/**
	 * Creates an implementor for a test component.  
	 * 
	 * @param implementedTest
	 * @param isExternalImplementor
	 * @return IImplementor
	 */	
	public IImplementor createImplementor(ITestComponent testComponent, String implementorName, boolean isExternalImplementor)
	{
		TPFBehaviorImpl testComponentBehavior = null;

		List implementors = testComponent.getImplementors();
		
		// TODO: Make sure we're not creating an implementor
		// with the same name as an implementor already in this
		// test component.
		
		// Create the behavior.	
		testComponentBehavior = (TPFBehaviorImpl) Common_TestprofileFactoryImpl.eINSTANCE.createTPFBehavior();
		testComponentBehavior.setName(implementorName);

		// Add the implementor to the set of implementors of the
		// test component.
		implementors.add(testComponentBehavior);
		
		configureImplementor(isExternalImplementor, testComponentBehavior, 
			implementorName, (CFGClassImpl)testComponent);
		
		return testComponentBehavior;
	}


	/**
	 * Creates a stub.  
	 * 	 
	 * @return ITestComponent 
	 */		
	public ITestComponent createTestComponent()
	{
		TPFTestComponent testComponent = Common_TestprofileFactoryImpl.eINSTANCE.createTPFTestComponent();

		return (TPFTestComponentImpl)testComponent;
	}

	/**
	 * Creates a decision.  
	 * 	 
	 * @return IDecision 
	 */		
	public IDecision createDecision()
	{	
		BVRCombinedFragmentImpl testDecisionImpl = (BVRCombinedFragmentImpl)Common_Behavior_FragmentsFactoryImpl.eINSTANCE.createBVRCombinedFragment();
		testDecisionImpl.setInteractionOperator(BVRInteractionOperator.get(BVRInteractionOperator.ALT));
		BVRInteractionOperandImpl successOperand = (BVRInteractionOperandImpl) Common_Behavior_FragmentsFactory.eINSTANCE.createBVRInteractionOperand();
		BVRInteractionOperandImpl failureOperand = (BVRInteractionOperandImpl) Common_Behavior_FragmentsFactory.eINSTANCE.createBVRInteractionOperand();
		
		testDecisionImpl.getInteractionOperands().add(0, successOperand);
		testDecisionImpl.getInteractionOperands().add(1, failureOperand);
		
		// Create an empty constraint for this decision.  
		// The caller will need to add a condition to this constraint 
		BVRInteractionConstraintImpl testConditionImpl = (BVRInteractionConstraintImpl)Common_Behavior_FragmentsFactoryImpl.eINSTANCE.createBVRInteractionConstraint();
		successOperand.setInteractionConstraint(testConditionImpl);

		return testDecisionImpl;
	}

	/**
	 * Creates a loop.  
	 * 	 
	 * @return ILoop 
	 */		
	public ILoop createLoop()
	{
		BVRCombinedFragmentImpl testLoopImpl = (BVRCombinedFragmentImpl)Common_Behavior_FragmentsFactoryImpl.eINSTANCE.createBVRCombinedFragment();
		testLoopImpl.setInteractionOperator(BVRInteractionOperator.get(BVRInteractionOperator.LOOP));
		BVRInteractionOperandImpl loopOperand = (BVRInteractionOperandImpl) Common_Behavior_FragmentsFactory.eINSTANCE.createBVRInteractionOperand();
		testLoopImpl.getInteractionOperands().add(loopOperand);

		// Create an empty constraint for this loop.  
		// The caller will need to add a condition to this constraint 
		BVRInteractionConstraintImpl testConditionImpl = (BVRInteractionConstraintImpl)Common_Behavior_FragmentsFactoryImpl.eINSTANCE.createBVRInteractionConstraint();
		loopOperand.setInteractionConstraint(testConditionImpl);
				
		return testLoopImpl;
	}

	/**
	 * Creates a system under test.  
	 * 	 
	 * @return ISystemUnderTest 
	 */		
	public ISystemUnderTest createSystemUnderTest()
	{
		TPFSUTImpl testSUTImpl = (TPFSUTImpl)Common_TestprofileFactoryImpl.eINSTANCE.createTPFSUT();
		return testSUTImpl;
	}

	/**
	 * Creates a verification point.  
	 * 	 
	 * @return IVerificationPoint 
	 */		
	public IVerificationPoint createVerificationPoint()
	{
		TPFValidationActionImpl testVPImpl = (TPFValidationActionImpl)Common_TestprofileFactoryImpl.eINSTANCE.createTPFValidationAction();
		return testVPImpl;
	}

	/**
	 * Creates a target invocation to execute the method specificed. 
	 * 	
	 * @param method 
	 * @return ITargetInvocation
	 */		
	// TODO: write another implementation of this method to associate with instance instead of assuming
	// that there is only one instance per component.  Also extend facade to allow creation of 
	// additional instances of components.
	public ITargetInvocation createTargetInvocation(IMethod method)
	{
		// Create a MessageEnd
		BVRMessageEndImpl msgEnd = (BVRMessageEndImpl)Common_Behavior_InteractionsFactoryImpl.eINSTANCE.createBVRMessageEnd();
		msgEnd.associateOperationToMessageEnd(method);
		return (ITargetInvocation) msgEnd;
	}

	/**
	 * Creates a test invocation to execute the test specified.
	 * 	 
	 * @param test
	 * @return ITestInvocation
	 */		
	public ITestInvocation createTestInvocation(ITest test)
	{
		BVRExecutionOccurrenceImpl execOccur = (BVRExecutionOccurrenceImpl)Common_Behavior_InteractionsFactoryImpl.eINSTANCE.createBVRExecutionOccurrence();
		execOccur.setInvokedTest(test);		
		return execOccur;
	}
	
	/**
	 * Creates a test component invocation to execute the specified test component 
	 * implementor.
	 * 	 
	 * @param test
	 * @return ITestInvocation
	 */		
	public ITestComponentInvocation createTestComponentInvocation(
		ITestComponent component,
		IImplementor implementor) 
	{
		BVRExecutionOccurrenceImpl execOccur = (BVRExecutionOccurrenceImpl)Common_Behavior_InteractionsFactoryImpl.eINSTANCE.createBVRExecutionOccurrence();
		execOccur.setInvokedTestComponent(component, implementor);
		return execOccur;
	}

	/**
	 * Creates a test component invocation to execute the specified test component 
	 * implementor.
	 * 	 
	 * @param test
	 * @return ITestInvocation
	 */		
	public ITestComponentInvocation createTestComponentInvocation(
		ITestComponent component,
		String implementorName) 
	{
		BVRExecutionOccurrenceImpl execOccur = (BVRExecutionOccurrenceImpl)Common_Behavior_InteractionsFactoryImpl.eINSTANCE.createBVRExecutionOccurrence();
		execOccur.setInvokedTestComponent(component, implementorName);
		return execOccur;
	}

	/**
	 * Creates a method.
	 * 	 
	 * @return IMethod
	 */			
	public IMethod createMethod()
	{
		CFGOperationImpl op = (CFGOperationImpl) Common_ConfigurationFactoryImpl.eINSTANCE.createCFGOperation();
		return op;		
	}

	/**
	 * Creates a property.
	 * 	 
	 * @return IProperty
	 */			
	public IProperty createProperty()
	{
		BVRPropertyImpl property = (BVRPropertyImpl) Common_Behavior_InteractionsFactoryImpl.eINSTANCE.createBVRProperty();
		return property;
	}

	/**
	 * Creates a variable.
	 * 	 
	 * @return IVariable
	 */				
	public IVariable createVariable()
	{
		CFGInstanceImpl instance = (CFGInstanceImpl) Common_ConfigurationFactoryImpl.eINSTANCE.createCFGInstance();
		return instance;
	}
	
	public IAnnotation createAnnotation()
	{
		CMNAnnotationImpl instance = (CMNAnnotationImpl) CommonFactoryImpl.eINSTANCE.createCMNAnnotation();
		return instance;
	}

	/**
	 * Given a unique indentifier and a test suite, returns the object associated 
	 * with that identifier in the test suite.  Returns null if the object is not 
	 * int he test suite.
	 * 	 
	 * @param testSuite
	 * @param id
	 * @return Object
	 */				
	public Object getObjectByID(ITestSuite testSuite, String id)
	{
		if(!(testSuite instanceof EObject))
		{
			throw new IllegalArgumentException();
		}
		Object returnObject = null;
		
		// find the object
		returnObject = ((EObject)testSuite).eResource().getEObject(id);
		if(returnObject != null)
		{
			return returnObject;
		}
		else
		{
			// if it's not in the given suite, look in all the referenced suites
			List suites = ((TPFTestSuiteImpl)testSuite).getReferencedSuites();
			if(suites == null)
			{
				return null;
			}
			Iterator iter = suites.iterator();
			while(iter.hasNext())
			{
				returnObject = getObjectByID((ITestSuite)iter.next(), id);
				if(returnObject != null)
				{
					return returnObject;	
				}
			}
		}
		return null;		
	}

	/**
	 * Given an object, get its test suite.  If the object is not part of
	 * a test suite, return null.
	 * 	 
	 * @param object
	 * @return ITestSuite
	 */				
	public ITestSuite getTestSuiteFromObject(Object object)
	{
		if(!(object instanceof EObject))
		{
			throw new IllegalArgumentException();
		}
		
		Object returnObject = null;
		EObject rootContainer = EcoreUtil.getRootContainer((EObject)object);
		if(rootContainer instanceof ITestSuite)
		{
			return (ITestSuite)rootContainer;
		}
		return null;		
	}
	
	public ITestSuite loadTestSuite(String xmiFile)
	{
		ResourceSet resourceSet = new ResourceSetImpl();
		Resource suiteResource = resourceSet.getResource(URI.createFileURI(xmiFile), true);
		return (ITestSuite)suiteResource.getContents().get(0);				
		
	}

	protected void associateBehavior( TPFBehaviorImpl behavior, ITest implementedTest )
	{
		if(implementedTest instanceof TPFTest)
		{
			TPFTest test = (TPFTest)implementedTest;
			test.setBehavior(behavior);
		}
	}
	
	public TPFDeployment loadDeployment(String xmiFile)
	{
		ResourceSet resourceSet = new ResourceSetImpl();
		Resource suiteResource = resourceSet.getResource(URI.createFileURI(xmiFile), true);
		return (TPFDeployment)suiteResource.getContents().get(0);		
	}
}
