/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.sdb.internal.util;

import java.util.Locale;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.help.WorkbenchHelp;

import org.eclipse.hyades.models.internal.sdb.*;
import org.eclipse.hyades.sdb.internal.SDbPlugin;
import org.eclipse.hyades.sdb.internal.editors.EditorPage;
import org.eclipse.hyades.sdb.internal.editors.SymptomDBEditorDetailsPage;
import org.eclipse.hyades.sdb.internal.help.ContextIds;

public class SymptomDBSearchDialog extends Dialog implements SelectionListener {

	private Text txtPattern;
	private Button chkCase;
	private Button btnFind;
	private Button btnCancel;
	private Button btnUp;
	private Button btnDown;
	private Label lblMessage;
	private SymptomDBTextSearch textSearch = null;
	private SDBRuntime root;
	private SymptomDBEditorDetailsPage page;
	private boolean enabled = true;
	
	/**
	 * Constructor for SymptomDBSearchDialog.
	 * @param parentShell
	 */
	public SymptomDBSearchDialog(Shell parentShell, SDBRuntime root, SymptomDBEditorDetailsPage page) {
		super(parentShell);
		this.root = root;
		this.page = page;
		setShellStyle(SWT.CLOSE | SWT.MODELESS | SWT.BORDER | SWT.TITLE);
	}
	
	protected Control createContents(Composite parent) {
	
			getShell().setText(SDbPlugin.getResourceString("STR_SEARCHDIALOG_TITLE"));
			Composite c = new Composite(parent, SWT.NONE);
			GridLayout layout = new GridLayout();
			//GridData data = GridUtil.createFill();
			layout.numColumns = 2;
			c.setLayout(layout);
			//c.setLayoutData(data);

			createPatternGroup(c);			

			createButtonsGroup(c);
			createOptionsGroup(c);
			createFooter(c);

			btnFind.addSelectionListener(this);
			btnCancel.addSelectionListener(this);

			txtPattern.setFocus();
			txtPattern.addModifyListener(new ModifyListener(){
				public void modifyText(ModifyEvent e){
					if(txtPattern.getText().length()==0)
						btnFind.setEnabled(false);
					else
						if(enabled)
							btnFind.setEnabled(true);
				};
			} );

			return c;
	}
	
    private void createButtonsGroup(Composite c){

			Composite buttons = new Composite(c, SWT.NONE);
			GridLayout layout = new GridLayout();
			layout.numColumns = 1;
			GridData data = GridUtil.createHorizontalFill();
			data.verticalSpan = 2;
			data.verticalAlignment = GridData.BEGINNING;
			buttons.setLayout(layout);
			buttons.setLayoutData(data);

			btnFind = new Button(buttons, SWT.PUSH);
			btnFind.setText(SDbPlugin.getResourceString("STR_FINDNEXT_LABEL"));
			data = new GridData();
			//Need wider button for Chinese, and PTBR translation of "Find Next", i.e. "Localizar Proxima".
			if (Locale.getDefault().toString().equals("en_US"))
				data.widthHint = 75;
			else
				data.widthHint = 100;
			btnFind.setLayoutData(data);

			btnCancel = new Button(buttons, SWT.PUSH);
			btnCancel.setText(SDbPlugin.getResourceString("STR_CANCEL_LABEL"));
			data = new GridData();
			//Need wider button for Chinese, and PTBR translation of "Find Next", i.e. "Localizar Proxima".
			if (Locale.getDefault().toString().equals("en_US"))
				data.widthHint = 75;
			else
				data.widthHint = 100;
			btnCancel.setLayoutData(data);

			// set the dialog's default button 
			Shell shell = c.getShell();
			if (shell != null) {
				shell.setDefaultButton(btnFind);
			}

			btnFind.setEnabled(false);

    }

	private void createPatternGroup(Composite c){

			Composite pattern = new Composite(c, SWT.NONE);
			GridLayout layout = new GridLayout();
			layout.numColumns = 2;
			pattern.setLayoutData(GridUtil.createHorizontalFill());
			pattern.setLayout(layout);
			
			Label lblPattern = new Label(pattern, SWT.NONE);
			lblPattern.setText(SDbPlugin.getResourceString("STR_FINDWHAT_LABEL"));
//			lblPattern.setLayoutData(GridUtil.createHorizontalFill());			
			txtPattern = new Text(pattern, SWT.BORDER);
			GridData data = new GridData(GridData.FILL_HORIZONTAL);
			data.widthHint = 175;
			txtPattern.setLayoutData(data);					
			WorkbenchHelp.setHelp(txtPattern, ContextIds.SYMDB_EDITOR_FIND_WHAT);
	}
	
	private void createOptionsGroup(Composite c){

			Composite options = new Composite(c, SWT.NONE);
			GridLayout layout = new GridLayout();
			layout.numColumns = 2;
			options.setLayout(layout);
			options.setLayoutData(GridUtil.createHorizontalFill());
			
			chkCase = new Button(options, SWT.CHECK);
			chkCase.setText(SDbPlugin.getResourceString("STR_MATCHCASE_LABEL"));
			chkCase.setLayoutData(GridUtil.createHorizontalFill());
			WorkbenchHelp.setHelp(chkCase, ContextIds.SYMDB_EDITOR_FIND_CASE);
			
			Group direction = new Group(options, SWT.NONE);
			layout = new GridLayout();
			
			//Display buttons in two columns only if English locale.
			if (Locale.getDefault().toString().equals("en_US"))
				layout.numColumns = 2;
			else
				layout.numColumns = 1;
			
			direction.setText(SDbPlugin.getResourceString("STR_DIRECTION_LABEL"));
			direction.setLayout(layout);
			GridData data =  new GridData(GridData.FILL_HORIZONTAL);
			data.horizontalAlignment = GridData.END;
			data.widthHint = 100;
			direction.setLayoutData(data);
			btnUp = new Button(direction, SWT.RADIO);
			btnUp.setText(SDbPlugin.getResourceString("STR_DIR_UP_LABEL"));
			btnDown = new Button(direction, SWT.RADIO);
			btnDown.setText(SDbPlugin.getResourceString("STR_DIR_DOWN_LABEL"));
			btnDown.setSelection(true);
		
	}
	
	private void createFooter(Composite c){
	
		lblMessage = new Label(c, SWT.NONE);
		GridData data = new GridData(GridData.FILL_HORIZONTAL);
		data.horizontalSpan = 2;
		lblMessage.setLayoutData(data);
		
	}
		

	public void widgetDefaultSelected(SelectionEvent e){
	}
		
	public void widgetSelected(SelectionEvent e){
		
		if(e.widget==btnCancel)
			close();
		
		if(e.widget==btnFind){
			lblMessage.setText("");
			Object[] sel = getTextSearchEngine().search(txtPattern.getText(), btnDown.getSelection()?1:-1, chkCase.getSelection());
			setSelection(sel);
		}
			
	}	

	private Object[] getStartNode(){

			TreeItem[] sel = page.getViewer().getTree().getSelection();
			Object[] start = new Object[]{null,null,null,null};
			if(sel.length > 0){			
				Object object = sel[0].getData();
				if(object instanceof SDBRuntime){
					start[0] = object;
				}
				else if(object instanceof SDBSymptom){
					start[0] = root;
					start[1] = object;
				}
				else if(object instanceof SDBSolution){
					start[0] = root;
					TreeItem parent = sel[0].getParentItem();
					if(parent!=null){
						start[1] = sel[0].getParentItem().getData();
						start[2] = sel[0].getData();
					}
				}
				else if (object instanceof SDBDirective){
					start[0] = root;
					TreeItem parent = sel[0].getParentItem();

					if(parent!=null){
						TreeItem grand = parent.getParentItem();
						if(grand!=null){					
							start[1] = grand.getData();
							start[2] = parent.getData();
							start[3] = sel[0].getData();						
						}
					}
				}
			}
			return start;	
	}
	
	public  SymptomDBTextSearch getTextSearchEngine(){

		if(textSearch==null){
				
			textSearch = new SymptomDBTextSearch();
		}
		root = (SDBRuntime) page.getTreeRoot();		
		Object[] start = getStartNode();
		textSearch.initialize(root, start);
				
		return textSearch;	 
	}

	public void setTarget(EditorPage newPage){
		this.page = (SymptomDBEditorDetailsPage)newPage;
		
	}
	
	public void enableButton(boolean newVal){
		
		enabled = newVal;
		if(!newVal || (newVal && txtPattern.getText().length() > 0))
			btnFind.setEnabled(newVal);
	}
	
	public Shell getParentShell() {
		return super.getParentShell();
	}

	
	private void setSelection(Object[] sel){
		
		if(sel[0]==null){
			lblMessage.setText(SDbPlugin.getResourceString("STR_STRING_NOT_FOUND"));
			return;
		}
		
		page.setSelection(sel);
	}
	
	/**
	 * @see org.eclipse.jface.window.Window#close()
	 */
	public boolean close() {
		page.setDialog(null);
		return super.close();
	}

}
