/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM Rational - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.remote;

import java.util.HashMap;

import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.invocation.IRemoteObject;
import org.eclipse.hyades.execution.invocation.Marshaller;
import org.eclipse.hyades.execution.invocation.RemoteInvocationException;
import org.eclipse.hyades.execution.invocation.ReturnData;

/**
 * The execution component factory.
 * 
 * Each agent VM will have an instance of the factory. The factory acts as
 * it's own stub and is capable of remotely invoking the appropriate methods.
 * 
 * The factory's state is initialized when the class is loaded.
 * Something on remote agents needs to initiate the loading of this class.
 * A simple call to <code>RemoteFactory.getInstance()</code> will do.
 */
public class ExecutionComponentFactoryImpl implements IExecutionComponentFactory, IRemoteObject  {

	private static ExecutionComponentFactoryImpl instance;
	
	private static ISession sessionContext;
	
	private HashMap components=new HashMap();
	
	private ExecutionComponentFactoryImpl(ISession session){
		sessionContext=session;
		Marshaller.addInstanceToMap(getUniqueId(), this);
	}

	/**
	 * Get the component factory in this VM. Each VM has it's own factory.
	 * Each factory shares the same, well known id and provides the bootstrap
	 * for remote object instantiation.
	 * 
	 * @return
	 */
	public static ExecutionComponentFactoryImpl getInstance(ISession session) {
		if(instance==null) {
			instance = new ExecutionComponentFactoryImpl(session);
		}
		return instance;
	}
	
	public ISession getSessionContext() {
		return sessionContext;
	}

	/**
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#setSessionContext(org.eclipse.hyades.execution.core.ISession)
	 */
	public void setSessionContext(ISession session) {
		throw new UnsupportedOperationException("The session context of the factory should not be set via this method");
	}

	/**
	 * Overrides the superclass method to return a well-known unique id.
	 * 
	 * @return an <code>Integer</code> wrapper of the well-known value (-1)
	 *         associated with the component factory
	 */
	public Integer getUniqueId() {
		return new Integer(-1);
	}

	/**
	 * Create a component on a remote node.
	 * This method gets executed on the local side; it delegates the component
	 * creation to <code>doCreateExecutionComponentByType(String)</code>, which
	 * is executed on the remote side.
	 * 
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#createExecutionComponentByType(java.lang.String)
	 */
	public IExecutionComponent createExecutionComponentByType(String type) {	
		return null;
	}
	
	public IExecutionComponent createExecutionComponentByType(Integer remoteId, String type) {
		/* create the local instance first */
		ClassRelationship rel=(ClassRelationship)components.get(type);
		if(rel!=null) {
			try {
				Class impl=rel.impl;
				Class skel=rel.skeletonClass;
				IExecutionComponent implInstance=(IExecutionComponent)impl.newInstance();
				ExecutionComponentSkeleton skelInstance=(ExecutionComponentSkeleton)skel.newInstance();
				skelInstance.setId(remoteId);
				skelInstance.setDelegate(implInstance);
				skelInstance.setSessionContext(sessionContext);
				
				Marshaller.addInstanceToMap(skelInstance.getUniqueId(), skelInstance);
				skelInstance.init();
				
				return (IExecutionComponent)skelInstance;
			}
			catch(Throwable e) {
				throw new RemoteInvocationException(e);
			}
		}
		else {
			throw new RemoteInvocationException("Factory not configured for this type");
		}
	}
	
	public IRemoteObject createRemoteObjectByType(Integer remoteId, String type) {
		/* create the local instance first */
		ClassRelationship rel=(ClassRelationship)components.get(type);
		if(rel!=null) {
			try {
				Class impl=rel.impl;
				Class skel=rel.skeletonClass;
				Object implInstance=impl.newInstance();
				RemoteObjectSkeleton skelInstance=(RemoteObjectSkeleton)skel.newInstance();
				skelInstance.setId(remoteId);
				skelInstance.setDelegate(implInstance);
				skelInstance.setSessionContext(sessionContext);
				
				Marshaller.addInstanceToMap(skelInstance.getUniqueId(), skelInstance);
				skelInstance.init();
				
				return skelInstance;
			}
			catch(Throwable e) {
				throw new RemoteInvocationException(e);
			}
		}
		else {
			throw new RemoteInvocationException("Factory not configured for this type");
		}
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#addExecutionComponent(java.lang.String, java.lang.String)
	 */
	public void addExecutionComponent(String type, String implClass) throws ClassNotFoundException {
		/* Locate the impl class locally */
		try {
			Class clazz=Class.forName(
				implClass, true, this.getClass().getClassLoader()/*new UpdateablePathClassLoader(ExecutionComponentFactoryImpl.class.getClassLoader())*/);
						
			ClassRelationship rel=new ClassRelationship();
			rel.impl=clazz;
			components.put(type, rel);
		}
		catch(ClassNotFoundException e) {
			throw new RemoteInvocationException(e);
		}
		
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#addStub(java.lang.String, java.lang.String)
	 */
	public void addStub(String type, String stubClass) throws ClassNotFoundException {
		/* This will never be called as we are on the remote side */
		
	}

	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#addSkeleton(java.lang.String, java.lang.String)
	 */
	public void addSkeleton(String type, String skeletonClass) throws ClassNotFoundException {
		try {
			Class clazz=Class.forName(
				skeletonClass, true, this.getClass().getClassLoader() /*new UpdateablePathClassLoader(ExecutionComponentFactoryImpl.class.getClassLoader())*/);
			
			/* Update our component table */
			ClassRelationship rel=(ClassRelationship)components.get(type);
			if(rel!=null) {
				rel.skeletonClass=clazz;
			}
			
		}
		catch(ClassNotFoundException e) {
			throw new RemoteInvocationException(e);
		}
		
	}
	

	/**
	 * Delegate a remote method call to the target instance. This method
	 * orchestrates the marshalling of the method call and the remote invocation
	 * of the method against the target.
	 * 
	 * @param callArgs
	 * @param call
	 * @return
	 * @throws RemoteInvocationException
	 */
	public ReturnData delegateRemoteCall(Class[] argTypes, Object[] callArgs, String call)
		throws RemoteInvocationException {
		return null;
	}
	
	/**
	 * This implementation returns the factory instance. The factory is
	 * its own delegate.
	 *   
	 * @return
	 */
	public Object getDelegate() {
		return this;
	}

	/**
	 * This implementation does nothing. The factory is its own delegate.
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#setDelegate(java.lang.Object)
	 */
	public void setDelegate(Object delegate) {
	}

	/**
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#init()
	 */
	public void init() {
	}
}

class ClassRelationship {
	public Class impl;	
	public Class skeletonClass;
}
