/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.remote;

import java.util.Stack;

import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.impl.ExecutionComponentImpl;
import org.eclipse.hyades.execution.invocation.CallData;
import org.eclipse.hyades.execution.invocation.Invoker;
import org.eclipse.hyades.execution.invocation.Marshaller;
import org.eclipse.hyades.execution.invocation.ReturnData;
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.remote.AgentControllerListener;
import org.eclipse.hyades.internal.execution.remote.AgentControllerUnavailableException;
import org.eclipse.hyades.internal.execution.remote.CustomCommandHandler;
import org.eclipse.hyades.internal.execution.remote.RemoteClientListener;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;

public class SessionImpl  extends ExecutionComponentImpl implements ISession{

	/* Fields for controlling exit */
	private Object completionLock=new Object();
	private boolean completed=false;
	
	private boolean initializedWithRAC=false;
	
	/* The agent we use to communicate with the client */
	RemoteComponentSkeleton agent;
	
	private final Stack messageQueue=new Stack();
	

	public SessionImpl(String name, String type) {
		agent=new RemoteComponentSkeleton(name,type);
	}
	
	/**
	 * 
	 *
	 */
	public void init() {
		
		/* Add a listener for RAC events */
		agent.addAgentControllerListener(new AgentControllerListener() {

			public void agentControllerActive() {
				initializedWithRAC=true;
				
			}

			/* If somebody shuts down the RAC we are rendered useless in this scenario
			 * so we should exit
			 */
			public void agentControllerInactive() {
				if(initializedWithRAC) {
					System.exit(-2);
				}
				
			}
		});
		
		/* Add a listener for events when the client attaches/detaches */
		agent.addRemoteClientListener(new RemoteClientListener() {

			public void clientActive() {
				/* RKD:  This method will never be invoked at this point.  It is
				 * not currently implemented in the RAC but I feel it will be needed.
				 */
				
			}

			/**
			 * bugzilla_75084 (bugzilla_78259)
			 * 
			 * There seems to be differences in behavior between using System.exit(-3)
			 * and waiting for the VM to exit naturally after the last user thread has
			 * completed.  This difference in behavior is seen in the IBM J9 VM but not
			 * the Sun VM.  To provide consistent behavior in both VMs and since 
			 * sessions might not be explicitly released due to reusing of sessions
			 * in various clients, to play it safe, we're releasing the session
			 * manually when the client is found to be inactive.  
			 * 
			 * If System.exit(-3) is used to terminate the VM, the "java.exe" process 
			 * stays around and appears non-reusable as a Java VM as even remote debugger
			 * sessions can no longer attach to it (this has been reproducible only in 
			 * the IBM J9 VM so far, the Sun VM does not seem to have this problem).
			 */
			public void clientInactive() {
				/* The client driving the session has exited.  We need to shutdown. */
				SessionImpl.this.release();
			}
			
		});
		
		/*  Add our command handler */
		agent.addCommandListener(new CustomCommandHandler() {

			public void handleCommand(CustomCommand command) {
				/* This is where our agent is given the commands from
				 * the remote client.  We will put this on the local
				 * worker thread where they can be dispatched.
				 */ 	
				synchronized(messageQueue) {
					messageQueue.push(command);
					if(messageQueue.size()==1) {
						messageQueue.notify();
					}
				}			 	
			}
					
		});

		/* Create our factory */
		ExecutionComponentFactoryImpl.getInstance(this);
	
		/* try and handshake with the RAC */
		try {
			agent.initialize();
		}
		catch(AgentControllerUnavailableException e) {
			/* The RAC is not running */
		}
			
	}
	
	/**
	 *  This method blocks the calling thread until the exit() method
	 * is invoked on another thread
	 */ 
	public void waitForExit() {
		while(!completed) {
			synchronized(completionLock) {
				try {
					completionLock.wait();
				}
				catch(InterruptedException e) {
					/* We can ignore this */
				}
			}
		}
	}
	
	/**
	 *  This method releases the blocked thread inside of waitForExit() method
	 * 
	 */
	private void exit() {
		/* If we lose our RAC connection we will exit */
		synchronized(completionLock) {
			completed=true;
			completionLock.notifyAll();
		}
	}

	/**
	 * @see org.eclipse.hyades.execution.core.ISession#release()
	 */
	public void release() {
		exit();
		
	}
	
	public void replyRemote(CustomCommand command) {
		byte[] data=command.getDataBinary();
		agent.broadcastMessage(data, 0, data.length, 0);
	}
	
	public void handleCommands() {
		while(true) {
		
			synchronized(messageQueue) {		
				while(messageQueue.size()==0){
					synchronized(messageQueue) {
						try {
						
							/* Wait for upto two seconds for a new message to arrive */
							messageQueue.wait(2000);
							
							/* If the queue is empty ensure we have no exited */
							if(messageQueue.size()==0){
								synchronized(completionLock) {
									if(completed) {
										return;
									}
								}
							}
							
						}
						catch(InterruptedException e) {
						
						}
					}
				}

				BinaryCustomCommand command=(BinaryCustomCommand)messageQueue.pop();
				CallData callData = null;
				try {
					callData = Marshaller.unmarshalMethodCall(command.getDataBinary());
				}
				catch ( Exception e ) {
					e.printStackTrace(System.err);
					return;
				}

				try {
					/* Do the invocation and get the return value */
					ReturnData returnData = Invoker.invoke(callData);

					/* Marshal back our response */
					command.setData(Marshaller.marshalReturnValue(returnData));
					replyRemote(command);
				}
				catch(Throwable e) {
					e.printStackTrace();
				}
			}
		
		}
	}

}