package org.eclipse.hyades.logging.events;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamField;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.hyades.logging.events.exceptions.InvalidExtendedDataElementTypeException;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * This is the class that embodies the Extended Data Element as described in 
 * "Canonical Situation Data Format: The Common Base Event V1.0.1 
 * ACAB.BO0301.0.1"
 * 
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @author Jason Cornpropst
 * @version 1.0.1
 * @since 1.0
 * @see IExtendedDataElement 
 * @deprecated This class is deprecated as of 2004-05-17. Use the corresponding class available in org.eclipse.hyades.logging.events.cbe*
 */
public class ExtendedDataElementImpl implements IExtendedDataElement, Serializable, Cloneable {

    //define the serialiVersionUID for serialization
    //if this object is ever changed that breaks backward compatability
    //this value MUST be changed Every effort should be made not to
    //break backward compatability
    static final long serialVersionUID = 6633427043459375461L;

    // define the fields that will be serialized. 
    // if any new variables are added to the class
    // this table should be updated along with the readObject and writeObject methods
    private static final ObjectStreamField[] serialPersistentFields = { new ObjectStreamField("name", String.class), new ObjectStreamField("type", int.class), new ObjectStreamField("values", Object.class), new ObjectStreamField("valuesType", int.class), new ObjectStreamField("children", IExtendedDataElement[].class)};

    /**
     * Fields.
     */
    private String name = null;

    private int type = -1;

    /**
     * Holder for the data of this event property.
     */
    private Object values = null;

    /**
     * Internal type identifier for the values object used for performance reasons.  
     * This value is externally immutable.  
     */
    private int valuesType = -1;

    /**
     * The associations elements with the other artifacts.
     */
    private ArrayList children = null;

    /**
     * @deprecated Not required for CBE 1.0.1.
     */
    private String id = null;

    //~ Constructors .....................................................................
    /**
     * 
     */
    protected ExtendedDataElementImpl() {
        super();
    }

    //~ Methods ..........................................................................

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildDataElement(IExtendedDataElement)
     */
    public void addChildDataElement(IExtendedDataElement element) {

        if (children == null) {
            children = new ArrayList();
        }

        this.children.add(element);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getChildDataElement(java.lang.String)
     */
    public IExtendedDataElement getChildDataElement(String name) {

        if (children != null) {

            for (int counter = 0; counter < children.size(); counter++) {

                if (((IExtendedDataElement) (children.get(counter))).getName().equals(name)) {
                    return ((IExtendedDataElement) (children.get(counter)));
                }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getChildDataElements()
     */
    public List getChildDataElements() {

		if (children == null) {
			children = new ArrayList();
		}

        return children;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#removeChildDataElement(String)
     */
    public void removeChildDataElement(String name) {

        if (children != null) {

            for (int counter = 0; counter < children.size(); counter++) {

                if (((IExtendedDataElement) (children.get(counter))).getName().equals(name)) {
                    children.remove(counter--);
                }
            }
        }
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#clearChildDataElements()
     */
    public void clearChildDataElements() {
        children = null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getValues()
     */
    public List getValues() {

        if (values != null) {
            return (Arrays.asList(getValuesAsStringArray()));
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getValuesAsStringArray()
     */
    public String[] getValuesAsStringArray() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return ((String[]) (values));
                    }

                case TYPE_BYTE_ARRAY_VALUE :
                    {

                        byte[] byteArray = ((byte[]) (values));
                        String[] stringArray = new String[byteArray.length];

                        for (int counter = 0; counter < byteArray.length; counter++) {
                            stringArray[counter] = String.valueOf(byteArray[counter]);
                        }

                        return stringArray;
                    }

                case TYPE_SHORT_ARRAY_VALUE :
                    {

                        short[] shortArray = ((short[]) (values));
                        String[] stringArray = new String[shortArray.length];

                        for (int counter = 0; counter < shortArray.length; counter++) {
                            stringArray[counter] = String.valueOf(shortArray[counter]);
                        }

                        return stringArray;
                    }

                case TYPE_INT_ARRAY_VALUE :
                    {

                        int[] intArray = ((int[]) (values));
                        String[] stringArray = new String[intArray.length];

                        for (int counter = 0; counter < intArray.length; counter++) {
                            stringArray[counter] = String.valueOf(intArray[counter]);
                        }

                        return stringArray;
                    }

                case TYPE_LONG_ARRAY_VALUE :
                    {

                        long[] longArray = ((long[]) (values));
                        String[] stringArray = new String[longArray.length];

                        for (int counter = 0; counter < longArray.length; counter++) {
                            stringArray[counter] = String.valueOf(longArray[counter]);
                        }

                        return stringArray;
                    }

                case TYPE_FLOAT_ARRAY_VALUE :
                    {

                        float[] floatArray = ((float[]) (values));
                        String[] stringArray = new String[floatArray.length];

                        for (int counter = 0; counter < floatArray.length; counter++) {
                            stringArray[counter] = String.valueOf(floatArray[counter]);
                        }

                        return stringArray;
                    }

                case TYPE_DOUBLE_ARRAY_VALUE :
                    {

                        double[] doubleArray = ((double[]) (values));
                        String[] stringArray = new String[doubleArray.length];

                        for (int counter = 0; counter < doubleArray.length; counter++) {
                            stringArray[counter] = String.valueOf(doubleArray[counter]);
                        }

                        return stringArray;
                    }

                case TYPE_BOOLEAN_ARRAY_VALUE :
                    {

                        boolean[] booleanArray = ((boolean[]) (values));
                        String[] stringArray = new String[booleanArray.length];

                        for (int counter = 0; counter < booleanArray.length; counter++) {
                            stringArray[counter] = String.valueOf(booleanArray[counter]);
                        }

                        return stringArray;
                    }

                case TYPE_HEX_BINARY_VALUE :
                    {

                        char[] charArray = ((char[]) (values));
                        String[] stringArray = new String[charArray.length];

                        for (int counter = 0; counter < charArray.length; counter++) {
                            stringArray[counter] = String.valueOf(charArray[counter]);
                        }

                        return stringArray;
                    }

                default :
                    {
                        return (new String[] { values.toString()});
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getId()
     * @deprecated Not required for CBE 1.0.1.
     */
    public String getId() {
        return id;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getName()
     */
    public String getName() {
        return name;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getTypeValue()
     */
    public int getTypeValue() {
        return type;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getType()
     */
    public String getType() {

        switch (type) {
            case TYPE_BYTE_VALUE :
                return TYPE_BYTE;

            case TYPE_SHORT_VALUE :
                return TYPE_SHORT;

            case TYPE_INT_VALUE :
                return TYPE_INT;

            case TYPE_LONG_VALUE :
                return TYPE_LONG;

            case TYPE_BOOLEAN_VALUE :
                return TYPE_BOOLEAN;

            case TYPE_FLOAT_VALUE :
                return TYPE_FLOAT;

            case TYPE_DOUBLE_VALUE :
                return TYPE_DOUBLE;

            case TYPE_STRING_VALUE :
                return TYPE_STRING;

            case TYPE_DATE_TIME_VALUE :
                return TYPE_DATE_TIME;

            case TYPE_BYTE_ARRAY_VALUE :
                return TYPE_BYTE_ARRAY;

            case TYPE_SHORT_ARRAY_VALUE :
                return TYPE_SHORT_ARRAY;

            case TYPE_INT_ARRAY_VALUE :
                return TYPE_INT_ARRAY;

            case TYPE_LONG_ARRAY_VALUE :
                return TYPE_LONG_ARRAY;

            case TYPE_BOOLEAN_ARRAY_VALUE :
                return TYPE_BOOLEAN_ARRAY;

            case TYPE_FLOAT_ARRAY_VALUE :
                return TYPE_FLOAT_ARRAY;

            case TYPE_DOUBLE_ARRAY_VALUE :
                return TYPE_DOUBLE_ARRAY;

            case TYPE_STRING_ARRAY_VALUE :
                return TYPE_STRING_ARRAY;

            case TYPE_DATE_TIME_ARRAY_VALUE :
                return TYPE_DATE_TIME_ARRAY;

            case TYPE_HEX_BINARY_VALUE :
                return TYPE_HEX_BINARY;

            case TYPE_NO_VALUE_VALUE :
                return TYPE_NO_VALUE;
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setId(String)
     * @deprecated Not required for CBE 1.0.1.
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setName(String)
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @see IExtendedDataElement#setType(int)
     */
    public void setType(int type) {

        if ((type >= 0) && (type <= TYPE_NO_VALUE_VALUE)) {
            this.type = type;
        }
        else {
            throw new InvalidExtendedDataElementTypeException("setType was called with an invalid datatype value of  " + getType());
        }
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setType(String)
     */
    public void setType(String type) {

        if (type.equals(TYPE_BYTE)) {
            this.type = TYPE_BYTE_VALUE;
        }
        else if (type.equals(TYPE_BOOLEAN)) {
            this.type = TYPE_BOOLEAN_VALUE;
        }
        else if (type.equals(TYPE_SHORT)) {
            this.type = TYPE_SHORT_VALUE;
        }
        else if (type.equals(TYPE_INT)) {
            this.type = TYPE_INT_VALUE;
        }
        else if (type.equals(TYPE_LONG)) {
            this.type = TYPE_LONG_VALUE;
        }
        else if (type.equals(TYPE_FLOAT)) {
            this.type = TYPE_FLOAT_VALUE;
        }
        else if (type.equals(TYPE_DOUBLE)) {
            this.type = TYPE_DOUBLE_VALUE;
        }
        else if (type.equals(TYPE_DATE_TIME)) {
            this.type = TYPE_DATE_TIME_VALUE;
        }
        else if (type.equals(TYPE_STRING)) {
            this.type = TYPE_STRING_VALUE;
        }
        else if (type.equals(TYPE_BYTE_ARRAY)) {
            this.type = TYPE_BYTE_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_BOOLEAN_ARRAY)) {
            this.type = TYPE_BOOLEAN_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_SHORT_ARRAY)) {
            this.type = TYPE_SHORT_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_INT_ARRAY)) {
            this.type = TYPE_INT_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_LONG_ARRAY)) {
            this.type = TYPE_LONG_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_FLOAT_ARRAY)) {
            this.type = TYPE_FLOAT_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_DOUBLE_ARRAY)) {
            this.type = TYPE_DOUBLE_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_DATE_TIME_ARRAY)) {
            this.type = TYPE_DATE_TIME_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_STRING_ARRAY)) {
            this.type = TYPE_STRING_ARRAY_VALUE;
        }
        else if (type.equals(TYPE_HEX_BINARY)) {
            this.type = TYPE_HEX_BINARY_VALUE;
        }
        else if (type.equals(TYPE_NO_VALUE)) {
            this.type = TYPE_NO_VALUE_VALUE;
        }
        else {
            throw new InvalidExtendedDataElementTypeException("setType was called with an invalid datatype value of  " + getType());
        }
    }

    /**
     * Sets the value.
     * @param value The value to set
     */
    public void setValues(String[] values) {
        this.values = values;
    }

    public void setHexValues(char[] hexValues) {

        values = hexValues;

        type = TYPE_HEX_BINARY_VALUE;

        valuesType = TYPE_HEX_BINARY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setHexValue(byte[])
     */
    public void setHexValue(byte[] hexValue) {

        char[] hexValues = null;

        if (hexValue != null) {

            hexValues = new char[hexValue.length];

            for (int counter = 0; counter < hexValue.length; counter++) {

                //Adjust the signed 8-bit byte (e.g. -128 to 127) to a unsigned 8-bit character (e.g. 0 to 255):
                hexValues[counter] = ((char) ((hexValue[counter] + 128)));
            }
        }

        setHexValues(hexValues);
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getString()
     */
    public String getString() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_STRING_VALUE :
                    {
                        return ((String) (values));
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return (((String[]) (values))[0]);
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getString was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getBoolean()
     */
    public boolean getBoolean() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_BOOLEAN_VALUE :
                    {
                        return (((Boolean) (values)).booleanValue());
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return (new Boolean(((String[]) (values))[0]).booleanValue());
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getBoolean was called for an illegal event property type");
                    }
            }
        }

        return true;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getByte()
     */
    public byte getByte() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_BYTE_VALUE :
                    {
                        return (((Byte) (values)).byteValue());
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return (new Byte(((String[]) (values))[0]).byteValue());
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getByte was called for an illegal event property type");
                    }
            }
        }

        return 0;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getShort()
     */
    public short getShort() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_SHORT_VALUE :
                    {
                        return (((Short) (values)).shortValue());
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return (new Short(((String[]) (values))[0]).shortValue());
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getShort was called for an illegal event property type");
                    }
            }
        }

        return 0;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getInt()
     */
    public int getInt() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_INT_VALUE :
                    {
                        return (((Integer) (values)).intValue());
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return (new Integer(((String[]) (values))[0]).intValue());
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getInt was called for an illegal event property type");
                    }
            }
        }

        return 0;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getLong()    
     */
    public long getLong() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_LONG_VALUE :
                    {
                        return (((Long) (values)).longValue());
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return (new Long(((String[]) (values))[0]).longValue());
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getLong was called for an illegal event property type");
                    }
            }
        }

        return 0;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getFloat()
     */
    public float getFloat() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_FLOAT_VALUE :
                    {
                        return (((Float) (values)).floatValue());
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return (new Float(((String[]) (values))[0]).floatValue());
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getFloat was called for an illegal event property type");
                    }
            }
        }

        return 0;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDouble()
     */
    public double getDouble() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_DOUBLE_VALUE :
                    {
                        return (((Double) (values)).doubleValue());
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return (new Double(((String[]) (values))[0]).doubleValue());
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getDouble was called for an illegal event property type");
                    }
            }
        }

        return 0;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getStringArray()
     */
    public String[] getStringArray() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return ((String[]) (values));
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getStringArray was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDateTimeAsLong()
     */
    public long getDateTimeAsLong() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_STRING_VALUE :
                    {
                        return (CbeFormatter.convertXmlSchemaDateTimeToDate(((String) (values))));
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return (CbeFormatter.convertXmlSchemaDateTimeToDate((((String[]) (values))[0])));
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getDateTimeAsLong was called for an illegal event property type");
                    }
            }
        }

        return 0;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDateTime()
     */
    public String getDateTime() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_STRING_VALUE :
                    {
                        return ((String) (values));
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return (((String[]) (values))[0]);
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getDateTime was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDateTimeArrayAsLong()
     */
    public long[] getDateTimeArrayAsLong() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_STRING_ARRAY_VALUE :
                    {

                        String[] dateTimeArray = ((String[]) (values));
                        long[] dateTimeAsLongArray = new long[dateTimeArray.length];

                        for (int counter = 0; counter < dateTimeArray.length; counter++) {
                            dateTimeAsLongArray[counter] = (CbeFormatter.convertXmlSchemaDateTimeToDate(dateTimeArray[counter]));
                        }

                        return dateTimeAsLongArray;
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getDateTimeArrayAsLong was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDateTimeArray()
     */
    public String[] getDateTimeArray() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        return ((String[]) (values));
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getDateTimeArray was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getBooleanArray()
     */
    public boolean[] getBooleanArray() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_BOOLEAN_ARRAY_VALUE :
                    {
                        return ((boolean[]) (values));
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {

                        String[] stringArray = ((String[]) (values));
                        boolean[] returnArray = new boolean[stringArray.length];

                        for (int counter = 0; counter < stringArray.length; counter++) {
                            returnArray[counter] = new Boolean(stringArray[counter]).booleanValue();
                        }

                        return returnArray;
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getBooleanArray was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getByteArray()
     */
    public byte[] getByteArray() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_BYTE_ARRAY_VALUE :
                    {
                        return ((byte[]) (values));
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {

                        String[] stringArray = ((String[]) (values));
                        byte[] returnArray = new byte[stringArray.length];

                        for (int counter = 0; counter < stringArray.length; counter++) {
                            returnArray[counter] = new Byte(stringArray[counter]).byteValue();
                        }

                        return returnArray;
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getByteArray was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getShortArray()
     */
    public short[] getShortArray() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_SHORT_ARRAY_VALUE :
                    {
                        return ((short[]) (values));
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {

                        String[] stringArray = ((String[]) (values));
                        short[] returnArray = new short[stringArray.length];

                        for (int counter = 0; counter < stringArray.length; counter++) {
                            returnArray[counter] = new Short(stringArray[counter]).shortValue();
                        }

                        return returnArray;
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getShortArray was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getIntArray()
     */
    public int[] getIntArray() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_INT_ARRAY_VALUE :
                    {
                        return ((int[]) (values));
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {

                        String[] stringArray = ((String[]) (values));
                        int[] returnArray = new int[stringArray.length];

                        for (int counter = 0; counter < stringArray.length; counter++) {
                            returnArray[counter] = new Integer(stringArray[counter]).intValue();
                        }

                        return returnArray;
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getIntArray was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getLongArray()
     */
    public long[] getLongArray() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_LONG_ARRAY_VALUE :
                    {
                        return ((long[]) (values));
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {

                        String[] stringArray = ((String[]) (values));
                        long[] returnArray = new long[stringArray.length];

                        for (int counter = 0; counter < stringArray.length; counter++) {
                            returnArray[counter] = new Long(stringArray[counter]).longValue();
                        }

                        return returnArray;
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getLongArray was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getFloatArray()
     */
    public float[] getFloatArray() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_FLOAT_ARRAY_VALUE :
                    {
                        return ((float[]) (values));
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {

                        String[] stringArray = ((String[]) (values));
                        float[] returnArray = new float[stringArray.length];

                        for (int counter = 0; counter < stringArray.length; counter++) {
                            returnArray[counter] = new Float(stringArray[counter]).floatValue();
                        }

                        return returnArray;
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getFloatArray was called for an illegal event property type");
                    }
            }
        }

        return null;

    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getDoubleArray()
     */
    public double[] getDoubleArray() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_DOUBLE_ARRAY_VALUE :
                    {
                        return ((double[]) (values));
                    }

                case TYPE_STRING_ARRAY_VALUE :
                    {

                        String[] stringArray = ((String[]) (values));
                        double[] returnArray = new double[stringArray.length];

                        for (int counter = 0; counter < stringArray.length; counter++) {
                            returnArray[counter] = new Double(stringArray[counter]).doubleValue();
                        }

                        return returnArray;
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getDoubleArray was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getHexValues()
     */
    public char[] getHexValues() {
        return getHexBinarys();
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getHexBinarys()
     */
    public char[] getHexBinarys() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_HEX_BINARY_VALUE :
                    {
                        return ((char[]) (values));
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getHexBinarys was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getHexValue()
     */
    public byte[] getHexValue() {
        return getHexBinary();
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#getHexBinary()
     */
    public byte[] getHexBinary() {

        if (values != null) {

            switch (valuesType) {

                case TYPE_HEX_BINARY_VALUE :
                    {
                        char[] hexValues = ((char[]) (values));
                        byte[] hexValuesAsBytes = new byte[hexValues.length];

                        for (int counter = 0; counter < hexValues.length; counter++) {

                            //Check if higher order characters (e.g. 256 - 65535), if so, set to the largest signed 8-bit byte (e.g. 127):
                            if (hexValues[counter] > 255)
                                hexValuesAsBytes[counter] = 127;

                            //Adjust the unsigned 8-bit character (e.g. 0 to 255) to a signed 8-bit byte (e.g. -128 to 127):
                            else
                                hexValuesAsBytes[counter] = ((byte) ((hexValues[counter] - 128)));
                        }

                        return hexValuesAsBytes;
                    }

                default :
                    {
                        throw new InvalidExtendedDataElementTypeException("getHexBinarys was called for an illegal event property type");
                    }
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setString(String)
     */
    public void setString(String value) {

        this.values = value;

        type = TYPE_STRING_VALUE;

        valuesType = TYPE_STRING_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDateTime(long)
     */
    public void setDateTime(long value) {

        this.values = CbeFormatter.convertDateToXmlSchemaDateTime(value);

        type = TYPE_DATE_TIME_VALUE;

        valuesType = TYPE_STRING_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDateTime(String)
     */
    public void setDateTime(String value) {

        this.values = value;

        type = TYPE_DATE_TIME_VALUE;

        valuesType = TYPE_STRING_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setBoolean(boolean)
     */
    public void setBoolean(boolean value) {

        this.values = new Boolean(value);

        type = TYPE_BOOLEAN_VALUE;

        valuesType = TYPE_BOOLEAN_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setByte(byte)
     */
    public void setByte(byte value) {

        this.values = new Byte(value);

        type = TYPE_BYTE_VALUE;

        valuesType = TYPE_BYTE_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setShort(short)
     */
    public void setShort(short value) {

        this.values = new Short(value);

        type = TYPE_SHORT_VALUE;

        valuesType = TYPE_SHORT_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setInt(int)
     */
    public void setInt(int value) {

        this.values = new Integer(value);

        type = TYPE_INT_VALUE;

        valuesType = TYPE_INT_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setLong(long)
     */
    public void setLong(long value) {

        this.values = new Long(value);

        type = TYPE_LONG_VALUE;

        valuesType = TYPE_LONG_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setFloat(float)
     */
    public void setFloat(float value) {

        this.values = new Float(value);

        type = TYPE_FLOAT_VALUE;

        valuesType = TYPE_FLOAT_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDouble(double)
     */
    public void setDouble(double value) {

        this.values = new Double(value);

        type = TYPE_DOUBLE_VALUE;

        valuesType = TYPE_DOUBLE_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setStringArray(String[])
     */
    public void setStringArray(String[] values) {

        this.values = values;

        type = TYPE_STRING_ARRAY_VALUE;

        valuesType = TYPE_STRING_ARRAY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDateTimeArray(long[])
     */
    public void setDateTimeArray(long[] values) {

        if (values != null) {

            String[] dateTimeArray = new String[values.length];

            for (int counter = 0; counter < values.length; counter++) {
                dateTimeArray[counter] = (CbeFormatter.convertDateToXmlSchemaDateTime(values[counter]));
            }

            this.values = dateTimeArray;

        }
        else {
            values = null;
        }

        type = TYPE_DATE_TIME_ARRAY_VALUE;

        valuesType = TYPE_STRING_ARRAY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDateTimeArray(String[])
     */
    public void setDateTimeArray(String[] values) {

        this.values = values;

        type = TYPE_DATE_TIME_ARRAY_VALUE;

        valuesType = TYPE_STRING_ARRAY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setBooleanArray(boolean[])
     */
    public void setBooleanArray(boolean[] values) {

        this.values = values;

        type = TYPE_BOOLEAN_ARRAY_VALUE;

        valuesType = TYPE_BOOLEAN_ARRAY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setByteArray(byte[])
     */
    public void setByteArray(byte[] values) {

        this.values = values;

        type = TYPE_BYTE_ARRAY_VALUE;

        valuesType = TYPE_BYTE_ARRAY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setShortArray(short[])
     */
    public void setShortArray(short[] values) {

        this.values = values;

        type = TYPE_SHORT_ARRAY_VALUE;

        valuesType = TYPE_SHORT_ARRAY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setIntArray(int[])
     */
    public void setIntArray(int[] values) {

        this.values = values;

        type = TYPE_INT_ARRAY_VALUE;

        valuesType = TYPE_INT_ARRAY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setLongArray(long[])
     */
    public void setLongArray(long[] values) {

        this.values = values;

        type = TYPE_LONG_ARRAY_VALUE;

        valuesType = TYPE_LONG_ARRAY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setFloatArray(float[])
     */
    public void setFloatArray(float[] values) {

        this.values = values;

        type = TYPE_FLOAT_ARRAY_VALUE;

        valuesType = TYPE_FLOAT_ARRAY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setDoubleArray(double[])
     */
    public void setDoubleArray(double[] values) {

        this.values = values;

        type = TYPE_DOUBLE_ARRAY_VALUE;

        valuesType = TYPE_DOUBLE_ARRAY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setHexBinarys(char[])
     */
    public void setHexBinarys(char[] values) {

        this.values = values;

        type = TYPE_HEX_BINARY_VALUE;

        valuesType = TYPE_HEX_BINARY_VALUE;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#setHexBinary(byte[])
     */
    public void setHexBinary(byte[] values) {

        if (values != null) {

            char[] hexValues = new char[values.length];

            for (int counter = 0; counter < values.length; counter++) {
                hexValues[counter] = ((char) ((values[counter] + 128)));
            }

            this.values = hexValues;
        }
        else {
            values = null;
        }

        type = TYPE_HEX_BINARY_VALUE;

        valuesType = TYPE_HEX_BINARY_VALUE;
    }

    /**
     * @deprecated Use the init() as of 1.0.1.
     */
    public void clearValues() {
        init();
    }

    /**
     * Resets the object's properties to their initial (e.g. null) state.
     * 
     * @since 1.0.1
     */
    public void init() {

        type = -1;

        name = null;
        id = null;
        values = null;
        valuesType = -1;

        children = null;
    }

    /**
     * Determines is two different objects are equal.
     */
    public boolean equals(Object obj) {

        //It's the same object reference, so it HAS to be equal
        if (this == obj) {
            return true;
        }
        else if (obj instanceof ExtendedDataElementImpl) {

            ExtendedDataElementImpl extendedDataElement = ((ExtendedDataElementImpl) (obj));

            //Does the object represent the same data type?
            if (((this.name != null) && (this.name.equals(extendedDataElement.name)) || (this.name == null) && (extendedDataElement.name == null)) && (this.type == extendedDataElement.type)) {

                if (((this.children != null) && (this.children.equals(extendedDataElement.children))) || (((this.children == null) || (this.children.size() == 0)) && ((extendedDataElement.children == null) || (extendedDataElement.children.size() == 0)))) {

                    if ((this.values != null) && (extendedDataElement.values != null)) {

                        switch (valuesType) {

                            case TYPE_STRING_ARRAY_VALUE :
                                {

                                    if (extendedDataElement.valuesType == TYPE_STRING_ARRAY_VALUE) {

                                        String[] thisValuesProperty = ((String[]) (this.values));
                                        String[] thatValuesProperty = ((String[]) (extendedDataElement.values));

                                        if (thisValuesProperty.length == thatValuesProperty.length) {

                                            for (int counter = 0; counter < thisValuesProperty.length; counter++) {

                                                if (!thisValuesProperty[counter].equals(thatValuesProperty[counter])) {
                                                    return false;
                                                }
                                            }

                                            return true;
                                        }
                                    }

                                    break;
                                }

                            case TYPE_BYTE_ARRAY_VALUE :
                                {

                                    if (extendedDataElement.valuesType == TYPE_BYTE_ARRAY_VALUE) {

                                        byte[] thisValuesProperty = ((byte[]) (this.values));
                                        byte[] thatValuesProperty = ((byte[]) (extendedDataElement.values));

                                        if (thisValuesProperty.length == thatValuesProperty.length) {

                                            for (int counter = 0; counter < thisValuesProperty.length; counter++) {

                                                if (thisValuesProperty[counter] != thatValuesProperty[counter]) {
                                                    return false;
                                                }
                                            }

                                            return true;
                                        }
                                    }

                                    break;
                                }

                            case TYPE_SHORT_ARRAY_VALUE :
                                {

                                    if (extendedDataElement.valuesType == TYPE_SHORT_ARRAY_VALUE) {

                                        short[] thisValuesProperty = ((short[]) (this.values));
                                        short[] thatValuesProperty = ((short[]) (extendedDataElement.values));

                                        if (thisValuesProperty.length == thatValuesProperty.length) {

                                            for (int counter = 0; counter < thisValuesProperty.length; counter++) {

                                                if (thisValuesProperty[counter] != thatValuesProperty[counter]) {
                                                    return false;
                                                }
                                            }

                                            return true;
                                        }
                                    }

                                    break;
                                }

                            case TYPE_INT_ARRAY_VALUE :
                                {

                                    if (extendedDataElement.valuesType == TYPE_INT_ARRAY_VALUE) {

                                        int[] thisValuesProperty = ((int[]) (this.values));
                                        int[] thatValuesProperty = ((int[]) (extendedDataElement.values));

                                        if (thisValuesProperty.length == thatValuesProperty.length) {

                                            for (int counter = 0; counter < thisValuesProperty.length; counter++) {

                                                if (thisValuesProperty[counter] != thatValuesProperty[counter]) {
                                                    return false;
                                                }
                                            }

                                            return true;
                                        }
                                    }

                                    break;
                                }

                            case TYPE_LONG_ARRAY_VALUE :
                                {

                                    if (extendedDataElement.valuesType == TYPE_LONG_ARRAY_VALUE) {

                                        long[] thisValuesProperty = ((long[]) (this.values));
                                        long[] thatValuesProperty = ((long[]) (extendedDataElement.values));

                                        if (thisValuesProperty.length == thatValuesProperty.length) {

                                            for (int counter = 0; counter < thisValuesProperty.length; counter++) {

                                                if (thisValuesProperty[counter] != thatValuesProperty[counter]) {
                                                    return false;
                                                }
                                            }

                                            return true;
                                        }
                                    }

                                    break;
                                }

                            case TYPE_FLOAT_ARRAY_VALUE :
                                {

                                    if (extendedDataElement.valuesType == TYPE_FLOAT_ARRAY_VALUE) {

                                        float[] thisValuesProperty = ((float[]) (this.values));
                                        float[] thatValuesProperty = ((float[]) (extendedDataElement.values));

                                        if (thisValuesProperty.length == thatValuesProperty.length) {

                                            for (int counter = 0; counter < thisValuesProperty.length; counter++) {

                                                if (thisValuesProperty[counter] != thatValuesProperty[counter]) {
                                                    return false;
                                                }
                                            }

                                            return true;
                                        }
                                    }

                                    break;
                                }

                            case TYPE_DOUBLE_ARRAY_VALUE :
                                {

                                    if (extendedDataElement.valuesType == TYPE_DOUBLE_ARRAY_VALUE) {

                                        double[] thisValuesProperty = ((double[]) (this.values));
                                        double[] thatValuesProperty = ((double[]) (extendedDataElement.values));

                                        if (thisValuesProperty.length == thatValuesProperty.length) {

                                            for (int counter = 0; counter < thisValuesProperty.length; counter++) {

                                                if (thisValuesProperty[counter] != thatValuesProperty[counter]) {
                                                    return false;
                                                }
                                            }

                                            return true;
                                        }
                                    }

                                    break;
                                }

                            case TYPE_BOOLEAN_ARRAY_VALUE :
                                {

                                    if (extendedDataElement.valuesType == TYPE_BOOLEAN_ARRAY_VALUE) {

                                        boolean[] thisValuesProperty = ((boolean[]) (this.values));
                                        boolean[] thatValuesProperty = ((boolean[]) (extendedDataElement.values));

                                        if (thisValuesProperty.length == thatValuesProperty.length) {

                                            for (int counter = 0; counter < thisValuesProperty.length; counter++) {

                                                if (thisValuesProperty[counter] != thatValuesProperty[counter]) {
                                                    return false;
                                                }
                                            }

                                            return true;
                                        }
                                    }

                                    break;
                                }

                            case TYPE_HEX_BINARY_VALUE :
                                {

                                    if (extendedDataElement.valuesType == TYPE_HEX_BINARY_VALUE) {

                                        char[] thisValuesProperty = ((char[]) (this.values));
                                        char[] thatValuesProperty = ((char[]) (extendedDataElement.values));

                                        if (thisValuesProperty.length == thatValuesProperty.length) {

                                            for (int counter = 0; counter < thisValuesProperty.length; counter++) {

                                                if (thisValuesProperty[counter] != thatValuesProperty[counter]) {
                                                    return false;
                                                }
                                            }

                                            return true;
                                        }
                                    }

                                    break;
                                }

                            default :
                                {
                                    return (this.values.equals(extendedDataElement.values));
                                }
                        }
                    }
                    else if ((this.values == null) && (extendedDataElement.values == null)) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    private void writeObject(ObjectOutputStream out) throws IOException {

        // set the values of the Serializable fields
        ObjectOutputStream.PutField fields = out.putFields();
        fields.put("name", name);
        fields.put("type", type);
        fields.put("values", values);
        fields.put("valuesType", valuesType);

        if ((children != null) && (!children.isEmpty())) {
            fields.put("children", ((IExtendedDataElement[]) (children.toArray(new IExtendedDataElement[children.size()]))));
        }

        // save them
        out.writeFields();
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {

        // prepare to read the alternate persistent fields
        ObjectInputStream.GetField fields = null;
        fields = in.readFields();

        // read the alternate persistent fields
        name = (String) fields.get("name", name);
        type = (int) fields.get("type", type);
        values = (Object) fields.get("values", values);
        valuesType = (int) fields.get("valuesType", valuesType);

        Object tempArray[] = null;
        tempArray = (Object[]) fields.get("children", null);
        if (tempArray != null) {
            int arraySize = tempArray.length;
            for (int i = 0; i < arraySize; i++) {
                addChildDataElement((IExtendedDataElement) tempArray[i]);
            }
        }
    }

    /**
     * return a cloned (copy by value) object. This is a deep copy version, in which all the
     * objects within this object will be copied by value.
     *  
     * @return Object
     */
    public Object clone() throws CloneNotSupportedException {

        ExtendedDataElementImpl copy = ((ExtendedDataElementImpl) (super.clone()));

        Object tempElement = null;

        //Clone the children array list:
        if (this.children != null) {
            copy.children = ((ArrayList) (this.children.clone()));
            for (int ctr = 0; ctr < this.children.size(); ctr++) {
                tempElement = this.children.get(ctr);
                /* if the Elements in children arrayList are some other implementations,
                 * it will only be shallow cloned.
                 */
                if (tempElement instanceof ExtendedDataElementImpl) {
                    copy.children.set(ctr, ((IExtendedDataElement) ((ExtendedDataElementImpl) (tempElement)).clone()));
                }
            }
        }

        if (this.values != null) {

            switch (this.valuesType) {

                case TYPE_STRING_ARRAY_VALUE :
                    {
                        copy.values = ((String[]) (this.values)).clone();
                        break;
                    }

                case TYPE_BYTE_ARRAY_VALUE :
                    {
                        copy.values = ((byte[]) (this.values)).clone();
                        break;
                    }

                case TYPE_SHORT_ARRAY_VALUE :
                    {
                        copy.values = ((short[]) (this.values)).clone();
                        break;
                    }

                case TYPE_INT_ARRAY_VALUE :
                    {
                        copy.values = ((int[]) (this.values)).clone();
                        break;
                    }

                case TYPE_LONG_ARRAY_VALUE :
                    {
                        copy.values = ((long[]) (this.values)).clone();
                        break;
                    }

                case TYPE_FLOAT_ARRAY_VALUE :
                    {
                        copy.values = ((float[]) (this.values)).clone();
                        break;
                    }

                case TYPE_DOUBLE_ARRAY_VALUE :
                    {
                        copy.values = ((double[]) (this.values)).clone();
                        break;
                    }

                case TYPE_BOOLEAN_ARRAY_VALUE :
                    {
                        copy.values = ((boolean[]) (this.values)).clone();
                        break;
                    }

                case TYPE_HEX_BINARY_VALUE :
                    {
                        copy.values = ((char[]) (this.values)).clone();
                        break;
                    }
                case TYPE_STRING_VALUE :
                    {
                        copy.values = new String(((String) (this.values)));
                        break;
                    }

                case TYPE_BYTE_VALUE :
                    {
                        copy.values = new Byte(((Byte) (this.values)).byteValue());
                        break;
                    }

                case TYPE_SHORT_VALUE :
                    {
                        copy.values = new Short(((Short) (this.values)).shortValue());
                        break;
                    }

                case TYPE_INT_VALUE :
                    {
                        copy.values = new Integer(((Integer) (this.values)).intValue());
                        break;
                    }

                case TYPE_LONG_VALUE :
                    {
                        copy.values = new Long(((Long) (this.values)).longValue());
                        break;
                    }

                case TYPE_FLOAT_VALUE :
                    {
                        copy.values = new Float(((Float) (this.values)).floatValue());
                        break;
                    }

                case TYPE_DOUBLE_VALUE :
                    {
                        copy.values = new Double(((Double) (this.values)).doubleValue());
                        break;
                    }

                case TYPE_BOOLEAN_VALUE :
                    {
                        copy.values = new Boolean(((Boolean) (this.values)).booleanValue());
                        break;
                    }
            }
        }

        return copy;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChild(java.lang.String, int, java.lang.String)
     */
    public IExtendedDataElement addChild(String name, int type, String value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(type);
        extendedDataElement.setString(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChild(java.lang.String, int, java.lang.String[])
     */
    public IExtendedDataElement addChild(String name, int type, String[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setType(type);
        extendedDataElement.setStringArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChild(java.lang.String, java.lang.String)
     */
    public IExtendedDataElement addChild(String name, String value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setString(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChild(java.lang.String, java.lang.String[])
     */
    public IExtendedDataElement addChild(String name, String[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setStringArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChild(java.lang.String)
     */
    public IExtendedDataElement addChild(String name) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithBooleanArrayValue(java.lang.String, boolean[])
     */
    public IExtendedDataElement addChildWithBooleanArrayValue(String name, boolean[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setBooleanArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithBooleanValue(java.lang.String, boolean)
     */
    public IExtendedDataElement addChildWithBooleanValue(String name, boolean value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setBoolean(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithByteArrayValue(java.lang.String, byte[])
     */
    public IExtendedDataElement addChildWithByteArrayValue(String name, byte[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setByteArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithByteValue(java.lang.String, byte)
     */
    public IExtendedDataElement addChildWithByteValue(String name, byte value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setByte(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDateArrayValue(java.lang.String, java.lang.String[])
     */
    public IExtendedDataElement addChildWithDateArrayValue(String name, String[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setDateTimeArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDateAsLongValue(java.lang.String, long)
     */
    public IExtendedDataElement addChildWithDateAsLongValue(String name, long value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setDateTime(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDatesAsLongValue(java.lang.String, long[])
     */
    public IExtendedDataElement addChildWithDatesAsLongValue(String name, long[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setDateTimeArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDateValue(java.lang.String, java.lang.String)
     */
    public IExtendedDataElement addChildWithDateValue(String name, String value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setDateTime(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDoubleArrayValue(java.lang.String, double[])
     */
    public IExtendedDataElement addChildWithDoubleArrayValue(String name, double[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setDoubleArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithDoubleValue(java.lang.String, double)
     */
    public IExtendedDataElement addChildWithDoubleValue(String name, double value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setDouble(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithFloatArrayValue(java.lang.String, float[])
     */
    public IExtendedDataElement addChildWithFloatArrayValue(String name, float[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setFloatArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithFloatValue(java.lang.String, float)
     */
    public IExtendedDataElement addChildWithFloatValue(String name, float value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setFloat(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithHexValue(java.lang.String, java.lang.String)
     */
    public IExtendedDataElement addChildWithHexValue(String name, String value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);

        char[] hexValues = new char[value.length()];

        for (int counter = 0; counter < hexValues.length; counter++) {
            hexValues[counter] = value.charAt(counter);
        }

        extendedDataElement.setHexValues(hexValues);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithHexValueAsByteArray(java.lang.String, byte[])
     */
    public IExtendedDataElement addChildWithHexValueAsByteArray(String name, byte[] value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setHexValue(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithIntArrayValue(java.lang.String, int[])
     */
    public IExtendedDataElement addChildWithIntArrayValue(String name, int[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setIntArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithIntValue(java.lang.String, int)
     */
    public IExtendedDataElement addChildWithIntValue(String name, int value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setInt(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithLongArrayValue(java.lang.String, long[])
     */
    public IExtendedDataElement addChildWithLongArrayValue(String name, long[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setLongArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithLongValue(java.lang.String, long)
     */
    public IExtendedDataElement addChildWithLongValue(String name, long value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setLong(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithShortArrayValue(java.lang.String, short[])
     */
    public IExtendedDataElement addChildWithShortArrayValue(String name, short[] values) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setShortArray(values);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.hyades.logging.events.IExtendedDataElement#addChildWithShortValue(java.lang.String, short)
     */
    public IExtendedDataElement addChildWithShortValue(String name, short value) {

        IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
        extendedDataElement.setName(name);
        extendedDataElement.setShort(value);

        addChildDataElement(extendedDataElement);

        return extendedDataElement;
    }
}
