package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;

import org.w3c.dom.CDATASection;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
/**
 * Abstract base class <code>SVGDocumentGenerator</code> implements document-building methods 
 * which are common to SVG-based charts. 
 * 
 * @version 1.79.1.31
 */
public abstract class SVGDocumentGenerator extends GraphicDocumentGenerator implements IGraphicDocumentStyle, IGraphicDocumentDefaults, Serializable {
	/**
	 * The SVG version is "-//W3C//DTD SVG 1.0//EN".
	 */	
	static final String SVG_VERSION = "-//W3C//DTD SVG 1.0//EN";
	
	/**
	 * The SVG DTD is "http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd".
	 */	
	static final String SVG_DTD = "http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd";

	/**
	 * The SVG XMLNS is "http://www.w3.org/1999/xlink".
	 */	
	static final String SVG_XMLNS = "http://www.w3.org/1999/xlink";
	
	
	/**
	 * The JavaScript functions conform to ECMAScript.
	 */	
	static final String JAVASCRIPT_VERSION = "text/ecmascript";
		
	/**
	 * Font size delta (from basic) for chart titles
	 */
	protected final int TITLE_FONTSIZE_DELTA=			3;
	
	/**
	 * Font size delta (from basic)for chart labels
	 */
	protected final int LABEL_FONTSIZE_DELTA=			2;
	
	
	/**
	 * Sole constructor
	 */ 
	public SVGDocumentGenerator() {
		super();			
	}
		
		
	/**
	 * Adds definitions for user preferences icon and legend item graphics.
	 * 
	 * @param generatedDocument the SVG document
	 * @param attrs object containing the graphic properties
	 */
	protected Element addDefinitions(Document generatedDocument, GraphicAttributes attrs) {
		Element parent = generatedDocument.getDocumentElement();		
		Element defsElement = generatedDocument.createElement("defs");
		
        parent.appendChild(defsElement);       
        
        // user preference icon
        if (!attrs.isUserPreferencesSuppressed()) {
        	definePreferenceIcon(generatedDocument, defsElement);
        }
        
        // default legend item graphic
        if (!attrs.isLegendSuppressed()) {
        	defineLegendGraphic(generatedDocument, defsElement);
        }		
        
        return defsElement;	
	}    
	
	
	/**
	 * Adds the outline for this object's graphic.
	 * 
	 * @param generatedDocument the SVG document
	 * @param attrs object containing the graphic properties	
	 */
	protected void addOutline(Document generatedDocument, GraphicAttributes attrs) {			
		Element parent = generatedDocument.getDocumentElement();
		short offset = 1;		
		Element newElement = generatedDocument.createElement("rect");		
		newElement.setAttribute("class", "st1, outline");		
		newElement.setAttribute("width", Integer.toString(attrs.getGraphicWidth()-1));
		newElement.setAttribute("height", Integer.toString(attrs.getGraphicHeight()-1));
		parent.appendChild(newElement);		
	}
	
		
	/**
	 * Adds the title bar area with title and preferences icon.
	 *  	
	 * @param generatedDocument the SVG document
	 * @param attrs object containing the graphic properties	
	 */
	protected void addTitleBar(Document generatedDocument, GraphicAttributes attrs) {		
		Element parent = generatedDocument.getDocumentElement();	
		boolean titleBarLineAdded = false;		
		
		// arbitrary value ???
		final int yValue = 28;	
		
	    // add main title
	    if (attrs.getGraphicTitle() != null) {
	    	addTitle(generatedDocument, attrs, parent);	    	
	    	// add title bar separator if not already added
	    	if (!titleBarLineAdded) {
	    		addTitleBarLine(generatedDocument, attrs, parent, yValue);
	    		titleBarLineAdded = true;
	    		
	    	}	
	    }
	    
	    // add preferences icon	
/*	    if (!attrs.isUserPreferencesSuppressed() && attrs.getSvgFormatOnly()) {
	    	addPreferenceIcon(generatedDocument, attrs, parent);  	 
	 		// add title bar separator if not already added
	    	if (!titleBarLineAdded) {
	    		addTitleBarLine(generatedDocument, attrs, parent, yValue);
	    		titleBarLineAdded = true;
	    	}	
	    }*/
	}

	/**
	 * Adds the preferences icon.
	 *  	
	 * @param generatedDocument the SVG document
	 * @param attrs object containing the graphic properties	
	 */
	protected void addPreferenceIcon(Document generatedDocument, GraphicAttributes attrs) {
		// add preferences icon	
		Element parent = generatedDocument.getDocumentElement();
		if (!attrs.isUserPreferencesSuppressed() && attrs.getSvgFormatOnly()) {
			addPreferenceIcon(generatedDocument, attrs, parent);  	 
		}
	}
	
		
	/**
	 * Adds the timestamp to the document element after prepending the timestamp
	 * prefix (if exists) to the timestamp.
	 * 	 
  	 * @param generatedDocument the SVG document
	 * @param attrs object containing the graphic properties	 
	 * @param timestamp timestamp (from input data)
	 */	
	protected void addTimeStamp(Document generatedDocument, GraphicAttributes attrs, String timestamp) {	
		if (timestamp ==  null) {
			return;
		}
		
		Element parent = generatedDocument.getDocumentElement();	
		Element gElement = generatedDocument.createElement("g");						
		String prefix = attrs.getGraphicTimestampPrefix();		
		
		if (prefix != null) {
			timestamp = prefix + " " + timestamp;
		}		
		
		gElement.setAttribute("id", "timeStamp");
		gElement.setAttribute("class", "timeStamp");
		addLabel(generatedDocument, attrs, gElement, timestamp, null, (attrs.getGraphicWidth() - 5), (attrs.getGraphicHeight() - DEFAULT_FONTSIZE), 0);		
		parent.appendChild(gElement);
	}
	
	
	/**
	 * Adds the legend to this object's graphic.
	 * 
	 * @param parent	element to which the legend-related elements are appended 
	 * @param labels	legend text labels
	 * @param xOffset	starting x position
	 * @param yOffset	starting y position
	 */	
	protected void addLegend(Document generatedDocument, GraphicAttributes attrs, Element gTextBoxElement, String [] labels, String [] flyOverLabels,int xOffset, int yOffset) {		
		Element parent = generatedDocument.getDocumentElement();
		int numberOfLabels = labels.length;	
		// why are we adding to these sizes???
		int legendTitleFontSize = DEFAULT_FONTSIZE + 2;
		int legendLabelFontSize = DEFAULT_FONTSIZE + 1;
		
		// from definitions legendsquareshapeWithoutLine width - why are we adding 3???
		int shapeWidth = 11;
		
		// eyeball positioning ???
		int legendMargin = 15;
		int legendLabelMargin = shapeWidth + 6;
		int delta = 16;
		
		int rotation = 0;
		int x = xOffset + legendMargin;
		int y = yOffset + legendTitleFontSize;
		
		// set up for label positions
		int legendLabelStartY = yOffset - legendLabelFontSize + 8;
		
		String title = attrs.getGraphicLegendTitle();
		if (title != null) {
			// add font group for legend title			
			Element gElement = generatedDocument.createElement("g");
			gElement.setAttribute("id", "legendTitle");
			gElement.setAttribute("class", "legendTitle");
			parent.appendChild(gElement);				
			// add title to font group
			addLabel(generatedDocument, attrs, gElement, title, null, x, y, rotation);
			// adjust for label positions
			legendLabelStartY = yOffset + legendTitleFontSize + 7;
		}
		// font group for legend labels		
		Element gElement = generatedDocument.createElement("g");
		gElement.setAttribute("id", "legendData");
		gElement.setAttribute("class", "legendData");
		parent.appendChild(gElement);	
		
		// add color squares and legend labels
		for (int i = 0, yPrevious = legendLabelStartY ; i < numberOfLabels; i++) {
 			Element gColourElement = addLegendItemGraphic(generatedDocument, parent, i, legendLabelFontSize, x, yPrevious);
			if (graphicType.equals(IGraphicTypeConstants.LINE_CHART) || graphicType.equals(IGraphicTypeConstants.AREA_CHART) || graphicType.equals(IGraphicTypeConstants.SCATTER_CHART)) {
				parent.appendChild(gColourElement);	
			}
			// bump label over by margin of 15
			// labels are placed using lower left corner
			yPrevious += delta;
			// font style set on group				
			addHoverText(generatedDocument, attrs, i, gTextBoxElement, labels[i], flyOverLabels[i], x + legendLabelMargin, yPrevious - 7, rotation);
		}
	}
	

	/**
	 * Adds the legend shapes to this object's graphic.
	 * 
	 * @param parent	element to which the legend-related elements are appended 
	 * @param index	dataset number
	 * @param fontSize	y shift factor
	 * @param x		upper left x
	 * @param y 		upper left y 
	 */	
	protected Element addLegendItemGraphic(Document generatedDocument, Element parent, int index, int fontSize, int x, int y) {
		Element newElement, gColourElement;
		newElement = generatedDocument.createElement("use");
		newElement.setAttribute("xlink:href", "#legendsquareshapeWithoutLine");
		newElement.setAttribute("x", Integer.toString(x));
		// rectangles are placed using upper left corner
		newElement.setAttribute("y", Integer.toString(y + fontSize - 9));    
			
		// add the element to its color group for user preference changes
		gColourElement = generatedDocument.createElement("g");
		gColourElement.setAttribute("id", "legendsquareshapeWithoutLine" + index);
		gColourElement.setAttribute("class", "shape" + index);		
		if (graphicType.equals(IGraphicTypeConstants.LINE_CHART) || graphicType.equals(IGraphicTypeConstants.AREA_CHART) || graphicType.equals(IGraphicTypeConstants.SCATTER_CHART)) {
			gColourElement.setAttribute("onclick","toggleVisibility('dataColour" + index + "');");	
			if (graphicType.equals(IGraphicTypeConstants.SCATTER_CHART)){
				final String[] flyOverShapeType = {"flyoverCircleshape", "flyoverSquareshape", "flyoverHorizontalrectshape", "flyoverVerticalrectshape", "flyoverTriangleshape", "flyoverDiamondshape"};							
				gColourElement.setAttribute("onmouseover", "setVisibility('" + flyOverShapeType[index % (flyOverShapeType.length)] + "' , 'visible');");
				gColourElement.setAttribute("onmouseout", "setVisibility('" + flyOverShapeType[index % (flyOverShapeType.length)] + "' , 'hidden');");			
			}
		}
		gColourElement.appendChild(newElement);
		parent.appendChild(gColourElement);	
		return gColourElement;
	}
	
	
	/** 
	 * Adds the common JavaScript functions for this object's graphic. 
	 * 
	 * @param parent element to which the script-related elements are appended
	 * @param numberOfDataSets the number of data sets in the input
	 */
	protected void addJavaScriptFunctions(Document generatedDocument, GraphicAttributes attrs, int numberOfDataSets, int gridXOffset, int gridYOffset, int xAxisLength, int yAxisLength) {
		boolean hasLegend = !attrs.isLegendSuppressed();
		String cookieName = attrs.getBrowserCookieName();
		
		Element parent = generatedDocument.getDocumentElement();	
		Element newElement = generatedDocument.createElement("script");		
		newElement.setAttribute("type", JAVASCRIPT_VERSION);
		parent.appendChild(newElement);
		
		ECMAScriptGenerator jscript = new ECMAScriptGenerator();
		StringBuffer functions = new StringBuffer(jscript.generateGlobalVariables());
		
//		// for when SVG has to be transcoded into some other format which does not recognize browser variables.		
		if (attrs.getSvgFormatOnly()) {
			functions.append(jscript.generateClientSnifferVariables());		
		}
		else {
			functions.append(jscript.generateNullClientSnifferVariables());	
		}	
		
		// save user preferences to cookie only if a name is supplied					
		if (cookieName != null) {	
			functions.append(jscript.generateCookieFunctions());			
			functions.append(jscript.generateStorePreferencesFunction(cookieName));
			functions.append(jscript.generateGetPreferencesFunction(cookieName));		
		}	
			
		functions.append(jscript.generateInitFunction(attrs.getUserPreferences(), attrs.getGraphicColorPalette(), attrs.getGraphicColorPaletteName(), attrs.getGraphicFont(), cookieName, graphicType,  numberOfDataSets, gridXOffset, gridYOffset, xAxisLength, yAxisLength));				
		functions.append(jscript.generateToggleVisibilityFunction(numberOfDataSets,graphicType));					
		functions.append(jscript.generateUpdateStylesFunction(numberOfDataSets, cookieName, graphicType, hasLegend, attrs.getUserPreferencesUpdateVariableName(), gridXOffset, gridYOffset, xAxisLength, yAxisLength));		
		// generate legend-specific script
		if (hasLegend) {
			functions.append(jscript.generateUpdateLegendStylesFunction(numberOfDataSets, graphicType));
			functions.append(jscript.generateFunctionFlyOver());			
		}
		functions.append(jscript.generateFunctionPrefFlyOver());
		functions.append(jscript.generateGetCurrentValuesFunction());	
		functions.append(jscript.generateGetUserPreferencesFunction());	
		functions.append(jscript.generateResetStylesFunction());	
		functions.append(jscript.generateGetPalettesFunction());		
		functions.append(jscript.generateChangeOpacity());		
		// resourcesFolder should have been set by user.
		// preference page should have been set in widget CTOR and could have been 
		// overridden by user with 'setPreferencePage' - if resourcesFolder is null, 
		// page location has to be resolvable from folder containing the html page 
		// embedding the graphic.	
		String preferencesPage = attrs.getPreferencesPage();	
		if (preferencesPage != null) {			
			functions.append(jscript.generateOpenPreferencePageFunction(attrs.getResourcesFolder(), preferencesPage, graphicType, attrs.getPreferencesPageWidth(), attrs.getPreferencesPageHeight()));		
			// This is needed for basic types preference pages (i.e., those prefixed with "prefs_") only 
			// as long as support is needed for NS 4.7
			if (preferencesPage.indexOf(IUserPreferencesConstants.PREFS) == 0) {
				functions.append(jscript.generateUseNetscapePreferencePage(attrs.getResourcesFolder(), preferencesPage, graphicType, attrs.getPreferencesPageWidth(), attrs.getPreferencesPageHeight()));	
			}		
		}				
		
		// for debugging purposes, comment out when final
		functions.append(jscript.generateInspectFunction());		
		
		CDATASection newCDATASection = generatedDocument.createCDATASection(functions.toString());
		newElement.appendChild(newCDATASection);
		
		
	}	
			
	/**
	 * Convenience method for adding text elements.
	 * 
	 * @param generatedDocument the SVG document
	 * @param parent element to which the label (text element) is appended.
	 * @param stylesheetClass CSS rule for the label
	 * @param x x location for the label
	 * @param y y location for the label
	 * @param rotation the value by which the label is rotated (0 - 360)
	 */
	protected Element addLabel(Document generatedDocument, GraphicAttributes attrs, Element parent, String title, String stylesheetClass, double x, double y, int rotation) {		
		Element newElement = createDOMTextElement(generatedDocument, attrs.getGraphicFont(), title, stylesheetClass, Double.toString(x), Double.toString(y), rotation);
		parent.appendChild(newElement);   		
		return newElement;
	}	
		
	/**
	 * Convenience method for creating DOM text nodes.
	 * 
	 * @param text text of the element
	 * @param styleClass CSS rule for the element
	 * @param x The x location for the text
	 * @param y The y location for the text
	 * @param rotation the value by which the text is rotated (0 - 360)
	 */
	protected Element createDOMTextElement(Document generatedDocument, String font, String text, String styleClass, String x, String y, int rotation) {
		Element newElement = generatedDocument.createElement("text");
		if (styleClass != null) {
	   		newElement.setAttribute("class", styleClass);
		}
		newElement.setAttribute("font", font);
	    newElement.setAttribute("x", x);
	    newElement.setAttribute("y", y);
	    //newElement.setAttribute("textLength", Integer.toString(text.length()));
	    if (rotation > 0 && rotation < 360) {
	    	newElement.setAttribute("transform", "rotate(" + rotation + "," + x + "," + y + ")");
	    }	
	    newElement.appendChild(generatedDocument.createTextNode(text));
	    return newElement;
	}	
	
		
	/**
	 * Creates an SVG document and assigns width and height to the root "svg" element.
	 * 
	 * @param dImplement implementation to use for document creation.
	 * @param width the width to assign
	 * @param height the height to assign
	 * @return Document the SVG document
	 */
	protected Document createSvgDocument(DOMImplementation dImplement, String width, String height) {		
		DocumentType dType = dImplement.createDocumentType("svg", SVG_VERSION, SVG_DTD);
		Document svgDocument = dImplement.createDocument(null, "svg", dType);
		Element svgRoot = (Element) svgDocument.getDocumentElement();		

		svgRoot.setAttribute("xmlns:xlink", SVG_XMLNS);
		svgRoot.setAttribute("width", width);										   	
		svgRoot.setAttribute("height", height);
//		svgRoot.setAttribute("viewbox", "0 0 " + width + " " + height);		
		svgRoot.setAttribute("viewbox", "0 0 " + (Integer.parseInt(width)-1 ) + " " + (Integer.parseInt(height)-1));				
		
		return svgDocument;
	}	
	
	
	/**
	 * Adds a definition for the default legend item graphic.
	 * 
	 * @param generatedDocument the SVG document
	 * @param defs the element to which the definition is appended
	 */
	protected void defineLegendGraphic(Document generatedDocument, Element defs) {				
		Element gElement_legendsquareWithoutLine = generatedDocument.createElement("g");	
		Element legendsquareElementWithoutLine = generatedDocument.createElement("rect");
		
		gElement_legendsquareWithoutLine.setAttribute("id","legendsquareshapeWithoutLine");	
		defs.appendChild(gElement_legendsquareWithoutLine);		
		
		legendsquareElementWithoutLine.setAttribute("width", "8");
		legendsquareElementWithoutLine.setAttribute("height", "8");	
		legendsquareElementWithoutLine.setAttribute("transform", "translate(3,2)");
		gElement_legendsquareWithoutLine.appendChild(legendsquareElementWithoutLine);
    }	
    
    				
	/**
	 * Adds a definition for the user preferences graphic.
	 * 
	 * @param generatedDocument the SVG document
	 * @param defs the element to which the definition is appended
	 */
	protected void definePreferenceIcon(Document generatedDocument, Element defs) {
		Element gElement_preferences = generatedDocument.createElement("g");
		Element gElement = generatedDocument.createElement("g");
		Element newElement = generatedDocument.createElement("path");
		
		gElement_preferences.setAttribute("id", "preferences");
		defs.appendChild(gElement_preferences);
				
		gElement.setAttribute("id", "graph");
		gElement.setAttribute("style","fill-rule:nonzero;clip-rule:nonzero;fill:none;stroke:#999999;stroke-miterlimit:4;");
		gElement_preferences.appendChild(gElement);
				
		newElement.setAttribute("d", "M2.042,4.811H0.5v18h21v-6.542");
		gElement.appendChild(newElement);
		  
		newElement = generatedDocument.createElement("path");
		newElement.setAttribute("d", "M2.042,4.811H0.5");
		gElement.appendChild(newElement);
		    
		newElement = generatedDocument.createElement("path");
		newElement.setAttribute("d", "M2.042,10.801H0.5");
		gElement.appendChild(newElement);
		    		    
		newElement = generatedDocument.createElement("path");
		newElement.setAttribute("d", "M2.042,16.8H0.5");
		gElement.appendChild(newElement);
		
		newElement = generatedDocument.createElement("path");
		newElement.setAttribute("d", "M6.51,22.81v-18H3.521v18H6.51z");
		newElement.setAttribute("style", "fill:#000099;");
		gElement.appendChild(newElement);
				
		newElement = generatedDocument.createElement("path");
		newElement.setAttribute("d","M12.51,22.81v-11H9.521v11h2.989z");
		newElement.setAttribute("style", "fill:#0066CC;");
		gElement.appendChild(newElement);
		
		newElement = generatedDocument.createElement("path");
		newElement.setAttribute("d","M18.51,22.81v-5h-2.989v5h2.989z");
		newElement.setAttribute("style", "fill:#99CCFF;");
		gElement.appendChild(newElement);
		
		newElement = generatedDocument.createElement("path");
		newElement.setAttribute("d","M21.532,7.842l-5.145-5.248c-1.295,1.295-2.096,3.083-2.096,5.06c0,3.953,3.286,7.156,7.238,7.156l0.002-6.968z");
		newElement.setAttribute("style", "fill:#0066CC;");
		gElement.appendChild(newElement);
		
		newElement = generatedDocument.createElement("path");
		newElement.setAttribute("d","M21.557,14.81c3.952,0,7.047-3.11,7.047-7.063h-7.047v7.063z");
		newElement.setAttribute("style", "fill:#0066CC;");
		gElement.appendChild(newElement);
		
		newElement = generatedDocument.createElement("path");
		newElement.setAttribute("d","M28.604,7.654c0-3.953-3.204-7.156-7.156-7.156c-1.977,0-3.766,0.801-5.061,2.097l5.092,5.278h7.125");
		newElement.setAttribute("style", "fill:#99CCFF;");
		gElement.appendChild(newElement);
	
	}
		
	
	/**
	 * Registers a JavaScript event handler for an event type on an element.
	 * 
	 * @param target the element
	 * @param event the event type
	 * @param script the handler (a JavaScript function or code snippet)
	 * @return Document the svg document
	 */
	protected void registerEventHandler(Element target, String event, String script) {		
		if (target != null) {
			target.setAttribute(event, script);
		}		
	}		
	
	
	/**
	 * Adds a title bar separator line.
	 * 
	 * @param generatedDocument the SVG document
	 * @param object containing the graphic properties
	 * @param parent element to which the separator line is appended.
	 */
	private void addTitleBarLine(Document generatedDocument, GraphicAttributes attrs, Element parent, int yValue) {					
		StringBuffer points = new StringBuffer();						
		Element newElement = generatedDocument.createElement("polyline");
		
		newElement.setAttribute("class", "st1, keyline");
		points.append("0 ");    
	    points.append(yValue);
	    points.append(" ");
	    points.append(attrs.getGraphicWidth()-1);
	    points.append(" ");
	    points.append(yValue); 
	    newElement.setAttribute("points", points.toString());
	    parent.appendChild(newElement);	      
	}
	
		
	/**
	 * Adds the preferences icon to this object's graphic.
	 * 
	 * @param generatedDocument 
	 * @param attrs object containing the graphic properties
	 * @param parent element to which the separator line is appended.
	 */
	private void addPreferenceIcon(Document generatedDocument, GraphicAttributes attrs, Element parent) {			
		// arbitrary preference icon location based on size(w=29.1, h=23.3) 		
		final int iconXOffset = attrs.getGraphicWidth() - 33;
		final int iconYOffset = 2;		
				    
	    Element newElement = generatedDocument.createElement("use");
	    newElement.setAttribute("xlink:href", "#preferences");
		newElement.setAttribute("x", Integer.toString(iconXOffset));
		newElement.setAttribute("y", Integer.toString(iconYOffset)); 
		newElement.setAttribute("id", "preferencesIconHoverBox");		
		registerEventHandler(newElement, "onclick", "openPreferencePage()");
		String preferencesIconText = attrs.getPreferencesIconText();
		
		if (preferencesIconText != null) {	
			newElement.setAttribute("onmouseover", "PrefFlyOver();");
			newElement.setAttribute("onmouseout", "PrefFlyOut();");		
		}
		parent.appendChild(newElement);

		// add the hover text
		if (preferencesIconText != null) {			
			Element gElement = generatedDocument.createElement("g");
			gElement.setAttribute("id","preferencesHoverText");				
			gElement.setAttribute("pointer-events","none");
			newElement = generatedDocument.createElement("rect");
			newElement.setAttribute("id", "hoverBox");
			newElement.setAttribute("width", "1");
			newElement.setAttribute("height", "1");
			newElement.setAttribute("fill", "#FFFFCC");
			newElement.setAttribute("stroke", "black");
			newElement.setAttribute("stroke-width", "1");
			gElement.appendChild(newElement);
			
			newElement = generatedDocument.createElement("text");
			newElement.setAttribute("id", "hoverBoxText");		
			// anchor text at icon mid=point and bottom		
			newElement.setAttribute("x",Integer.toString(iconXOffset + 15));
			newElement.setAttribute("y",Integer.toString(iconYOffset + 30));		
			newElement.setAttribute("class","timeStamp");
			newElement.setAttribute("visibility","hidden");					
			newElement.appendChild(generatedDocument.createTextNode(preferencesIconText));
			gElement.appendChild(newElement);
			parent.appendChild(gElement);
		}
    }
    
    
    /**
	 * Adds the main title. 
	 * 
     * @param generatedDocument the SVG document
	 * @param object containing the graphic properties
	 * @param parent element to which the separator line is appended.
	 */
	private void addTitle(Document generatedDocument, GraphicAttributes attrs, Element parent) {				
		int fontSize = DEFAULT_FONTSIZE + TITLE_FONTSIZE_DELTA;
		String title = attrs.getGraphicTitle();		
		if (title != null) {
			String stylesheetClass = "graphTitle anchorAtStart";
			Element gElement = generatedDocument.createElement("g");
			gElement.setAttribute("id", "graphTitle");
			gElement.setAttribute("class", stylesheetClass);
			parent.appendChild(gElement);	
			// arbitrary placement	???
			addLabel(generatedDocument, attrs, gElement, title, null, 10, fontSize + 8, 0);
		}			
	}	
	
 	/**
	 * Convenience method for adding text elements.
	 * 
	 * @param generatedDocument the SVG document
	 * @param parent element to which the label (text element) is appended.
	 * @param stylesheetClass CSS rule for the label
	 * @param x x location for the label
	 * @param y y location for the label
	 * @param rotation the value by which the label is rotated (0 - 360)
	 */
	protected Element addHoverText(Document generatedDocument, GraphicAttributes attrs, int index, Element parent, String title, String flyOverText, double x, double y, int rotation) {		

		Element newElement = createDOMTextElement(generatedDocument, attrs.getGraphicFont(), title, null, Double.toString(x), Double.toString(y), rotation);
		int xValue = attrs.getGraphicWidth() - 15;
		newElement.setAttribute("id", "legent"+index);
		newElement.setAttribute("onmouseover", "FlyOver("+index+");");
		newElement.setAttribute("onmouseout", "FlyOut("+index+");");
		parent.appendChild(newElement);   		

		if (flyOverText != null) 
		{			
			Element gElement = generatedDocument.createElement("g");
			gElement.setAttribute("id","flyOverGroup"+index);				
			gElement.setAttribute("pointer-events","none");
			newElement = generatedDocument.createElement("rect");
			newElement.setAttribute("id", "flyOverBox"+index);
			newElement.setAttribute("width", "1");
			newElement.setAttribute("height", "1");
			newElement.setAttribute("fill", "#FFFFCC");
			newElement.setAttribute("stroke", "gray");
			newElement.setAttribute("stroke-width", "1");
			gElement.appendChild(newElement);
			
			newElement = generatedDocument.createElement("text");
			newElement.setAttribute("id", "flyOverText"+index);		
			newElement.setAttribute("x",Double.toString(xValue));
			newElement.setAttribute("y",Double.toString(y));		
			newElement.setAttribute("class","dataValues anchorAtEnd");
			newElement.setAttribute("visibility","hidden");
			newElement.appendChild(generatedDocument.createTextNode(flyOverText));
			gElement.appendChild(newElement);
			parent.appendChild(gElement);
		}
		return newElement;
	}
	
	/**
	 * Returns the legendFlyOverLabels.
	 * @return String[]
	 */
	public String[] getLegendFlyOverLabels(DataSet[] dataSets) {
		String [] legendFlyOverLabels = null;
		
		legendFlyOverLabels = new String[dataSets.length];
			for (int i = 0; i < dataSets.length; i++) {
				legendFlyOverLabels[i] = dataSets[i].getFlyoverLabel();
				if(legendFlyOverLabels[i] == null){
					legendFlyOverLabels[i] = "";
				}
			}
		return legendFlyOverLabels;
	}			

	/**
	 * Calculate the area the legends need
	 * @return double the maximum width of the legend labels
	 */
	protected double getMaxLengthOfLegendLabel(GraphicAttributes graphicAttrs, DataSet[] dataSets) {
		int maxLengthOfLegendLable = 0;
		String legendTitle = graphicAttrs.getGraphicLegendTitle();
		if (legendTitle != null) {
			maxLengthOfLegendLable = legendTitle.length() - 2;
			if (maxLengthOfLegendLable < 0) maxLengthOfLegendLable = 0;
		}
		
		for (int i = 0; i < dataSets.length; i++) 
		{
			DataSet dataset = dataSets[i];
			String label = dataset.getLabel();
			if (label != null) {
				int length = label.length();
				if(maxLengthOfLegendLable < length)
				{
					maxLengthOfLegendLable = length;
				}
			}
		}
		
		//Calculate the max size of the Legent text width in pixel. 16 gap between Header and text and 7.0 is a fond size.
		return (double)(16 + (maxLengthOfLegendLable) * 7.0);		
	}


}
