package org.eclipse.hyades.logging.commons;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;

import org.apache.commons.logging.Log;
import org.eclipse.hyades.internal.logging.core.XmlGenerator;
import org.eclipse.hyades.logging.core.IExternalizableToXml;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.StartSituation;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Implementation of the Apache Commons
 * <code>org.apache.commons.logging.Log</code> interface which represents a
 * logger for logging messages to a local log file.
 * <p>
 * A Common Base Event start event is logged whenever this logger begins logging
 * to the local log file. Conversely, a Common Base Event stop event is logged
 * whenever this logger finishes logging to the local log file.
 * <p>
 * The logging level is used to filter which severity of logged messages are
 * sent to the local log file. The logging level may be set to one of the
 * following logging levels:
 * <p>
 * -TRACE (least serious and most verbose) -DEBUG -INFO -WARN -ERROR -FATAL
 * (most serious and least verbose)
 * <p>
 * An instance of this class will be returned from the <code>getLog()</code>
 * and/or <code>getInstance()</code> APIs on the default
 * <code>org.apache.commons.logging.LogFactory</code> implementation. The
 * <code>org.apache.commons.logging.LogFactory</code> implementation uses the
 * following discovery process to resolve the configured
 * <code>org.apache.commons.logging.Log</code> implementation:
 * <p>
 * 1) Look for a system property named
 * <code>org.apache.commons.logging.Log</code> or
 * <code>org.apache.commons.logging.log</code>. For example,
 * 
 * <code>...-Dorg.apache.commons.logging.Log=org.eclipse.hyades.logging.commons.FileLogger...</code>
 *  
 * - or -
 * 
 * <code>System.setProperty("org.apache.commons.logging.Log","org.eclipse.hyades.logging.commons.FileLogger");</code>
 * 
 * 2) Look for a configuration attribute of the default
 * <code>org.apache.commons.logging.LogFactory</code> implementation named
 * <code>org.apache.commons.logging.Log</code>. For example,
 * 
 * <code>org.apache.commons.logging.LogFactory.getFactory().setAttribute("org.apache.commons.logging.Log","org.eclipse.hyades.logging.commons.FileLogger");</code>
 * <p>
 * An instance of this class may be configured to log to a local log file (e.g.
 * absolute or relative path to the local log file) using one of the following
 * methods:
 * <p>
 * 1) Look for a system property named
 * <code>org.eclipse.hyades.logging.commons.FileLogger.FileName</code> at
 * logger creation. For example,
 * 
 * <code>...-Dorg.eclipse.hyades.logging.commons.FileLogger.FileName=/tmp/myLogFile.log...</code>
 * 
 * NOTE: The absolute or relative path to the local log file must be enclosed in
 * double quotes (e.g. "") if contains one or more spaces.
 *  
 * - or -
 * 
 * <code>System.setProperty("org.eclipse.hyades.logging.commons.FileLogger.FileName","/tmp/myLogFile.log");</code>
 * 
 * 2) Programmatically set the absolute or relative path to the local log file.
 * For example,
 * 
 * <code>logger.setFileName("/tmp/myLogFile.log");</code>
 * <p>
 * An instance of this class may be configured with a logging level using one of
 * the following methods (see above for the supported levels):
 * <p>
 * 1) Look for a system property named
 * <code>org.eclipse.hyades.logging.commons.FileLogger.LoggerLevel</code> at
 * logger creation. For example,
 * 
 * <code>...-Dorg.eclipse.hyades.logging.commons.FileLogger.Level=DEBUG...</code>
 *  
 * - or -
 * 
 * <code>System.setProperty("org.eclipse.hyades.logging.commons.FileLogger.Level","DEBUG");</code>
 * 
 * 2) Programmatically set the logging level. For example,
 * 
 * <code>logger.setLevelAsString("DEBUG");</code>
 *  
 * - or -
 * 
 * <code>logger.setLevel(FileLogger.DEBUG_LEVEL);</code>
 * <p>
 * A logger instance is created and returned when the <code>getLog()</code>
 * and/or <code>getInstance()</code> APIs on the default
 * <code>org.apache.commons.logging.LogFactory</code> implementation are
 * called. Once created, the logger is configurable (e.g. logging level). The
 * logger is generally consulted on its configuration before every message is
 * logged for performance reasons. For example,
 * <p>
 * <code>
 * System.setProperty("org.apache.commons.logging.Log", "org.eclipse.hyades.logging.commons.FileLogger");
 * System.setProperty("org.eclipse.hyades.logging.commons.FileLogger.Level", "DEBUG");
 * System.setProperty("org.eclipse.hyades.logging.commons.FileLogger.FileName", "/tmp/myLogFile.log");
 * <p>
 * Log logger = LogFactory.getLog("My Logger");
 * <p>
 * if(logger.isDebugEnabled()){
 *     logger.debug("My first DEBUG log message.");
 *     logger.debug("My second DEBUG log message.");
 * }
 * </code>
 * <p>
 * Alternatively, users may use the proxy factory specific for <code>org.eclipse.hyades.logging.commons.FileLogger</code> 
 * instances (see <code>org.eclipse.hyades.logging.commons.FileLoggerFactory</code>).
 * <p>
 * NOTE: The Apache Commons logging classes must be on the CLASSPATH at runtime
 * to utilize this logging class
 * <p>
 * 
 * 
 * @author Paul E. Slauenwhite
 * @version May 7, 2004
 * @since February 19, 2004
 * @see org.apache.commons.logging.Log
 * @see org.apache.commons.logging.LogFactory
 * @see org.eclipse.hyades.logging.commons.FileLoggerFactory
 */
public class FileLogger implements Log {

    /**
     * Trace logging level value.
     */
    public static final int TRACE_LEVEL = 0;

    /**
     * Debug logging level value.
     */
    public static final int DEBUG_LEVEL = 1;

    /**
     * Information logging level value.
     */
    public static final int INFO_LEVEL = 2;

    /**
     * Warning logging level value.
     */
    public static final int WARN_LEVEL = 3;

    /**
     * Error logging level value.
     */
    public static final int ERROR_LEVEL = 4;

    /**
     * Fatal logging level value.
     */
    public static final int FATAL_LEVEL = 5;

    /**
     * Array of string representations of the logging levels.
     * 
     * NOTE: The integer representation of each logging level (e.g. TRACE_LEVEL -
     * FATAL_LEVEL) MUST be equal to the index of the equivalent string
     * representation of each logging level (e.g. "TRACE" - "FATAL").
     */
    private static final String[] LEVEL_NAMES = { "TRACE", "DEBUG", "INFO", "WARN", "ERROR", "FATAL"};

    /**
     * System dependent line separator character.
     */
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");

    /**
     * The logger's current logging level value.
     */
    private int level = -1;

    /**
     * The logger's name.
     */
    private String name = null;

    /**
     * Utility XML generator to convert logged messages (e.g.
     * <code>java.lang.Object</code> and/or <code>java.lang.Throwable</code>)
     * to XML.
     */
    private XmlGenerator xmlGenerator = null;

    /**
     * Buffered writer to the local log file.
     */
    private BufferedWriter fileWriter = null;

    /**
     * Absolute or relative path to the local log file.
     */
    private String fileName = null;

    /**
     * A Common Base Event start event to be logged whenever this logger begins
     * logging to the local log file.
     */
    private CommonBaseEvent startEvent = null;

    /**
     * A Common Base Event stop event to be logged whenever this logger finishes
     * logging to the local log file.
     */
    private CommonBaseEvent stopEvent = null;

    /**
     * IP address (IPv4) of the local host, otherwise "127.0.0.1".
     */
    private static String localHostIP = null;

    static {
        try {
            localHostIP = InetAddress.getLocalHost().getHostAddress();
        } catch (UnknownHostException u) {
            localHostIP = "127.0.0.1";
        }
    }

    /**
     * Constructor to create a logger instance with the parameter name.
     * 
     * NOTE: The default logging level is set to WARN until explicitly set.
     * 
     * @param name
     *            The name of the newly created logger.
     */
    public FileLogger(String name) {

        this.level = WARN_LEVEL;
        this.name = name;

        xmlGenerator = new XmlGenerator(name);

        createStartStopEvents();

        configure();
    }

    /**
     * Checks if TRACE logging is currently enabled. TRACE logging infers the
     * logging level is currently set at TRACE or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if TRACE logging is currently enabled, otherwise false.
     */
    public boolean isTraceEnabled() {
        return (level <= TRACE_LEVEL);
    }

    /**
     * Checks if DEBUG logging is currently enabled. DEBUG logging infers the
     * logging level is currently set at DEBUG or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if DEBUG logging is currently enabled, otherwise false.
     */
    public boolean isDebugEnabled() {
        return (level <= DEBUG_LEVEL);
    }

    /**
     * Checks if INFO logging is currently enabled. INFO logging infers the
     * logging level is currently set at INFO or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if INFO logging is currently enabled, otherwise false.
     */
    public boolean isInfoEnabled() {
        return (level <= INFO_LEVEL);
    }

    /**
     * Checks if WARN logging is currently enabled. WARN logging infers the
     * logging level is currently set at WARN or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if WARN logging is currently enabled, otherwise false.
     */
    public boolean isWarnEnabled() {
        return (level <= WARN_LEVEL);
    }

    /**
     * Checks if ERROR logging is currently enabled. ERROR logging infers the
     * logging level is currently set at ERROR or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if ERROR logging is currently enabled, otherwise false.
     */
    public boolean isErrorEnabled() {
        return (level <= ERROR_LEVEL);
    }

    /**
     * Checks if FATAL logging is currently enabled. FATAL logging infers the
     * logging level is currently set at FATAL or lower.
     * 
     * This API permits an inexpensive check on logging level.
     * 
     * @return True if FATAL logging is currently enabled, otherwise false.
     */
    public boolean isFatalEnabled() {
        return (level <= FATAL_LEVEL);
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the
     * local log file if TRACE logging is currently enabled.
     * 
     * The log record is first converted to XML and then written to the local
     * log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     */
    public void trace(Object record) {

        if (isTraceEnabled()) {
            write(convertToXML(record));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and
     * <code>java.lang.Throwable</code> exception to the local log file if
     * TRACE logging is currently enabled.
     * 
     * The log record and exception are first converted to XML and then written
     * to the local log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     * @param throwable
     *            The exception to be logged to the local log file.
     */
    public void trace(Object record, Throwable throwable) {

        if (isTraceEnabled()) {
            write(convertToXML(record));
            write(convertToXML(throwable));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the
     * local log file if DEBUG logging is currently enabled.
     * 
     * The log record is first converted to XML and then written to the local
     * log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     */
    public void debug(Object record) {

        if (isDebugEnabled()) {
            write(convertToXML(record));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and
     * <code>java.lang.Throwable</code> exception to the local log file if
     * DEBUG logging is currently enabled.
     * 
     * The log record and exception are first converted to XML and then written
     * to the local log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     * @param throwable
     *            The exception to be logged to the local log file.
     */
    public void debug(Object record, Throwable throwable) {

        if (isDebugEnabled()) {
            write(convertToXML(record));
            write(convertToXML(throwable));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the
     * local log file if INFO logging is currently enabled.
     * 
     * The log record is first converted to XML and then written to the local
     * log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     */
    public void info(Object record) {

        if (isInfoEnabled()) {
            write(convertToXML(record));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and
     * <code>java.lang.Throwable</code> exception to the local log file if
     * INFO logging is currently enabled.
     * 
     * The log record and exception are first converted to XML and then written
     * to the local log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     * @param throwable
     *            The exception to be logged to the local log file.
     */
    public void info(Object record, Throwable throwable) {

        if (isInfoEnabled()) {
            write(convertToXML(record));
            write(convertToXML(throwable));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the
     * local log file if WARN logging is currently enabled.
     * 
     * The log record is first converted to XML and then written to the local
     * log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     */
    public void warn(Object record) {

        if (isWarnEnabled()) {
            write(convertToXML(record));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and
     * <code>java.lang.Throwable</code> exception to the local log file if
     * WARN logging is currently enabled.
     * 
     * The log record and exception are first converted to XML and then written
     * to the local log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     * @param throwable
     *            The exception to be logged to the local log file.
     */
    public void warn(Object record, Throwable throwable) {

        if (isWarnEnabled()) {
            write(convertToXML(record));
            write(convertToXML(throwable));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the
     * local log file if ERROR logging is currently enabled.
     * 
     * The log record is first converted to XML and then written to the local
     * log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     */
    public void error(Object record) {

        if (isErrorEnabled()) {
            write(convertToXML(record));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and
     * <code>java.lang.Throwable</code> exception to the local log file if
     * ERROR logging is currently enabled.
     * 
     * The log record and exception are first converted to XML and then written
     * to the local log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     * @param throwable
     *            The exception to be logged to the local log file.
     */
    public void error(Object record, Throwable throwable) {

        if (isErrorEnabled()) {
            write(convertToXML(record));
            write(convertToXML(throwable));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the
     * local log file if FATAL logging is currently enabled.
     * 
     * The log record is first converted to XML and then written to the local
     * log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     */
    public void fatal(Object record) {

        if (isFatalEnabled()) {
            write(convertToXML(record));
        }
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record and
     * <code>java.lang.Throwable</code> exception to the local log file if
     * FATAL logging is currently enabled.
     * 
     * The log record and exception are first converted to XML and then written
     * to the local log file.
     * 
     * @param record
     *            The log record to be logged to the local log file.
     * @param throwable
     *            The exception to be logged to the local log file.
     */
    public void fatal(Object record, Throwable throwable) {

        if (isFatalEnabled()) {
            write(convertToXML(record));
            write(convertToXML(throwable));
        }
    }

    /**
     * Retrieves the name of this logger.
     * 
     * @return Name of this logger.
     */
    public String getName() {
        return name;
    }

    /**
     * Sets the absolute or relative path to the local log file.
     * 
     * @param fileName
     *            The absolute or relative path to the local log file.
     * @exception java.lang.IllegalArgumentException
     *                Attempting to set an illegal absolute or relative path to
     *                the local log file.
     */
    public synchronized void setFileName(String fileName) throws IllegalArgumentException {

        this.fileName = fileName;

        try {
            openLogFile();
        } catch (IOException i) {

            this.fileName = null;

            throw new IllegalArgumentException("Invalid log file path");
        }
    }

    /**
     * Returns the absolute or relative path to the local log file.
     * 
     * @return The absolute or relative path to the local log file.
     */
    public String getFileName() {
        return fileName;
    }

    /**
     * Closes the file handle to the local log file.
     * 
     * @see java.lang.Object#finalize()
     */
    public final void finalize() {
        closeLogFile();
    }

    /**
     * Checks if the logger is logging to a local log file.
     * 
     * @return True if the logger is logging to a local log file, otherwise
     *         false.
     */
    public boolean isLogging() {
        return (fileWriter != null);
    }

    /**
     * Closes the file handle to the local log file.
     */
    private void closeLogFile() {

        if (isLogging()) {

            writeStopEvent();

            try {
                fileWriter.close();
            } catch (IOException i) {
            }
        }

        fileWriter = null;
    }

    /**
     * Opens the file handle to the local log file.
     * 
     * @exception IOException
     *                Attempting to open a handle to an illegal absolute or
     *                relative local log file path.
     */
    private void openLogFile() throws IOException {

        fileWriter = new BufferedWriter(new FileWriter(fileName, true));

        writeStartEvent();
    }

    /**
     * Retrieves the integer level of this logger.
     * 
     * @return Integer logging level of this logger (i.e. TRACE_LEVEL -
     *         FATAL_LEVEL).
     */
    public int getLevel() {
        return level;
    }

    /**
     * Retrieves the string level of this logger.
     * 
     * @return String logging level of this logger (i.e. "TRACE" - "FATAL").
     */
    public String getLevelAsString() {
        return LEVEL_NAMES[level];
    }

    /**
     * Sets the logging level to one of the integer logging levels (i.e.
     * TRACE_LEVEL - FATAL_LEVEL).
     * 
     * @param level
     *            The new integer logging level (i.e. TRACE_LEVEL -
     *            FATAL_LEVEL).
     * @exception java.lang.IllegalArgumentException
     *                Attempting to set an illegal integer logging level.
     */
    public synchronized void setLevel(int level) throws IllegalArgumentException {

        if (isValidLevel(level)) {
            this.level = level;
        } else {
            throw new IllegalArgumentException("Invalid logging level");
        }
    }

    /**
     * Sets the logging level to one of the string logging levels (i.e. "TRACE" -
     * "FATAL").
     * 
     * @param stringLevel
     *            The new string logging level (i.e. "TRACE" - "FATAL").
     * @exception java.lang.IllegalArgumentException
     *                Attempting to set an illegal string logging level.
     */
    public synchronized void setLevelAsString(String stringLevel) throws IllegalArgumentException {
        setLevel(getLevelInteger(stringLevel));
    }

    /**
     * Returns the integer level value for the parameter level String.
     * 
     * The parameter level string could be either the English (i.e. "TRACE" -
     * "FATAL") or the level number as a string (i.e. TRACE_LEVEL -
     * FATAL_LEVEL).
     * 
     * @param levelName
     *            a potential level string to be checked for its integer value.
     * @return The integer level value for the parameter level String, or -1 if
     *         invalid.
     */
    private int getLevelInteger(String levelName) {

        if ((levelName != null) && (levelName.trim().length() != 0)) {

            String tempLevel = levelName.trim();

            //Check against the English levels:
            for (int counter = 0; counter < LEVEL_NAMES.length; counter++) {

                if (tempLevel.equalsIgnoreCase(LEVEL_NAMES[counter])) { return counter; }
            }

            //Check for a valid integer:
            try {
                int levelNum = Integer.parseInt(tempLevel);

                if (isValidLevel(levelNum)) { return levelNum; }
            } catch (NumberFormatException n) {
            }
        }

        return -1;
    }

    /**
     * Checks if the parameter check level is one of the valid integer logging
     * levels (i.e. TRACE_LEVEL - FATAL_LEVEL).
     * 
     * @param checkLevel
     *            a potential level to be checked for validity.
     * @return true if the parameter check level is one of the valid integer
     *         logging levels (i.e. TRACE_LEVEL - FATAL_LEVEL), otherwise false.
     */
    private boolean isValidLevel(int checkLevel) {
        return ((checkLevel >= TRACE_LEVEL) || (checkLevel <= FATAL_LEVEL));
    }

    /**
     * Checks if the parameter check level name is one of the valid string
     * logging level (i.e. "TRACE" - "FATAL").
     * 
     * @param checkLevel
     *            a potential level name to be checked for validity.
     * @return true if the parameter check level name is one of the valid string
     *         logging level (i.e. "TRACE" - "FATAL"), otherwise false.
     */
    private boolean isValidLevelString(String checkLevel) {
        return (isValidLevel(getLevelInteger(checkLevel)));
    }

    /**
     * Internal utility API for converting the parameter
     * <code>java.lang.Object</code> to XML using the
     * <code>org.eclipse.hyades.internal.logging.core.XmlGenerator</code>.
     * 
     * @param object
     *            The object to be converted to XML.
     * @return The XML fragment of the converted parameter object.
     */
    private String convertToXML(Object object) {

        if (object instanceof IExternalizableToXml) {
            return ((IExternalizableToXml) object).externalizeCanonicalXmlString();
        } else {

            xmlGenerator.reset(LEVEL_NAMES[level], false, 4);

            return (xmlGenerator.objectToXML(object));
        }
    }

    /**
     * Writes the string message to the local log file.
     * 
     * @param message
     *            String message to be written to the local log file.
     */
    private void write(String message) {

        if (isLogging()) {

            try {

                fileWriter.write(message);
                fileWriter.flush();

                fileWriter.write(LINE_SEPARATOR);
                fileWriter.flush();

            } catch (IOException i) {
            }
        }
    }

    /**
     * Configures the logger using system properties.
     */
    private void configure() {

        String systemProperty = System.getProperty("org.eclipse.hyades.logging.commons.FileLogger.FileName");

        if ((systemProperty != null) && (systemProperty.trim().length() > 0)) {
            setFileName(systemProperty);
        }

        systemProperty = System.getProperty("org.eclipse.hyades.logging.commons.FileLogger.Level");

        if ((systemProperty != null) && (systemProperty.trim().length() > 0) && (isValidLevelString(systemProperty))) {
            setLevelAsString(systemProperty);
        }
    }

    /**
     * Creates a Common Base Event start event and a Common Base Event stop
     * event to the local log file.
     */
    private void createStartStopEvents() {

        EventFactory eventFactory = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();

        startEvent = eventFactory.createCommonBaseEvent();

        ComponentIdentification startSourceComponentId = eventFactory.createComponentIdentification();
        startSourceComponentId.setLocation(localHostIP);
        startSourceComponentId.setLocationType("IPV4");
        startSourceComponentId.setComponent("org.eclipse.hyades.logging.commons");
        startSourceComponentId.setSubComponent("FileLogger");
        startSourceComponentId.setComponentIdType("Application");
        startSourceComponentId.setComponentType("ApacheCommonsLog");

        startEvent.setSourceComponentId(startSourceComponentId);

        StartSituation startSituation = eventFactory.createStartSituation();
        startSituation.setReasoningScope("INTERNAL");
        startSituation.setSituationQualifier("START COMPLETED");
        startSituation.setSuccessDisposition("SUCCESSFUL");

        Situation startEventSituation = eventFactory.createSituation();
        startEventSituation.setCategoryName(Situation.START_SITUATION_CATEGORY);
        startEventSituation.setSituationType(startSituation);

        startEvent.setSituation(startEventSituation);

        stopEvent = eventFactory.createCommonBaseEvent();

        ComponentIdentification stopSourceComponentId = eventFactory.createComponentIdentification();
        stopSourceComponentId.setLocation(localHostIP);
        stopSourceComponentId.setLocationType("IPV4");
        stopSourceComponentId.setComponent("org.eclipse.hyades.logging.commons");
        stopSourceComponentId.setSubComponent("FileLogger");
        stopSourceComponentId.setComponentIdType("Application");
        stopSourceComponentId.setComponentType("ApacheCommonsLog");

        stopEvent.setSourceComponentId(stopSourceComponentId);

        StartSituation stopSituation = eventFactory.createStartSituation();
        stopSituation.setReasoningScope("INTERNAL");
        stopSituation.setSituationQualifier("STOP COMPLETED");
        stopSituation.setSuccessDisposition("SUCCESSFUL");

        Situation stopEventSituation = eventFactory.createSituation();
        stopEventSituation.setCategoryName(Situation.STOP_SITUATION_CATEGORY);
        stopEventSituation.setSituationType(stopSituation);

        stopEvent.setSituation(stopEventSituation);
    }

    /**
     * Writes a Common Base Event start event to the local log file.
     */
    private void writeStartEvent() {

        startEvent.setCreationTimeAsLong(System.currentTimeMillis());

        write(convertToXML(startEvent));
    }

    /**
     * Writes a Common Base Event stop event to the local log file.
     */
    private void writeStopEvent() {

        stopEvent.setCreationTimeAsLong(System.currentTimeMillis());

        write(convertToXML(stopEvent));
    }

    /**
     * Test main method.
     * 
     * @param args
     *            public static void main(String[] args) {
     * 
     * System.setProperty("org.apache.commons.logging.Log",
     * "org.eclipse.hyades.logging.commons.FileLogger");
     * System.setProperty("org.eclipse.hyades.logging.commons.FileLogger.Level",
     * "DEBUG");
     * System.setProperty("org.eclipse.hyades.logging.commons.FileLogger.FileName",
     * "/tmp/myLogFile.log");
     * 
     * Log logger = LogFactory.getLog("My Logger");
     * 
     * if (logger.isDebugEnabled()) { logger.debug("My first DEBUG log
     * message."); logger.debug("My second DEBUG log message."); }
     * 
     * System.out.println("Done!"); }
     */
}