/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.log.ui.internal.util;

import java.lang.reflect.Method;
import java.text.Collator;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.StringTokenizer;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.loaders.cbe.CBEUtils;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.models.cbe.CBECommonBaseEvent;
import org.eclipse.hyades.models.cbe.CBEPackage;
import org.eclipse.hyades.models.hierarchy.util.IFilterElement;

public class RecordFilterSearchUtil {
	
	private HashMap _methods = new HashMap();
	private ElementFilter _filter;	
	private Collator collator = Collator.getInstance();
	
	public class ElementFilter{
		protected String _pattern;
		protected String _prefix = "";
		protected String _suffix = "";
		protected ArrayList _textList;
		protected boolean _caseSensitive = true;
		protected boolean _exactMatch;
		protected boolean _noPattern = true;

		public ElementFilter() {
			super();
		}

		private void setPattern(String pattern, boolean caseSensitive) {
			_caseSensitive = caseSensitive;
			_pattern = _caseSensitive ? pattern : pattern.toLowerCase();
			if (pattern == "*" || pattern == "") {
				_noPattern = true;
			}
			else {
				_noPattern = false;
			}

			_exactMatch = (_pattern.indexOf("*") == -1);

			StringTokenizer st = new StringTokenizer(_pattern, "*");
			_textList = new ArrayList();
			while (st.hasMoreTokens()) {
				_textList.add(st.nextToken());
			}

			if (_pattern.startsWith("*")) {
				_prefix = "";
			}
			else if (_textList.size() > 0) {
				_prefix = (String) _textList.get(0);
				_textList.remove(0);
			}

			if (_pattern.endsWith("*")) {
				_suffix = "";
			}
			else if (_textList.size() > 0) {
				_suffix = (String) _textList.get(_textList.size() - 1);
				_textList.remove(_textList.size() - 1);
			}
		}
		
		public boolean match(String recordValue,String userValue) {
			setPattern(userValue,true);
			
			boolean flag = true;
									
			String compareText = "";
			
			if (_noPattern)
				return true;
			
		   compareText = recordValue;
			   
			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return collator.compare(compareText,_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {
					String str1 = (String) _textList.get(k);

					int index = compareText.indexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}
					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
	}
	
	public RecordFilterSearchUtil()
	{
		_filter = new ElementFilter();
	}

	public void dispose() {
		_methods.clear();
	}

	public boolean isAdvFilterApply(EObject logRecord,IFilterElement[] filterTableElements) {
		if (filterTableElements==null) return true;
		
		EClass eClass = (logRecord).eClass();
		EList attributes = eClass.getEAllAttributes();
		boolean filterApply = true; 
		int l = filterTableElements.length;
		for (int idx=0;idx<l;idx++)
		{
			IFilterElement filterTableElement = filterTableElements[idx];
			String elemAttribute = filterTableElement.getAttribute().toLowerCase();
			
			EAttribute attribute = findAttributeInList(elemAttribute, attributes);
			if (attribute!=null)
				filterApply = filterApply & matchFilter(logRecord,attribute,filterTableElement);
			else
				filterApply = filterApply & checkChildren(logRecord,elemAttribute,filterTableElement);
		}
		return filterApply;
	}
	
	public EAttribute findAttributeInList(String name, EList attributeList){	
			for(int i=0;i<attributeList.size();i++){			
				EAttribute attr = (EAttribute) attributeList.get(i);
				if(attr.getName().toLowerCase().equals(name))
					return attr;
			}
			return null;
	}
	
	private boolean matchFilter(EObject logRecord,EAttribute attribute, IFilterElement filterTableElement) {
		Object value = logRecord.eGet(attribute);
		String filterTableValue = filterTableElement.getValue();
		String filterTableOperator = filterTableElement.getOperator();
		
		if (value==null) value=new String("");
		if (filterTableValue==null) filterTableValue=new String("");

		if(attribute.getName().equals("creationTime")){
			value = CBEUtils.formatCreationTime((CBECommonBaseEvent)logRecord, LogUIPlugin.getDateTimeFormat());
		}
		
		if (filterTableOperator.equals(LogUIPlugin.getResourceString("STR_EQUAL")))
			return checkEqual(value.toString(),filterTableValue);
		else if (filterTableOperator.equals(LogUIPlugin.getResourceString("STR_NOT_EQUAL")))
			return checkNotEqual(value.toString(),filterTableValue);
		else if (filterTableOperator.equals(LogUIPlugin.getResourceString("STR_LARGER_THAN")))
			return checkLargerThan(value.toString(),filterTableValue);
		else if (filterTableOperator.equals(LogUIPlugin.getResourceString("STR_SMALLER_THAN")))
			return checkSmallerThan(value.toString(),filterTableValue);				
		else 
			return false; 
	}

	private boolean checkChildren(EObject logRecord,String elemAttribute, IFilterElement filterTableElement)  {
		boolean filterApply = true;
		boolean match = true;
		if (logRecord instanceof EObject && ((EObject)logRecord).eClass().getEPackage()==CBEPackage.eINSTANCE){
			match = false;
			final EClass eClass = ((EObject)logRecord).eClass();
			for (Iterator iter = eClass.getEAllStructuralFeatures().iterator(); iter.hasNext();) {
				EStructuralFeature feature = (EStructuralFeature) iter.next();
				if((feature instanceof EReference && feature.getFeatureID()!=CBEPackage.CBE_COMMON_BASE_EVENT__SYMPTOMS)
					|| (feature instanceof EAttribute && feature.isMany() &&  feature.getEType().getInstanceClass() != String.class))
				{
					Object refs = ((EObject)logRecord).eGet(feature, true);
					if(refs != null){ 
						if(refs instanceof EList){
							if(((EList)refs).size()>0){
								Object[] object = ((EList)refs).toArray();
								for (int i=0;i<object.length;i++)
								{
									int matchAttri = containsAttribute((EObject)object[i],elemAttribute,filterTableElement);
									if (matchAttri>0)
									{ 
										match=true;  		
										filterApply  = filterApply | (matchAttri==1?true:false);
									} 
								}
							}
						}
						else{
							int matchAttri = containsAttribute((EObject)refs,elemAttribute,filterTableElement);
							if (matchAttri>0)
							{
								match = true;
								filterApply  = filterApply & (matchAttri==1?true:false);
							}
						}
					}
				}
			}
		}
		else if (logRecord instanceof FeatureNode){
			final EStructuralFeature feature = ((FeatureNode)logRecord).getFeature();
			final EObject cElem = ((FeatureNode)logRecord).getElement();
			match = false;
			if(feature instanceof EReference)
			{
				if(feature.isMany()){				
					EList refs = (EList)cElem.eGet(feature, true);
					if(((EList)refs).size()>0){
						Object[] object = ((EList)refs).toArray();
						for (int i=0;i<object.length;i++)
						{
							int matchAttri = containsAttribute((EObject)object[i],elemAttribute,filterTableElement);
							if (matchAttri>0)
							{ 
								match=true;  	
								filterApply  = filterApply | (matchAttri==1?true:false);
							}
						}
					}
				}
				else{
					Object ref = cElem.eGet(feature, true);
					int matchAttri = containsAttribute((EObject)ref,elemAttribute,filterTableElement);
					if (matchAttri>0)
					{
						match = true;
						filterApply  = filterApply & (matchAttri==1?true:false);
					}
				}	
			}
		}
		else if(logRecord instanceof TerminalNode){}		
		return (match && filterApply);
	}	
	
	private int containsAttribute(EObject eObject, String elemAttribute, IFilterElement filterTableElement){
		EClass eClass = eObject.eClass();
		EList attributes =eClass.getEAllAttributes();
		EAttribute attribute = findAttributeInList(elemAttribute, attributes);
		if (attribute!=null)
		{
			return (matchFilter(eObject,attribute,filterTableElement)?1:-1);
		}
		return 0;
	}
	
	private Method getMethod(Class cls, String name) throws Exception{
			Method method = (Method)_methods.get(cls.getName() + name);
			if(method == null)
			{			
			   method = cls.getMethod(name, null);
			   _methods.put(cls.getName() + name, method);			  
			}
			
			return method;
	}

	private boolean checkEqual(String recordValue, String filterValue) {
		try {
			float recordFloat = Float.parseFloat(recordValue);
			float filterFloat = Float.parseFloat(filterValue);
			return (recordFloat==filterFloat?true:false);
		}
		catch (NumberFormatException nfe)
		{
			return _filter.match(recordValue,filterValue);
		}
	}
	
	private boolean checkNotEqual(String recordValue, String filterValue) {
		return !checkEqual(recordValue, filterValue);
	}
	
	private boolean checkLargerThan(String recordValue, String filterValue) {
		try {
			float recordFloat = Float.parseFloat(recordValue);
			float filterFloat = Float.parseFloat(filterValue);
			return (recordFloat>filterFloat?true:false);
		}
		catch (NumberFormatException nfe)
		{
			int value = collator.compare(recordValue,filterValue);
			return (value>0?true:false);
		}
	}
	
	private boolean checkSmallerThan(String recordValue, String filterValue) {
		return !checkLargerThan(recordValue, filterValue) && !checkEqual(recordValue,filterValue);
	}
	
}
