/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.adapter;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IAdapterFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.ui.IPersistableElement;
import org.eclipse.ui.model.IWorkbenchAdapter;
import org.eclipse.ui.views.properties.IPropertySource;

import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.internal.PropertySourceExtensionManager;
import org.eclipse.hyades.test.ui.internal.model.CMNNamedElementUtil;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.internal.model.ResourceUtil;
import org.eclipse.hyades.ui.adapter.IHyadesWorkbenchAdapter;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.hyades.ui.util.INamedElement;

/**
 * The Hyades Test adapter factory.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class TestAdapterFactory 
implements IAdapterFactory, IDisposable
{
	/**
	 * Singleton for TestAdapterFactory
	 */
	public static final TestAdapterFactory INSTANCE = new TestAdapterFactory();
	
	private TestWorkbenchAdapter testWorkbenchAdapter;
	
	/**
	 * Constructor for TestAdapterFactory
	 */
	protected TestAdapterFactory()
	{
		testWorkbenchAdapter = new TestWorkbenchAdapter();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		testWorkbenchAdapter = null;
	}

	/**
	 * Returns the <code>IHyadesWorkbenchAdapter</code> used by this factory 
	 * @return IHyadesWorkbenchAdapter
	 */
	public IHyadesWorkbenchAdapter getHyadesWorkbenchAdapter()
	{
		return testWorkbenchAdapter;
	}

	/**
	 * @see org.eclipse.core.runtime.IAdapterFactory#getAdapterList()
	 */
	public Class[] getAdapterList()
	{
		return new Class[] {
			CMNNamedElement.class,
			IWorkbenchAdapter.class,
			TestWorkbenchAdapter.class,
			IPropertySource.class,
			IPersistableElement.class,
			INamedElement.class,
			IResource.class,
			IFile.class
		};
	}

	/**
	 * @see org.eclipse.core.runtime.IAdapterFactory#getAdapter(java.lang.Object, java.lang.Class)
	 */
	public Object getAdapter(Object adaptableObject, Class adapterType)
	{
		if(adapterType == CMNNamedElement.class)
			return getCMNNamedElement(adaptableObject);

		if((adapterType == TestWorkbenchAdapter.class) || (adapterType == IWorkbenchAdapter.class))
			return getTestWorkbenchAdapter(adaptableObject);
			
		if(adapterType == IPropertySource.class)
			return getPropertySource(adaptableObject);

		if(adapterType == INamedElement.class)
			return getNamedElement(adaptableObject);

		if(adapterType == IPersistableElement.class)
			return getPersistableElement(adaptableObject);

		if((adapterType == IFile.class) || (adapterType == IResource.class))
			return getFile(adaptableObject);

		return null;
	}
	
	/**
	 * Returns the <code>CMNNamedElement</code> associated with a given 
	 * adaptable object.
	 * @param adaptableObject
	 * @return CMNNamedElement
	 */
	protected CMNNamedElement getCMNNamedElement(Object adaptableObject)
	{
		ResourceSet resourceSet = null;
		IFile file = null;
		
		if(adaptableObject instanceof IResourceSetFileAdaptable)
		{
			IResourceSetFileAdaptable adaptable = (IResourceSetFileAdaptable)adaptableObject;
			resourceSet = adaptable.getResourceSet();
			file = adaptable.getFile();
		}
		else if(adaptableObject instanceof IFile)
		{
			file = (IFile)adaptableObject;
		}
		
		if(file != null)
		{
			if(TestUI.isSupportedFileExtension(file.getFileExtension()))
			{
				EObject[] eObjects = EMFUtil.load(resourceSet, file);
				if((eObjects.length == 1) && (eObjects[0] instanceof CMNNamedElement))
					return (CMNNamedElement)eObjects[0];
			}
		}
			
		return null;
	}
	
	/**
	 * Returns the Hyades workbench adapter associated with a given 
	 * adaptable object.
	 * @param adaptableObject
	 * @return IHyadesWorkbenchAdapter
	 */
	protected IHyadesWorkbenchAdapter getTestWorkbenchAdapter(Object adaptableObject)
	{
		if(adaptableObject instanceof CMNNamedElement)
			return getHyadesWorkbenchAdapter();
			
		if(adaptableObject instanceof TPFExecutionEvent)
			return getHyadesWorkbenchAdapter();

		if(adaptableObject instanceof IResource)
			return getHyadesWorkbenchAdapter();

		if(adaptableObject instanceof LogicalFolder)
			return getHyadesWorkbenchAdapter();

		if(adaptableObject instanceof Resource)
			return getHyadesWorkbenchAdapter();

		return null;
	}
	
	/**
	 * Returns the property source associated with a given adaptable object.
	 * @param adaptableObject
	 * @return IPropertySource
	 */
	protected IPropertySource getPropertySource(Object adaptableObject)
	{
		if(PropertySourceExtensionManager.getInstance().isExtendableObject(adaptableObject)) {
			IPropertySource ps = PropertySourceExtensionManager.getInstance().getPropertySource(adaptableObject);
			if(ps != null) {
				return ps;
			}
		}
		
		if(adaptableObject instanceof CMNNamedElement)
			return new CMNNamedElementPropertySource((CMNNamedElement)adaptableObject);
			
		if(adaptableObject instanceof Resource)
			return (IPropertySource)getFile(adaptableObject).getAdapter(IPropertySource.class);
			
//		if(adaptableObject instanceof IAdaptable)
//			return (IPropertySource)((IAdaptable)adaptableObject).getAdapter(IPropertySource.class);
			
		return null;
	}

	/**
	 * Returns the persisted element associated with a given adaptable object.
	 * @param adaptableObject
	 * @return IPersistableElement
	 */
	protected IPersistableElement getPersistableElement(Object adaptableObject)
	{
		if(adaptableObject instanceof CMNNamedElement)
			return new CMNNamedElementUtil.PersistableElement((CMNNamedElement)adaptableObject);

		if(adaptableObject instanceof Resource)
			return new ResourceUtil.PersistableElement((Resource)adaptableObject);

//		if(adaptableObject instanceof IAdaptable)
//			return (IPersistableElement)((IAdaptable)adaptableObject).getAdapter(IPersistableElement.class);

		return null;
	}

	/**
	 * Returns the named element associated with a given adaptable object.
	 * @param adaptableObject
	 * @return IPropertySource
	 */
	protected INamedElement getNamedElement(Object adaptableObject)
	{
		if(adaptableObject instanceof CMNNamedElement)
			return new CMNNamedElementUtil.UINamedElement((CMNNamedElement)adaptableObject);
		
		return null;
	}
	
	/**
	 * Returns the file associated with a given adaptable object.
	 * @param adaptableObject
	 * @return IPropertySource
	 */
	protected IFile getFile(Object adaptableObject)
	{
		if(adaptableObject instanceof Resource)
			return EMFUtil.getWorkspaceFile((Resource)adaptableObject);
		
		if(adaptableObject instanceof EObject)
		{
			EObject eObject = (EObject)adaptableObject;
			if(eObject.eContainer() == null)
				return EMFUtil.getWorkspaceFile((EObject)adaptableObject);
		}
			
		return null;
	}	
	
}
