/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.editor.form.util;

import java.util.Iterator;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.common.CommonPackage;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormSection;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.util.NamedElementLink;
import org.eclipse.hyades.ui.editor.IEditorExtension;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

/**
 * @author marcelop
 * @since 0.2.0
 */
public class NamedElementsSection 
extends FormSection implements IDisposable
{
	private class NamedElementListAdapter
	extends AdapterImpl implements IDisposable
	{
		private EStructuralFeature feature;
		private String statusLinePrefix;
		
		public NamedElementListAdapter(EStructuralFeature feature, String statusLinePrefix)
		{
			this.feature = feature;
			this.statusLinePrefix = statusLinePrefix;
		}
		
		/**
		 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
		 */
		public void dispose()
		{
			feature = null;
			setTarget(null);
		}

		/**
		 * @see org.eclipse.emf.common.notify.Adapter#notifyChanged(org.eclipse.emf.common.notify.Notification)
		 */
		public void notifyChanged(Notification msg)
		{
			if(msg.getFeature() != feature)
				return;
				
			switch(msg.getEventType())
			{
				case Notification.ADD:
				case Notification.ADD_MANY:
				case Notification.REMOVE:
				case Notification.REMOVE_MANY:
				case Notification.MOVE:
					refreshComposite((EObject)msg.getNotifier());
					break;
			}
		}
		
		protected void refreshComposite(EObject notifier)
		{
			disposeChildren(hyperlinksParent);
			EList namedElements = (EList)notifier.eGet(feature);
			
			for (Iterator i = namedElements.iterator(); i.hasNext();)
			{
				CMNNamedElement element = (CMNNamedElement)i.next();
				if(element.eIsProxy())
					continue;
				NamedElementLink link = new NamedElementLink(editorExtension, destinationPage);
				link.setHoverMessagePrefix(statusLinePrefix);   
				link.create(factory, hyperlinksParent, element);
			}
			
			hyperlinksParent.pack(true);

			Composite parent = hyperlinksParent.getParent();
			parent.setSize(parent.computeSize(parent.getSize().x, SWT.DEFAULT));					
			parent.getParent().setSize(parent.getParent().computeSize(parent.getParent().getSize().x, SWT.DEFAULT));
		}		
		
		protected void disposeChildren(Composite parent)
		{
			Control[] children = parent.getChildren();
			for (int i = 0, maxi = children.length; i < maxi; i++)
				children[i].dispose();
		}
	}
	
	protected IEditorExtension editorExtension;
	protected int destinationPage;

	private String moreButtonToolTip;
	protected FormWidgetFactory factory;
	protected Composite hyperlinksParent;
	
	private NamedElementListAdapter adapter;

	/**
	 * Constructor for HyperlinkSection
	 */
	public NamedElementsSection(IEditorExtension editorExtension, int destinationPage)
	{
		this.editorExtension = editorExtension;
		this.destinationPage = destinationPage;
		moreButtonToolTip = ""; //$NON-NLS-1$
	}
	
	/**
	 * Disposes this section.
	 */
	public void dispose()
	{
		editorExtension = null;
		factory = null;
		hyperlinksParent = null;
		
		if(adapter != null)
		{
			if(adapter.getTarget() != null)
				adapter.getTarget().eAdapters().remove(adapter);
			adapter.dispose();
		}
		
		super.dispose();
	}

	public void setMoreButtonToolTip(String text)
	{
		if(text == null)
			moreButtonToolTip = ""; //$NON-NLS-1$
		else
			moreButtonToolTip = text;
	}

	/**
	 * Creates the content of this section.
	 * @param parent the parent composite.
	 * @param formWidgetFactory the widget factory.
	 * @return The composite which represents the content of the section.
	 */
	public Composite createClient(Composite parent, FormWidgetFactory formWidgetFactory)
	{
		this.factory = formWidgetFactory;
		
		Composite container = formWidgetFactory.createComposite(parent);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		container.setLayout(gridLayout);
		container.setLayoutData(GridDataUtil.createFill());
				
		hyperlinksParent = formWidgetFactory.createComposite(container);
		gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		gridLayout.verticalSpacing = 2;
		hyperlinksParent.setLayout(gridLayout);
		hyperlinksParent.setLayoutData(GridDataUtil.createFill());
		hyperlinksParent.getAccessible().addAccessibleListener(this);		

		Composite buttonContainer = formWidgetFactory.createComposite(container);
		buttonContainer.setLayoutData(new GridData(GridData.FILL_VERTICAL));
		gridLayout = new GridLayout();
		gridLayout.marginWidth = 0;
		gridLayout.marginHeight = 0;
		buttonContainer.setLayout(gridLayout);

		Button moreButton = formWidgetFactory.createButton(buttonContainer, TestUIPlugin.getString("ACT_MORE"), SWT.PUSH); //$NON-NLS-1$
		moreButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END | GridData.VERTICAL_ALIGN_BEGINNING));
		moreButton.setToolTipText(moreButtonToolTip);
		moreButton.addSelectionListener(new SelectionAdapter()
		{
			public void widgetSelected(SelectionEvent e)
			{
				editorExtension.getHyadesEditorPart().setActivePage(destinationPage);
			}
		});
		
		return container;
	}
	
	public void initialize(EObject container, EStructuralFeature namedElementFeature, String statusLinePrefix)
	{
		if(!CommonPackage.eINSTANCE.getCMNNamedElement().isSuperTypeOf((EClass)namedElementFeature.getEType()))
			return;
			
		if(!namedElementFeature.isMany())
			return;
			
		adapter = new NamedElementListAdapter(namedElementFeature, statusLinePrefix);
		container.eAdapters().add(adapter);
		adapter.setTarget(container);
		adapter.refreshComposite(container);
	}
	
	/**
	 * Utility method.  Clients may choose not to execute this in order to customize the
	 * section.
	 * @return Control
	 */
	public Control createControl(Composite parent, FormWidgetFactory formWidgetFactory, String headerText, String headerDescription)
	{
		setCollapsable(true);
		setHeaderText(headerText);
		setDescription(headerDescription);
		Control control = createControl(parent, formWidgetFactory);
		control.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING));
		return control;
	}
}
