/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.tabs;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.ui.internal.launch.providers.DeploymentProvider;
import org.eclipse.hyades.test.ui.internal.launch.providers.TestComponentProvider;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.launch.configurations.DeploymentLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.configurations.TestComponentLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.extensions.IDeploymentsProvider;
import org.eclipse.hyades.test.ui.launch.extensions.ITestLaunchConfigurationValidator;
import org.eclipse.jface.viewers.IElementComparer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.ui.IWorkbenchPage;

/**
 * Provides the Test & Deployment selection Tab used by the basic Hyades Launch Configuration
 * @author jcanches
 */
public class TestComponentAndDeploymentTab extends AbstractLaunchConfigurationTab {
	
	protected TreeViewer componentTreeViewer;
	protected ListViewer deploymentListViewer;
	private String mode;
	private String launchConfigurationType;
	
	public TestComponentAndDeploymentTab(String mode, String launchConfigurationType) {
		this.mode = mode;
		this.launchConfigurationType = launchConfigurationType;
	}
	
	public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NULL);
        GridLayout gl = new GridLayout();
        gl.numColumns = 2;
        gl.horizontalSpacing = 25;
        composite.setLayout(gl);
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        composite.setFont(parent.getFont());
        
        Label lblTest = new Label(composite, SWT.NONE);
        GridData gdLblTest = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);       
        lblTest.setLayoutData(gdLblTest);
        lblTest.setText(TestUIPlugin.getString("TestComponentTab.selectTestComponentPrompt"));      //$NON-NLS-1$
        
        Label lblDeployment = new Label(composite, SWT.NONE);
        GridData gdLblDeployment = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);       
        lblDeployment.setLayoutData(gdLblDeployment);
        lblDeployment.setText(TestUIPlugin.getString("TestComponentTab.selectDeploymentPrompt"));         //$NON-NLS-1$

        createTestComponentSelector(composite);
        
        createDeploymentSelector(composite);

        setControl(composite);
	}
	
    private void createTestComponentSelector(Composite composite) {
    	GridData grid = new GridData(GridData.FILL_BOTH);
		grid.widthHint	= 250 ;
		grid.heightHint	= 150;		

    	componentTreeViewer = new TreeViewer(composite);
    	componentTreeViewer.getControl().setLayoutData(grid);
    	
    	TestComponentProvider provider = new TestComponentProvider(mode, launchConfigurationType);
        componentTreeViewer.setContentProvider(provider);
        componentTreeViewer.setLabelProvider(provider);
    	componentTreeViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
    	
    	componentTreeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				TPFTestComponent deploymentInput = fillDeploymentList(getSelectedTestComponent());
				TPFDeployment defaultDeployment;
				if (deploymentInput == null) {
					defaultDeployment = null;
				} else {
					IDeploymentsProvider dprovider = LaunchConfigurationExtensionsManager.getInstance().getDeploymentsProvider(deploymentInput);
					defaultDeployment = dprovider.getDefaultDeployment(deploymentInput);
				}
				ISelection selection;
				if (defaultDeployment != null) {
					selection = new StructuredSelection(defaultDeployment);
				} else {
					selection  = new StructuredSelection();
				}
				deploymentListViewer.setSelection(selection);
				updateLaunchConfigurationDialog();
			}
		});
    }
    
    private void createDeploymentSelector(Composite composite){
    	GridData grid = new GridData(GridData.FILL_BOTH);
		grid.widthHint	= 250 ;
		grid.heightHint	= 150;
		
    	deploymentListViewer = new ListViewer(composite);
    	deploymentListViewer.getControl().setLayoutData(grid);
    	
    	DeploymentProvider provider = new DeploymentProvider();
    	deploymentListViewer.setContentProvider(provider);
    	deploymentListViewer.setLabelProvider(provider);
    	deploymentListViewer.setComparer(provider);
    	deploymentListViewer.setInput(null);
    	
    	deploymentListViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				updateLaunchConfigurationDialog();
			}
		});
    }
    
    protected TPFTestComponent fillDeploymentList(TPFTestComponent component) {
		TestComponentAndDeploymentTab.this.deploymentListViewer.setInput(component);
		return component;
    }
	
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
        TestComponentLaunchConfigurationFacade.setTestComponent(configuration, getContext());
	}
	
	protected TPFTestComponent getSelectedTestComponent() {
		ISelection selection = componentTreeViewer.getSelection();
		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {
			Object obj = ((IStructuredSelection)selection).getFirstElement();
			if (obj instanceof TPFTestComponent) {
				return (TPFTestComponent)obj;
			}
		}
		return null;
	}
	
	protected TPFDeployment getSelectedDeployment() {
		ISelection selection = deploymentListViewer.getSelection();
		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {
			Object obj = ((IStructuredSelection)selection).getFirstElement();
			if (obj instanceof TPFDeployment) {
				return (TPFDeployment)obj;
			}
		}
		return null;
	}
	
	private void setDeploymentSelection(TPFDeployment deployment) {
		// Because the list viewer's setSelection does not use the list viewer's comparer
		// for finding the element to select, we have to do this ourselves.
		List list = deploymentListViewer.getList();
		list.deselectAll();
		if (deployment == null) return;
		int n = list.getItemCount();
		IElementComparer comparer = deploymentListViewer.getComparer();
		for (int i = 0; i < n; i++) {
			if (comparer.equals(deployment, deploymentListViewer.getElementAt(i))) {
				list.select(i);
				return;
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	public void initializeFrom(ILaunchConfiguration configuration) {
        try {
        	TPFTestComponent testComponent = TestComponentLaunchConfigurationFacade.getTestComponent(configuration, EMFUtil.getResourceSet());
        	if (testComponent != null) {
        		componentTreeViewer.setSelection(new StructuredSelection(testComponent), /*reveal*/true);
        		fillDeploymentList(testComponent);
        		TPFDeployment depl = DeploymentLaunchConfigurationFacade.getDeployment(configuration, EMFUtil.getResourceSet());
        		setDeploymentSelection(depl);
        	} else {
        		componentTreeViewer.setSelection(new StructuredSelection());
        		deploymentListViewer.setSelection(new StructuredSelection());
        	}
        } catch (CoreException e) {
            TestUIPlugin.logError(e);
        }
	}
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
        System.out.println("Tab.performApply called"); //$NON-NLS-1$

        TPFTestComponent testComponent = this.getSelectedTestComponent();
        TestComponentLaunchConfigurationFacade.setTestComponent(configuration, testComponent);
        
        TPFDeployment depl = this.getSelectedDeployment();
        DeploymentLaunchConfigurationFacade.setDeployment(configuration, depl);
	}

	public String getName() {
		return TestUIPlugin.getString("TestComponentTab.tabName"); //$NON-NLS-1$
	}
	
	public Image getImage() {
		return TestUIImages.INSTANCE.getImage(TestUIImages.IMG_LCONFIG_TCOMPONENT_TAB);
	}

    /*
     * @see org.eclipse.debug.ui.ILaunchConfigurationTab#isValid(org.eclipse.debug.core.ILaunchConfiguration)
     */
    public boolean isValid(ILaunchConfiguration launchConfig) {
        //- Sets the state of the Run button
        setErrorMessage(null);
        TPFTestComponent testComponent;
        try {
			testComponent = TestComponentLaunchConfigurationFacade.getTestComponent(launchConfig, EMFUtil.getResourceSet());
		} catch (CoreException e) {
			testComponent = null;
		}
        if (testComponent == null) {
        	setErrorMessage(TestUIPlugin.getString("_ERROR_TestComponentTab.noComponentError")); //$NON-NLS-1$
        	return false;
        }
        TPFDeployment depl;
        try {
			depl = DeploymentLaunchConfigurationFacade.getDeployment(launchConfig, EMFUtil.getResourceSet());
		} catch (CoreException e1) {
			depl = null;
		}
        if (depl == null) {
        	setErrorMessage(TestUIPlugin.getString("_ERROR_TestComponentTab.noDeploymentError")); //$NON-NLS-1$
        	return false;
        }
        // Invoke an extension point's validator
        ITestLaunchConfigurationValidator validator = LaunchConfigurationExtensionsManager.getInstance().getLaunchConfigurationValidator(testComponent);
        if (validator != null) {
        	ITestLaunchConfigurationValidator.Diagnostic diag = validator.validate(launchConfig, EMFUtil.getResourceSet());
        	if (diag != null) {
        		setErrorMessage(diag.getMessage());
        		return diag.getSeverity() != ITestLaunchConfigurationValidator.Diagnostic.ERROR;
        	}
        }
        return true;
    }

    /**
     * @return the first test element from the workbench selection
     */
    public static TPFTestComponent getContext() {
        try {
            IWorkbenchPage page = TestUIPlugin.getInstance().getWorkbench().getActiveWorkbenchWindow().getActivePage();
            if (page != null) {
                ISelection selection = page.getSelection();
                if (selection instanceof IStructuredSelection) {
                    IStructuredSelection ss = (IStructuredSelection) selection;
                    Object obj = ss.getFirstElement();
                    if (obj != null) {
                        if (obj instanceof TPFTestComponent) {
                            return (TPFTestComponent)obj;
                        }
                        if (obj instanceof IFile) {
                            IFile file = (IFile) obj;
                            if (ICommonConstants.TEST_COMPONENT_EXTENSION.equals(file.getFileExtension())) {
                            	return loadTestComponent(file);
                            }
                        }
                    }
                }
            }
        } catch (Throwable t) {
            TestUIPlugin.logError(t);
        }
        return null;
    }

	private static TPFTestComponent loadTestComponent(IFile resource) {
		EObject[] eobjs = EMFUtil.load(null, resource);
		for (int i = 0; i < eobjs.length; i++) {
			if (eobjs[i] instanceof TPFTestComponent) {
				return (TPFTestComponent)eobjs[i];
			}
		}
		return null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#activated(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void activated(ILaunchConfigurationWorkingCopy workingCopy) {
		// This tab's content is not dependent on other tabs' content. Nothing
		// to do
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#deactivated(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void deactivated(ILaunchConfigurationWorkingCopy workingCopy) {
		// Everything is applied as soon as selected. Nothing to do
	}
	
}
