 /*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.model.ui;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;

/**
 * @author marcelop
 * @since 0.3.0
 */
public class EMFContentProvider 
implements ITreeContentProvider, IDisposable
{
	public static class ElementDescription
	{
		private String fileExtension;
		private List eClassifiers;
		
		public ElementDescription(String fileExtension)
		{
			this.fileExtension = fileExtension;
		}
		
		public String getFileExtension()
		{
			return fileExtension;
		}

		public List getEClassifiers()
		{
			if(eClassifiers == null)
				eClassifiers = new UniqueEList();
			return eClassifiers;
		}
	}
	
	public static class EnabledChildren
	{
		private EClassifier parentEClassifier;
		private List childrenEStructuralFeatures;
		
		public EnabledChildren(EClassifier parentEClassifier)
		{
			this.parentEClassifier = parentEClassifier;
		}
		
		public EClassifier getParentEClassifier()
		{
			return parentEClassifier;
		}

		public List getChildrenEStructuralFeatures()
		{
			if(childrenEStructuralFeatures == null)
				childrenEStructuralFeatures = new UniqueEList();
			return childrenEStructuralFeatures;
		}
	}
	
	private TreeViewer treeViewer;
	private ResourceSet resourceSet;
	
	private Map elementDescriptionByFileExtension;
	private Map enabledChildrenByParentEClassifier;
	private boolean showEMFResource;
	private boolean showEnabledChildren;
	
	private ArrayList loadedEObjects;

	/**
	 * Constructor for EMFContentProvider
	 */
	public EMFContentProvider(ResourceSet resourceSet)
	{
		this.resourceSet = resourceSet;
		
		elementDescriptionByFileExtension = new HashMap();
		enabledChildrenByParentEClassifier = new HashMap();
		showEMFResource = false;
		
		loadedEObjects = new ArrayList();
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
	 */
	public void dispose()
	{
		treeViewer = null;
		if(elementDescriptionByFileExtension != null)
		{
			elementDescriptionByFileExtension.clear();
			elementDescriptionByFileExtension = null;
		}
		if(enabledChildrenByParentEClassifier != null)
		{
			enabledChildrenByParentEClassifier.clear();
			enabledChildrenByParentEClassifier = null;
		}
		resourceSet = null;
		System.gc();
	}
	
	public void addElementDescription(ElementDescription elementDescription)
	{
		elementDescriptionByFileExtension.put(elementDescription.getFileExtension(), elementDescription);
	}

	public void removeElementDescription(ElementDescription elementDescription)
	{
		elementDescriptionByFileExtension.remove(elementDescription.getFileExtension());
	}

	public void addEnabledChildren(EnabledChildren enabledChildren)
	{
		enabledChildrenByParentEClassifier.put(enabledChildren.getParentEClassifier(), enabledChildren);
	}

	public void removeEnabledChildren(EnabledChildren enabledChildren)
	{
		enabledChildrenByParentEClassifier.remove(enabledChildren.getParentEClassifier());
	}

	/**
	 * @return
	 */
	public boolean isShowEMFResource()
	{
		return showEMFResource;
	}

	/**
	 * @param showEMFResource
	 */
	public void setShowEMFResource(boolean showEMFResource)
	{
		this.showEMFResource = showEMFResource;
	}

	/**
	 * @return
	 */
	public boolean isShowEnabledChildren()
	{
		return showEnabledChildren;
	}
	
	private boolean isChildrenEnabled(EObject parentObject)
	{
		if(isShowEnabledChildren())
		{
			EnabledChildren enabledChildren = (EnabledChildren)enabledChildrenByParentEClassifier.get(parentObject.eClass());
			if(enabledChildren != null)
				return true;
		}
		
		return false;
	}

	/**
	 * @param showEnabledChildren
	 */
	public void setShowEnabledChildren(boolean showEnabledChildren)
	{
		this.showEnabledChildren = showEnabledChildren;
	}

	/**
	 * Returns the tree viewer this provider is associated with.
	 * @return TreeViewer
	 */
	protected TreeViewer getTreeViewer()
	{
		return treeViewer;
	}

	/**
	 * The resource set that is presente by this provider.
	 * @return ResourceSet
	 */
	protected ResourceSet getResourceSet()
	{
		return resourceSet;
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
	{
		treeViewer = (TreeViewer)viewer;
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
	 */
	public Object getParent(Object element)
	{
		if(element instanceof IResource)
			return ((IResource)element).getParent();
			
		else if(element instanceof Resource)
		{
			IFile file = EMFUtil.getWorkspaceFile((Resource)element);
			if(file != null)
				return file.getParent();	
		}
		
		else if(element instanceof EObject)
		{
			EObject eObject = (EObject)element;
			if(eObject.eContainer() != null)
				return eObject.eContainer();
				
			if(isShowEMFResource())
				return eObject.eResource();
				
			IFile file = EMFUtil.getWorkspaceFile(eObject);
			if(file != null)
				return file.getParent();
		}
		
		return null;
	}

	/**
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement)
	{
		List list = new ArrayList();
		if(inputElement instanceof IWorkspaceRoot)
			list.addAll(Arrays.asList(((IWorkspaceRoot)inputElement).getProjects()));
		
		list.addAll(NavigatorExtensionUtil.getAllChildren(inputElement, TestNavigator.ID));
		return getChildren(inputElement);
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object element)
	{
		if (NavigatorExtensionUtil.hasChildren(element, TestNavigator.ID))
			return true;
		
		if(element instanceof IContainer)
		{
			IContainer container = (IContainer)element;
			if(!container.exists())
				return false;
			
			IResource[] resources = null;
			try
			{
				resources = container.members();
				if(resources != null && resources.length > 0)
					return true;
				else
					return false;
			}
			catch (CoreException e)
			{
				TestUIPlugin.logError(e);
				return false;
			}
		}
		
		else if (element instanceof Resource)
			return !((Resource)element).getContents().isEmpty();
			
		else if(element instanceof EObject)
		{
			if(isChildrenEnabled((EObject)element))
				return true;
		}
		
		return false;
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object parentElement)
	{
		List children = new ArrayList();
		if(parentElement instanceof IContainer)
		{
			IResource[] resources = null;
			try
			{
				resources = ((IContainer)parentElement).members();
			}
			catch (CoreException e)
			{
				TestUIPlugin.logError(e);
				children.addAll(NavigatorExtensionUtil.getAllChildren(parentElement, TestNavigator.ID));
				return children.toArray();
			}
			
			for(int i = 0, maxi = resources.length; i < maxi; i++)
			{
				if(resources[i].getType() != IResource.FILE)
				{
					children.add(resources[i]);
				}
				else
				{
					String fileExtension = resources[i].getFileExtension();
					if(fileExtension == null)
						continue;
						
					ElementDescription elementDescription = (ElementDescription)elementDescriptionByFileExtension.get(fileExtension);
					if(elementDescription != null)
					{
						EObject[] eObjects = EMFUtil.load(resourceSet, (IFile)resources[i]);	
						loadedEObjects.addAll(Arrays.asList(eObjects));
						for (Iterator iter = elementDescription.getEClassifiers().iterator(); iter.hasNext();)
						{
							Collection validObjects = EcoreUtil.getObjectsByType(Arrays.asList(eObjects), (EClassifier)iter.next());
							if(isShowEMFResource())
							{
								for (Iterator iter2 = validObjects.iterator(); iter2.hasNext();)
									children.add(((EObject)iter2.next()).eResource());
							}
							else
							{
								children.addAll(validObjects);
							}
						}
					}
				}
			}
		}
		
		else if(parentElement instanceof Resource)
			children.addAll(((Resource)parentElement).getContents());
		
		else if(parentElement instanceof EObject)
		{
			if(isShowEnabledChildren())
			{
				EObject eObject = (EObject)parentElement;
				EnabledChildren enabledChildren = (EnabledChildren)enabledChildrenByParentEClassifier.get(eObject.eClass());
				if(enabledChildren != null)
				{
					for (Iterator i = enabledChildren.getChildrenEStructuralFeatures().iterator(); i.hasNext();)
					{
						Object child = eObject.eGet((EStructuralFeature)i.next(), false);
						if(child != null)
						{
							if(child instanceof Collection)
								children.addAll((Collection)child);
							else
								children.add(child);	
						}
					}
				}
			}
		}
		
		children.addAll(NavigatorExtensionUtil.getAllChildren(parentElement, TestNavigator.ID));
		return children.toArray();
	}
	
	public List getLoadedEObjects()
	{
		return loadedEObjects;
	}
}
