/*******************************************************************************
* Copyright (c) 2003 Hyades project.
* All rights reserved. This program and the accompanying materials 
* are made available under the terms of the Common Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/cpl-v10.html
* 
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.ui.IHyadesTestNavigatorConverter;
import org.eclipse.hyades.test.ui.IHyadesTestNavigatorFilter;
import org.eclipse.hyades.test.ui.IHyadesTestNavigatorProvider;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.editor.ExecutionEditorPart;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.internal.model.ResourceUtil.ResourceAdaptable;
import org.eclipse.hyades.test.ui.internal.model.ui.TestNavigatorProvider;
import org.eclipse.hyades.test.ui.internal.navigator.action.TestNavigatorActionGroup;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyManager;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyNode;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.hyades.test.ui.navigator.CMNNamedElementProxyNode;
import org.eclipse.hyades.test.ui.navigator.EObjectProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.navigator.NavigatorViewerSorter;
import org.eclipse.hyades.ui.internal.navigator.TreeNavigator;
import org.eclipse.hyades.ui.internal.navigator.TreeNavigatorFrameSource;
import org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater;
import org.eclipse.hyades.ui.internal.provider.ResourceChangeUpdaterProvider;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.ILabelDecorator;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionGroup;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.plugin.AbstractUIPlugin;

	/**
	 * Concreate implementation of the 
	 * {@link org.eclipse.hyades.ui.internal.navigator.Navigator} for the Hyades Test 
	 * objects
	 * 
	 * @author marcelop
	 * @since 0.0.1 
	 */
	public class TestNavigator extends TreeNavigator implements ISelectionListener {
		public static final String ID = "org.eclipse.hyades.test.ui.TestNavigator"; //$NON-NLS-1$

		/**
		 * Map with the instances of this class by the workbench window.  It is
		 * necessary to keep this cache because adpaters and content providers
		 * may need to check properties of a test navigator - such as if the
		 * folders are visible. 
		 */
		private static Map testNavigatorByWorkbenchWindow;

		/**
		 * Adds a test navigator to the <code>testNavigatorByWorkbenchWindow</code> 
		 * map.
		 * @param testNavigator
		 */
		private static void addTestNavigator(TestNavigator testNavigator) {
			if (testNavigatorByWorkbenchWindow == null)
				testNavigatorByWorkbenchWindow = new HashMap();
			testNavigatorByWorkbenchWindow.put(testNavigator.getSite().getWorkbenchWindow(), testNavigator);
		}

		/**
		 * Remvoes a test navigator to the 
		 * <code>testNavigatorByWorkbenchWindow</code> map.
		 * @param testNavigator
		 */
		private static void removeTestNavigator(TestNavigator testNavigator) {
			if (testNavigatorByWorkbenchWindow != null) {
				testNavigatorByWorkbenchWindow.remove(testNavigator.getSite().getWorkbenchWindow());
				if (testNavigatorByWorkbenchWindow.isEmpty())
					testNavigatorByWorkbenchWindow = null;
			}
		}

		/**
		 * Returns the test navigator opened in a given workbench window or 
		 * <code>null</code> if there isn't any.
		 * @param workbenchWindow
		 * @return TestNavigator
		 */
		public static TestNavigator getTestNavigator(IWorkbenchWindow workbenchWindow) {
			if ((workbenchWindow != null) && (testNavigatorByWorkbenchWindow != null))
				return (TestNavigator) testNavigatorByWorkbenchWindow.get(workbenchWindow);
			return null;
		}

		/*
		 * SET_* constants are used in the IDialogSettings
		 */
		protected final static String SET_SHOW_FOLDERS = "SHOW_FOLDERS"; //$NON-NLS-1$
		protected final static String SET_SHOW_EOBJECT_CHILDREN = "SHOW_EOBJECT_CHILDREN"; //$NON-NLS-1$
		protected final static String SET_SORTER_CRITERIA = "SORTER_CRITERIA"; //$NON-NLS-1$

		protected final static int CP_FOLDER = 0;
		protected final static int CP_LOGICAL = 1;

		private List rootLogicalFolders;
		private boolean showingFolders = true;
		private boolean showingEObjectChildren = true;

		private ResourceChangeUpdaterProvider resourceChangeUpdaterProvider;
		//- resourceSet should be managed by the model ...
		//		private ResourceSet resourceSet;
		private TestNavigatorProvider provider;

		/**
		 * Constructor for TestNavigator
		 */
		public TestNavigator() {
			super(ID);
			//		initResourceSet();
			initExtension();
		}

		/**
		 * 
		 */
		private void initExtension() {
			provider = new TestNavigatorProvider();

			String extName;
			IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorFile"); //$NON-NLS-1$
			if (extPoint != null) {
				IConfigurationElement[] members = extPoint.getConfigurationElements();
				for (int i = 0; i < members.length; i++) {
					if (members[i].getName() != null) {
						//- get extension of files we want to make visible
						extName = members[i].getAttribute("targetExtension"); //$NON-NLS-1$
						//- save class in array to delegate specific process
						try {
							IHyadesTestNavigatorProvider delegatedProvider = null;
							IHyadesTestNavigatorConverter converter = null;
							IConfigurationElement [] prov = members[i].getChildren("provider"); //$NON-NLS-1$
							if (prov.length != 0) {
								delegatedProvider = (IHyadesTestNavigatorProvider) prov[0].createExecutableExtension("class"); //$NON-NLS-1$
							}
							IConfigurationElement [] conv = members[i].getChildren("converter"); //$NON-NLS-1$
							if (conv.length != 0) {
								converter = (IHyadesTestNavigatorConverter) conv[0].createExecutableExtension("class"); //$NON-NLS-1$
							}
							provider.addVisibleFile(extName, delegatedProvider, converter);
						} catch (CoreException e) {
							TestUIPlugin.logError(e);
						}
					}
				}
			}
			extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorObject"); //$NON-NLS-1$
			if (extPoint != null) {
				IConfigurationElement[] members = extPoint.getConfigurationElements();
				String objectName;
				for (int i = 0; i < members.length; i++) {
					if (members[i].getName() != null) {
						//- get extension of files we want to make visible
						objectName = members[i].getAttribute("targetObject"); //$NON-NLS-1$
						//- save class in array to delegate specific process
						try {
							IHyadesTestNavigatorProvider delegatedProvider = (IHyadesTestNavigatorProvider) members[i].createExecutableExtension("provider"); //$NON-NLS-1$
							provider.addVisibleObject(objectName, delegatedProvider);
						} catch (CoreException e) {
							TestUIPlugin.logError(e);
						}
					}
				}
			}
			extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorFilter"); //$NON-NLS-1$
			if (extPoint != null) {
				IConfigurationElement[] members = extPoint.getConfigurationElements();
				for (int i = 0; i < members.length; i++) {
					if (members[i].getName() != null) {
						//- save filter to delegate specific process
						try {
							IHyadesTestNavigatorFilter filter = (IHyadesTestNavigatorFilter) members[i].createExecutableExtension("class"); //$NON-NLS-1$
							provider.addFilter(filter);
						} catch (CoreException e) {
							TestUIPlugin.logError(e);
						}
					}
				}
			}
		}

		/**
		 * @see org.eclipse.ui.IWorkbenchPart#dispose()
		 */
		public void dispose() {
			if (rootLogicalFolders != null)
				rootLogicalFolders.clear();
			removeTestNavigator(this);

			ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceChangeUpdaterProvider);
			resourceChangeUpdaterProvider.getResourceChangeUpdater().dispose();
			resourceChangeUpdaterProvider.dispose();
			resourceChangeUpdaterProvider = null;

			provider.dispose();
			provider = null;

			//			EMFUtil.removeAdapters(resourceSet.getAllContents());
			//			resourceSet.getResources().clear();
			//			resourceSet.eAdapters().clear();
			//			resourceSet = null;

			super.dispose();
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getStoreSection()
		 */
		public String getStoreSection() {
			return "TestNavigator"; //$NON-NLS-1$
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#initSettings()
		 */
		protected void initSettings() {
			setShowingFolders(true);
			setShowingEObjectChildren(false);
		}

		//	/**
		//	 * Initializes the resource set.
		//	 */
		//	protected void initResourceSet()
		//	{
		//		resourceSet = new ResourceSetImpl();
		//		modelContentProvider = new EMFContentProvider(getResourceSet());
		//		
		//		EMFContentProvider.ElementDescription elementDescription = new EMFContentProvider.ElementDescription(ICommonConstants.TEST_SUITE_FILE_EXTENSION);
		////		elementDescription.getEClassifiers().add(Common_TestprofilePackage.eINSTANCE.getTPFTestSuite());
		//		elementDescription.getEClassifiers().add("TPFTestSuite");
		//		modelContentProvider.addElementDescription(elementDescription); 
		//		
		//		EMFContentProvider.EnabledChildren enabledChildren = new EMFContentProvider.EnabledChildren(Common_TestprofilePackage.eINSTANCE.getTPFTestSuite());
		//		enabledChildren.getChildrenEStructuralFeatures().add(Common_TestprofilePackage.eINSTANCE.getTPFTestSuite_TestCases());
		//		modelContentProvider.addEnabledChildren(enabledChildren);
		//		
		//		elementDescription = new EMFContentProvider.ElementDescription(ICommonConstants.DEPLOYMENT_FILE_EXTENSION);
		////		elementDescription.getEClassifiers().add(Common_TestprofilePackage.eINSTANCE.getTPFDeployment());
		//		elementDescription.getEClassifiers().add("TPFDeployment");
		//		modelContentProvider.addElementDescription(elementDescription);		
		//	}

		/**
		 * Returns this navigator's resource set.
		 * @return ResourceSet
		 */
		//		public ResourceSet getResourceSet() {
		//			return resourceSet;
		//		}

		/**
		 * @see org.eclipse.ui.IViewPart#init(org.eclipse.ui.IViewSite, org.eclipse.ui.IMemento)
		 */
		public void init(IViewSite site, IMemento memento) throws PartInitException {
			super.init(site, memento);
			addTestNavigator(this);
		}

		/**
		 * Returns the file associated to a given object that is located in this
		 * navigator.
		 * @param object
		 * @return IFile
		 */
		public IFile getFile(Object object) {
			if (object instanceof IFile)
				return (IFile) object;

			if(object instanceof FileProxyNode) {
				FileProxyNode proxy = (FileProxyNode)object;
				return proxy.getFile();
			}
			
			if (object instanceof EObjectProxyNode) {
				EObjectProxyNode proxy = (EObjectProxyNode) object;
				if (proxy.getParent() instanceof IContainer) {
					//- this proxy node is a root object (place instead of a file node)
					URI uri = proxy.getOriginatorURI();
					String path = uri.trimFragment().toString().substring("platform:/resource".length()); //$NON-NLS-1$
					Path filePath = new Path(path);
					return (IFile)ResourcesPlugin.getWorkspace().getRoot().findMember(filePath);
				}
			}
			if (object instanceof Resource)
				return EMFUtil.getWorkspaceFile((Resource) object);

			if (object instanceof EObject) {
				EObject eObject = (EObject) object;
				if (eObject.eContainer() != null)
					return null;

				Resource resource = eObject.eResource();
				if (resource == null)
					return null;

				//				if (getResourceSet().getResource(resource.getURI(), false) == null)
				//					return null;

				return EMFUtil.getWorkspaceFile(resource);
			}

			if (object instanceof IAdaptable) {
				IFile file = (IFile) ((IAdaptable) object).getAdapter(IFile.class);
				if (file != null)
					return file;
			}

			return null;
		}

		/**
		 * Sets whether the resource navigator should monitor resource
		 * changes.
		 * @param enabled
		 */
		public void setMonitoringResourceChange(boolean enabled) {
			if (resourceChangeUpdaterProvider != null)
				resourceChangeUpdaterProvider.getResourceChangeUpdater().setActive(enabled);
		}

		/**
		 * Returns whether the test navigator is monitoring resource changes
		 */
		public boolean isMonitoringResourceChange() {
			if (resourceChangeUpdaterProvider != null)
				resourceChangeUpdaterProvider.getResourceChangeUpdater().isActive();
			return false;
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#loadSettings()
		 */
		protected void loadSettings() {
			super.loadSettings();
			//			setShowingFolders(getSettings().getBoolean(SET_SHOW_FOLDERS));
			setShowingEObjectChildren(getSettings().getBoolean(SET_SHOW_EOBJECT_CHILDREN));
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.TreeNavigator#adjustTreeViewer(org.eclipse.jface.viewers.TreeViewer)
		 */
		protected void adjustTreeViewer(TreeViewer treeViewer) {
			int criteria = NavigatorViewerSorter.NAME;
			try {
				criteria = getSettings().getInt(SET_SORTER_CRITERIA);
			} catch (NumberFormatException e) {
			}
			NavigatorViewerSorter viewerSorter = new NavigatorViewerSorter(this, criteria) {
				protected String getValue(Viewer viewer, Object e) {
					String value = super.getValue(viewer, e);
					if (e instanceof EObject) {
						EObject eObject = (EObject) e;
						if (eObject.eContainer() != null)
							return ""; //$NON-NLS-1$

						if (getCriteria() == NavigatorViewerSorter.TYPE) {
							if (eObject instanceof TPFTest) {
								String type = ((TPFTest) eObject).getType();
								if (type != null)
									value = type + value;
								else
									value = "zZz" + type; //$NON-NLS-1$
							}
							value = eObject.eClass().getName() + value;
						}
					}

					return value;
				}
			};
			treeViewer.setSorter(viewerSorter);

			ViewerFilter viewerFilter = new ViewerFilter() {
				public boolean select(Viewer viewer, Object parentElement, Object element) {
					if (!isShowingEObjectChildren())
						return (!(parentElement instanceof EObject));

					return true;
				}
			};
			treeViewer.addFilter(viewerFilter);

			resourceChangeUpdaterProvider = new ResourceChangeUpdaterProvider.UIUpdaterProvider();
			ResourcesPlugin.getWorkspace().addResourceChangeListener(resourceChangeUpdaterProvider, IResourceChangeEvent.POST_CHANGE);
			IResourceChangeUpdater resourceChangeUpdater = new TestResourceChangeUpdater(this);
			resourceChangeUpdaterProvider.setResourceChangeUpdater(resourceChangeUpdater);

			treeViewer.setContentProvider(provider);
			ILabelDecorator decorator = PlatformUI.getWorkbench().getDecoratorManager().getLabelDecorator();
			//		treeViewer.setLabelProvider(new DecoratingLabelProvider(new WorkbenchAdapterLabelProvider(TestWorkbenchAdapter.class), decorator));
			treeViewer.setLabelProvider(new DecoratingLabelProvider(provider, decorator));
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#editorActivated(org.eclipse.ui.IEditorPart)
		 */
		protected boolean editorActivated(IEditorPart editor) {
			if (getViewer() == null)
				return false;

			IEditorInput input = editor.getEditorInput();
			if (input instanceof IFileEditorInput) {
				IFileEditorInput fileInput = (IFileEditorInput) input;
				IFile file = fileInput.getFile();

				Resource resource = EMFUtil.getResource(null, file);
				if ((resource != null) && (!resource.getContents().isEmpty())) {
					IStructuredSelection structuredSelection = getStructuredSelection();
					if (structuredSelection.size() == 1) {
						Object currentSelection = structuredSelection.getFirstElement();
						if (currentSelection instanceof EObject) {
							if (((EObject) currentSelection).eResource() == resource)
								return true;
						}
					}

					selectReveal(new StructuredSelection(resource.getContents().get(0)));
					return true;
				}
			}

			return false;
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#linkToEditor(org.eclipse.jface.viewers.IStructuredSelection)
		 */
		protected void linkToEditor(IStructuredSelection structuredSelection) {
			IFile file = getFile(structuredSelection.getFirstElement());
			if (file != null) {
				IWorkbenchPage page = getSite().getPage();

				IEditorPart editor = page.findEditor(new FileEditorInput(file));
				if (editor != null) {
					page.bringToTop(editor);
					return;
				}
			}
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getPlugin()
		 */
		public AbstractUIPlugin getPlugin() {
			return TestUIPlugin.getInstance();
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#createViewerInput()
		 */
		protected Object createViewerInput() {
			return ResourcesPlugin.getWorkspace().getRoot();
			//return getResourceSet();
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#createActions()
		 */
		protected void createActions() {
			setActionGroup(new TestNavigatorActionGroup(this));
			//			String id = IWorkbenchActionConstants.REFRESH;
			//			RetargetAction refreshAction = new RetargetAction(id, WorkbenchMessages.getString("Workbench." + id));
			//			getViewSite().getActionBars().setGlobalActionHandler(id, refreshAction);
		}

		/**
		 * Returns whether the navigator is showing projects and folders.
		 * @return <code>true</code> if is showing project and folders, 
		 * <code>false</code> if not.  
		 */
		public boolean isShowingFolders() {
			return showingFolders;
		}

		/**
		 * Sets whether this navigator's is showing projects and folders.
		 * @param enabled <code>true</code> to enable, <code>false</code> to disable
		 */
		public void setShowingFolders(boolean enabled) {
			showingFolders = enabled;

			// remember the last setting in the dialog settings		
			getSettings().put(SET_SHOW_FOLDERS, showingFolders);
			if (getViewer() == null)
				return;

			getViewer().refresh();
		}

		/**
		 * Returns whether the navigator is showing the children of the 
		 * emf objects.
		 * @return <code>true</code> if is showing the children, 
		 * <code>false</code> if not.  
		 */
		public boolean isShowingEObjectChildren() {
			return showingEObjectChildren;
		}

		/**
		 * Sets whether this navigator's is showing the children of the EMF objects.
		 * @param enabled <code>true</code> to enable, <code>false</code> to disable
		 */
		public void setShowingEObjectChildren(boolean enabled) {
			//			showingEObjectChildren = enabled;
			//			modelContentProvider.setShowEnabledChildren(enabled);
			//
			//			// remember the last setting in the dialog settings		
			//			getSettings().put(SET_SHOW_EOBJECT_CHILDREN, showingEObjectChildren);
			//			if (getViewer() == null)
			//				return;
			//
			//			getViewer().refresh();
			//			selectionChanged(new SelectionChangedEvent(getViewer(), StructuredSelection.EMPTY));
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getStatusLineMessage(org.eclipse.jface.viewers.IStructuredSelection)
		 */
		protected String getStatusLineMessage(IStructuredSelection structuredSelection) {
			if (structuredSelection.size() == 1) {
				Object selection = structuredSelection.getFirstElement();
				if (selection instanceof IResource)
					return ((IResource) selection).getFullPath().toString();

				if (selection instanceof LogicalFolder) {
					LogicalFolder logicalFolder = (LogicalFolder) selection;
					if (logicalFolder.getChildren().size() == 0)
						return logicalFolder.getName();
					return TestUIPlugin.getString("STS_LNE_LOG_FLD",  //$NON-NLS-1$
					        new String[] { logicalFolder.getName(), 
					        Integer.toString(logicalFolder.getChildren().size())});
				}

				if (selection instanceof CMNNamedElement) {
					CMNNamedElement namedElement = (CMNNamedElement) selection;
					String label = ((ILabelProvider) getViewer().getLabelProvider()).getText(namedElement);
					IFile file = EMFUtil.getWorkspaceFile(namedElement);
					String filePath = ""; //$NON-NLS-1$
					if (file != null)
						filePath = file.getFullPath().toString();
					return TestUIPlugin.getString("STS_LNE_FILE_ELEM", new String[] { label, filePath }); //$NON-NLS-1$
				}
				
				if (selection instanceof CMNNamedElementProxyNode) {
					CMNNamedElementProxyNode proxy = (CMNNamedElementProxyNode) selection;
					return TestUIPlugin.getString("STS_LNE_FILE_ELEM", new String[] { proxy.getText(), proxy.getUnderlyingResource().getFullPath().toString() }); //$NON-NLS-1$
				}
				if (selection instanceof FileProxyNode) {
					FileProxyNode proxy = (FileProxyNode) selection;
					return TestUIPlugin.getString("STS_LNE_FILE_ELEM", new String[] { proxy.getText(), proxy.getUnderlyingResource().getFullPath().toString() }); //$NON-NLS-1$				
				}

				if (selection instanceof Resource) {
					Resource resource = (Resource) selection;
					String label = ((ILabelProvider) getViewer().getLabelProvider()).getText(resource);
					IFile file = EMFUtil.getWorkspaceFile(resource);
					String filePath = ""; //$NON-NLS-1$
					if (file != null)
						filePath = file.getFullPath().toString();
					return TestUIPlugin.getString("STS_LNE_FILE_ELEM", new String[] { label, filePath }); //$NON-NLS-1$
				}

			}

			return super.getStatusLineMessage(structuredSelection);
		}

		/**
		 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
		 */
		public void doubleClick(DoubleClickEvent event) {
			if (event.getViewer() == getTreeViewer()) {
				IStructuredSelection selection = (IStructuredSelection) event.getSelection();
				Object element = selection.getFirstElement();
				if (element instanceof IContainer) {
					TreeViewer treeViewer = getTreeViewer();
					if (treeViewer.isExpandable(element))
						treeViewer.setExpandedState(element, !treeViewer.getExpandedState(element));
				}
			}
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#convertToViewer(org.eclipse.jface.viewers.ISelection)
		 */
		protected IStructuredSelection convertToViewer(ISelection selection) {
			IStructuredSelection structuredSelection = super.convertToViewer(selection);
			if (structuredSelection.isEmpty())
				return structuredSelection;

			List elements = new ArrayList(structuredSelection.size());
			for (Iterator i = structuredSelection.iterator(); i.hasNext();) {
				Object object = i.next();
				if (object instanceof IFile) {
					EObject[] eObjects = EMFUtil.getEObjects(null, (IFile) object);
					for (int j = 0, maxj = eObjects.length; j < maxj; j++) {
						if (eObjects[j] instanceof CMNNamedElement)
							elements.add(eObjects[j]);
					}
				} else if (object instanceof EObject) {
					EObject eObject = EMFUtil.getEObject(null, (EObject) object, false);
					if (eObject != null) {
						if (eObject.eContainer() == null) {
							elements.add(eObject);
						} else if (isShowingEObjectChildren()) {
							elements.add(eObject);
						} else {
							elements.add(EcoreUtil.getRootContainer(eObject));
						}
					}
				} else
					elements.add(object);
			}

			return new StructuredSelection(elements);
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.INavigator#handleChange(int)
		 */
		public void handleChange(int type) {
			if (NavigatorViewerSorter.SORTER_TYPE == type) {
				getSettings().put(SET_SORTER_CRITERIA, ((NavigatorViewerSorter) getViewer().getSorter()).getCriteria());
				ActionGroup actionGroup = getActionGroup();
				if (actionGroup != null)
					actionGroup.updateActionBars();
			}
			super.handleChange(type);
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#getInitialViewerInput()
		 */
		public Object getInitialViewerInput() {
			return super.getInitialViewerInput();
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.TreeNavigator#createFrameSource()
		 */
		protected TreeNavigatorFrameSource createFrameSource() {
			return new TestNavigatorFrameSource(this);
		}

		/**
		 * Registers a root logical folder to this test navigator.
		 * @param logicalFolder
		 */
		public void registerRootLogcalFolder(LogicalFolder logicalFolder) {
			if ((logicalFolder != null) && (logicalFolder.getParent() == null)) {
				if (rootLogicalFolders == null)
					rootLogicalFolders = new ArrayList();
				rootLogicalFolders.add(logicalFolder);
			}
		}

		/**
		 * Returns the root logical folder that has the given name or <code>null</code>
		 * if there is no such folder.
		 * @param name
		 */
		public LogicalFolder getRootLogicalFolder(String name) {
			if ((rootLogicalFolders != null) && (name != null)) {
				for (Iterator i = rootLogicalFolders.iterator(); i.hasNext();) {
					LogicalFolder logicalFolder = (LogicalFolder) i.next();
					if (name.equals(logicalFolder.getName()))
						return logicalFolder;
				}
			}
			return null;
		}

		/**
		 * @see org.eclipse.hyades.ui.internal.navigator.Navigator#loadElements(org.eclipse.ui.IMemento)
		 */
		protected List loadElements(IMemento memento) {
			Object[] objects = super.loadElements(memento).toArray();
			List elements = new ArrayList(objects.length);
			for (int i = 0, maxi = objects.length; i < maxi; i++) {
				if (objects[i] instanceof LogicalFolder) {
					LogicalFolder logicalFolder = (LogicalFolder) objects[i];
					Object parent = logicalFolder.getParent();
					if (parent == null) {
						logicalFolder = getRootLogicalFolder(logicalFolder.getName());
						if (logicalFolder != null)
							elements.add(logicalFolder);
					} else if (parent instanceof EObject) {
						EObject eObject = EMFUtil.getEObject(null, (EObject) parent, true);
						if (eObject != null) {
							LogicalFolder newLogicalFolder = new LogicalFolder(eObject, logicalFolder.getName());
							newLogicalFolder.setData(logicalFolder.getData());
							newLogicalFolder.setDescription(logicalFolder.getDescription());

							elements.add(newLogicalFolder);
						}
					} else {
						elements.add(logicalFolder);
					}
				} else if (objects[i] instanceof EObject) {
					EObject eObject = EMFUtil.getEObject(null, (EObject) objects[i], true);
					if (eObject != null)
						elements.add(eObject);
				} else if (objects[i] instanceof ResourceAdaptable) {
					Resource resource = ((ResourceAdaptable) objects[i]).getResource();
					if (resource != null) {
						resource = EMFUtil.getResource(null, resource.getURI(), true);
						if (resource != null)
							elements.add(resource);
					}
				} else {
					elements.add(objects[i]);
				}
			}

			return elements;
		}

		/**
		 * Returns the help context id for this navigator or <code>null</code> if
		 * no help is provided.
		 * @return String
		 */
		protected String getHelpContextId() {
			return TestUIPlugin.getID() + ContextIds.TEST_NAVIGATOR;
		}

		/** Returns the array of file extensions handled by the navigator 
		 * @return array of file extensions
		 * @deprecated
		 */
		public String[] getVisibleFiles() {
			return provider.getVisibleFiles();
		}

        /* (non-Javadoc)
         * @see org.eclipse.hyades.ui.internal.navigator.INavigator#getID()
         */
        public String getID() {
            return ID;
        }
        
        public boolean isVisibleResource(IResource resource) {
        	return provider.isVisibleResource(resource);
        }
        
    	/**
    	 * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
    	 */
        public void selectionChanged(IWorkbenchPart part, ISelection selection)
		{
        	if(part instanceof ExecutionEditorPart)
        	{
        		if(!(selection instanceof IStructuredSelection))
        			return;
        		
        		IStructuredSelection structuredSelection = (IStructuredSelection)selection;
        		if(structuredSelection.size() != 1) 
        			return;
        		
        		Object element = structuredSelection.getFirstElement();
        		if (!(element instanceof TPFExecutionResult))
        			return;
        		
        		TreeViewer treeViewer = getTreeViewer();
        		if ( treeViewer == null )
        			return;
		      	IProxyNode proxy = getCorrespondingPRoxy((EObject)element);
		      	if(proxy != null) {
		      		treeViewer.setSelection(new StructuredSelection(proxy));
		      	}
			}
		}
    	
 		private IProxyNode getCorrespondingPRoxy(EObject object) {
			IFile file = EMFUtil.getWorkspaceFile(object);
			IProxyNode fileProxy = FileProxyManager.getInstance().getProxy(file);
			return FileProxyManager.getInstance().findProxyByID(fileProxy, EMFUtil.getObjectURI(object).fragment());
		}


		/**
		 * @param resource
		 * @deprecated
		 */
		public Object getCorrespondingObjectFromFile(IResource resource) {
			return provider.getCorrespondingObjectFromFile(resource);
		}

		/**
		 * @param resource
		 * @deprecated
		 */
		public void cleanConvertedObjectEntryFor(IResource resource) {
			provider.cleanConvertedObjectEntryFor(resource);
		}

		/** Use the converter of the file (if there is one) to get an object associated to this file
		 *  Add this association in the association map 
		 * @param resource
		 * @return
		 * @deprecated
		 */
		public Object getConvertedObjectFromFile(IResource resource) {
			IHyadesTestNavigatorConverter converter = provider.getConverterFromFile((IFile) resource);
			if (converter != null) {
				Object convertedObject = converter.getObjectFromResource(resource);
				//- update the map
				provider.addCorrespondingObjectFromFile(resource, convertedObject);
				return convertedObject;
			}
			return null;
		}
    	
	}