/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import org.eclipse.core.resources.IFile;
import org.eclipse.hyades.test.ui.navigator.IFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;

/**
 * @author jgout
 * @since 3.2
 */
public class FileProxyManager {
	
	private static FileProxyManager instance;
	private HashMap proxies;
	
	public static FileProxyManager getInstance() {
		if(instance == null) {
			instance = new FileProxyManager();
		}
		return instance;
	}
	
	private FileProxyManager() {
		proxies = new HashMap();
	}
		
	/** Returns the proxy node corresponding to the given file.
	 *  The proxy is stored locally to avoid multiple creation.
	 * @param file the file to convert
	 * @return the IProxyNode matching with the given file or null if no factory can handle this file
	 */
	public IProxyNode getProxy(IFile file) {
		if(isaKnownFile(file)) {
			//- this file has already been converted, thus return the proxy is stored in the map
			return (IProxyNode)proxies.get(file);
		} else {
			return updateProxy(file); 
		}
	}

	/** Returns whether the given file is already in the local proxy data base. 
	 * @param file
	 * @return true if the given file exists as proxy and false otherwise
	 */
	public boolean isaKnownFile(IFile file) {
		return proxies.containsKey(file);
	}

	/** Computes a new proxy using registered file proxy factories.
	 *  This method updates the cached proxy for the given file 
	 * @param file the file from which the proxy is derived.
	 * @return the proxy associated to the given file.
	 */
	public IProxyNode updateProxy(IFile file) {
		IProxyNode proxy = null;
		//- get all factories (FileProxyFactory) for this file element
		ArrayList factories = FileFactoryManager.getInstance().getFactories(file.getFileExtension());
		//- iterate until one can find a correct factory
		for (Iterator it = factories.iterator(); it.hasNext();) {
			IFileProxyFactory factory = (IFileProxyFactory) it.next();
			proxy = factory.create(file);
			if (proxy != null) {
				break;
			}
		}
		//- register this file and its proxy in the map
		if(proxy != null) {
			proxies.put(file, proxy);
		}
		return proxy;
	}
	
	/** Returns the proxy which has the given uid. This search starts from the given proxy node and cross its complete sub tree of proxies.
	 * 
	 * @param proxy root of the proxy tree where the search is done. 
	 * @param uid the uid of the searched proxy.
	 * @return a proxy node which has the given uid or <code>null</code> if there is no such proxy found in the complete given tree of proxy.
	 */
	public IProxyNode findProxyByID(IProxyNode proxy, String uid) {
		if(proxy == null) return proxy;
		//- only for root node that has no fragment part
		if(uid.length() > 0) {
			//- proxy found ?
			if(proxy.getIdentifier().equals(uid)) return proxy;
			else {
				//- search in its children array
				IProxyNode[] children = proxy.getChildren();
				for (int i = 0; i < children.length; i++) {
					proxy = findProxyByID(children[i], uid);
					if(proxy != null) {
						//- found !!
						return proxy;
					}
				}
				//- not found in this sub tree
				return null;
			}
		} else return proxy;
	}
}
