/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.launch.configurations;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.launch.extensions.IExecutionHistoryDefaultsProvider;
import org.eclipse.hyades.test.ui.launch.extensions.IExecutionHistoryDefaultsProvider2;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IWorkbenchPage;

/**
 * This class provides a facade for getting/setting properties regarding the Test Execution
 * History file generated by a Test Execution. These properties are stored into a Launch
 * Configuration.
 * @author jcanches
 */
public class ExecutionHistoryLaunchConfigurationFacade extends AbstractLaunchConfigurationFacade {
	
	private static final String PROPERTY_USER_TEH_NAME = PROPERTY_PREFIX + ".tehName"; //$NON-NLS-1$
	private static final String PROPERTY_USER_TEH_OVERRIDE = PROPERTY_PREFIX + ".tehOverride"; //$NON-NLS-1$
	private static final String PROPERTY_USER_TEH_LOCATION = PROPERTY_PREFIX + ".tehLocation"; //$NON-NLS-1$
    private static final String PROPERTY_DEFAULT_TEH_NAME = PROPERTY_PREFIX + ".defaultName"; //$NON-NLS-1$
    private static final String PROPERTY_DEFAULT_TEH_OVERRIDE_STATE = PROPERTY_PREFIX + ".defaultOverride"; //$NON-NLS-1$
    private static final String PROPERTY_DEFAULT_TEH_LOCATION = PROPERTY_PREFIX + ".defaultLocation"; //$NON-NLS-1$
    
	/**
	 * Get the Test Execution History name from the supplied Launch Configuration.
	 * @param configuration
	 * @return
	 * @throws CoreException
	 */
	public static String getExecutionHistoryName(ILaunchConfiguration configuration) throws CoreException {
		if (isUsingDefaultName(configuration)) {
			return computeDefaultName(configuration);
		} else {
			return getUserName(configuration);
		}
	}
	
	/**
	 * Set the Test Execution History name to the supplied Launch Configuration.
	 * Invoking this method causes the Launch Configuration to not use the
	 * default name any more.
	 * @param configuration
	 * @param name
	 */
	public static void setExecutionHistoryName(ILaunchConfigurationWorkingCopy configuration, String name) {
		setUsingDefaultName(configuration, false);
		setUserName(configuration, name);
	}
	
	/**
	 * Returns whether the produced Test Execution History resource is to be overriden or not.
	 * If it is not, a unique name will be produced, starting with the one specified in the
	 * name property of the configuration.
	 * @param configuration
	 * @return
	 * @throws CoreException
	 */
	public static boolean isExecutionHistoryOverriden(ILaunchConfiguration configuration) throws CoreException {
		if (isUsingDefaultOverrideState(configuration)) {
			return computeDefaultOverrideState(configuration);
		} else {
			return isUserOverriden(configuration);
		}
	}
	
	/**
	 * Set whether the produced Test Execution History resource is to be overriden or not.
	 * Invoking this method causes the Launch Configuration to not use the default
	 * override state any more.
	 * @param configuration
	 * @param value
	 * @see ExecutionHistoryLaunchConfigurationFacade#isExecutionHistoryOverriden(ILaunchConfiguration)
	 */
	public static void setExecutionHistoryOverriden(ILaunchConfigurationWorkingCopy configuration, boolean value) {
		setUsingDefaultOverrideState(configuration, false);
		setUserOverriden(configuration, value);
	}
	
	/**
	 * Get the Test Execution History location from the supplied Launch Configuration.
	 * @param configuration
	 * @return The Project or Folder Location of the Test Execution History file.
	 * @throws CoreException
	 */
	public static IContainer getExecutionHistoryLocation(ILaunchConfiguration configuration) throws CoreException {
		if (isUsingDefaultLocation(configuration)) {
			return computeDefaultLocation(configuration);
		} else {
			return getUserLocation(configuration);
		}
	}
	
	/**
	 * Set the Test Execution History location. The value is stored into the supplied Launch Configuration.
	 * Invoking this method causes the Launch Configuration to not use the default
	 * location any more.
	 * @param configuration
	 * @param location This location must be either a project or a folder resource. The Workspace root
	 * is not acceptable.
	 */
	public static void setExecutionHistoryLocation(ILaunchConfigurationWorkingCopy configuration, IContainer location) {
		if (location != null && location.getType() == IResource.ROOT) {
			throw new IllegalArgumentException("location argument cannot be the Workspace Root"); //$NON-NLS-1$
		}
		setUsingDefaultLocation(configuration, false);
		setUserLocation(configuration, location);
	}
	
	/**
	 * Returns whether the supplied Launch Configuration is using the default
	 * Execution History name.
	 * @param configuration
	 * @return
	 */
    public static boolean isUsingDefaultName(ILaunchConfiguration configuration) {
    	try {
    		return configuration.getAttribute(PROPERTY_DEFAULT_TEH_NAME, true);
    	} catch (CoreException e) {
    		TestUIPlugin.logError(e);
    		return true;
    	}
    }

    /**
     * Returns whether the supplied Launch Configuration is using the default override
     * state.
     * @param configuration
     * @return
     */
    public static boolean isUsingDefaultOverrideState(ILaunchConfiguration configuration) {
    	try {
    		return configuration.getAttribute(PROPERTY_DEFAULT_TEH_OVERRIDE_STATE, true);
    	} catch (CoreException e) {
    		TestUIPlugin.logError(e);
    		return true;
    	}
    }

    /**
     * Returns whether the supplied Launch Configuration is using the default
     * location.
     * @param configuration
     * @return
     */
    public static boolean isUsingDefaultLocation(ILaunchConfiguration configuration) {
    	try {
    		return configuration.getAttribute(PROPERTY_DEFAULT_TEH_LOCATION, true);
    	} catch (CoreException e) {
    		TestUIPlugin.logError(e);
    		return true;
    	}
    }
    
    /**
     * Set whether the supplied Launch Configuration should use the default name.
     * If value is true, any name that was previously set is discarded and any subsequent
     * call to getExecutionHistoryName will return the default name.
     * @param configuration
     * @param value
     */
    public static void setUsingDefaultName(ILaunchConfigurationWorkingCopy configuration, boolean value) {
    	configuration.setAttribute(PROPERTY_DEFAULT_TEH_NAME, value);
    }

    /**
     * Set whether the supplied Launch Configuration should use the default override state.
     * If value is true, any state that was previously set is discarded and any subsequent
     * call to isExecutionHistoryOverriden will return the default override state.
     * @param configuration
     * @param value
     */
    public static void setUsingDefaultOverrideState(ILaunchConfigurationWorkingCopy configuration, boolean value) {
    	configuration.setAttribute(PROPERTY_DEFAULT_TEH_OVERRIDE_STATE, value);
    }

    /**
     * Set whether the supplied Launch Configuration should use the default location.
     * If value is true, any location folder that was previously set is discarded and any subsequent
     * call to getExecutionHistoryLocation will return the default location.
     * @param configuration
     * @param value
     */
    public static void setUsingDefaultLocation(ILaunchConfigurationWorkingCopy configuration, boolean value) {
    	configuration.setAttribute(PROPERTY_DEFAULT_TEH_LOCATION, value);
    }
    
	private static String getUserName(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(PROPERTY_USER_TEH_NAME, (String)null);
	}
	
	private static void setUserName(ILaunchConfigurationWorkingCopy configuration, String name) {
		configuration.setAttribute(PROPERTY_USER_TEH_NAME, name);
	}
	
	private static boolean isUserOverriden(ILaunchConfiguration configuration) throws CoreException {
		return configuration.getAttribute(PROPERTY_USER_TEH_OVERRIDE, false);
	}
	
	private static void setUserOverriden(ILaunchConfigurationWorkingCopy configuration, boolean value) {
		configuration.setAttribute(PROPERTY_USER_TEH_OVERRIDE, value);
	}
	
	private static IContainer getUserLocation(ILaunchConfiguration configuration) throws CoreException {
		return getContainerProperty(configuration, PROPERTY_USER_TEH_LOCATION);
	}
	
	private static void setUserLocation(ILaunchConfigurationWorkingCopy configuration, IContainer location) {
		if (location != null && location.getType() == IResource.ROOT) {
			throw new IllegalArgumentException("location argument cannot be the Workspace Root"); //$NON-NLS-1$
		}
		setContainerProperty(configuration, PROPERTY_USER_TEH_LOCATION, location);
	}
	
	private static IExecutionHistoryDefaultsProvider2 getExecutionHistoryDefaultsProvider2(ILaunchConfiguration configuration) {
		String launchConfigurationType = null;
		try {
			launchConfigurationType = configuration.getType().getIdentifier();
		} catch (CoreException e) {
			TestUIPlugin.logError(e);
		}
		// Search an IExecutionHistoryDefaultsProvider2 for the launch configuration first
		if (launchConfigurationType != null) {
			IExecutionHistoryDefaultsProvider2 defaultsProvider2 = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider2(launchConfigurationType);
			if (defaultsProvider2 != null) {
				return defaultsProvider2;
			}
		}
		return null;
	}
	
	/**
	 * @return
	 */
	private static String computeDefaultName(ILaunchConfiguration configuration) {
		String name = null;
		IExecutionHistoryDefaultsProvider2 defaultsProvider2 = getExecutionHistoryDefaultsProvider2(configuration);
		if (defaultsProvider2 != null) {
			name = defaultsProvider2.getDefaultName(configuration);
		}
		
		if (name == null) {
			TPFTest test = null;
			try {
				test = TestLaunchConfigurationFacade.getTest(configuration, EMFUtil.getResourceSet());
			} catch (CoreException e) {
				TestUIPlugin.logError(e);
			}
			IExecutionHistoryDefaultsProvider defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider(test);
			if (defaultsProvider != null) {
				name = defaultsProvider.getDefaultName(test);
			}
			if (name == null) {
				// Use the DefaultDeploymentsProvider suggestion
				name = LaunchConfigurationExtensionsManager.getInstance().getDefaultExecutionHistoryDefaultsProvider().getDefaultName(test);
			}
		}
		return name;
	}
	
	private static boolean computeDefaultOverrideState(ILaunchConfiguration configuration) {
		IExecutionHistoryDefaultsProvider2 defaultsProvider2 = getExecutionHistoryDefaultsProvider2(configuration);
		if (defaultsProvider2 != null) {
			return defaultsProvider2.getDefaultOverrideState(configuration);
		}
		
		TPFTest test = null;
		try {
			test = TestLaunchConfigurationFacade.getTest(configuration, EMFUtil.getResourceSet());
		} catch (CoreException e) {
			TestUIPlugin.logError(e);
		}
		
		IExecutionHistoryDefaultsProvider defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider(test);
		// The return value is never null
		return defaultsProvider.getDefaultOverrideState(test);
	}
	
	private static IContainer computeDefaultLocation(ILaunchConfiguration configuration) {
		IContainer location = null;
		IExecutionHistoryDefaultsProvider2 defaultsProvider2 = getExecutionHistoryDefaultsProvider2(configuration);
		if (defaultsProvider2 != null) {
			location = defaultsProvider2.getDefaultLocation(configuration);
		}
		
		if (location == null) {
			TPFTest test = null;
			try {
				test = TestLaunchConfigurationFacade.getTest(configuration, EMFUtil.getResourceSet());
			} catch (CoreException e) {
				TestUIPlugin.logError(e);
			}
			
			IExecutionHistoryDefaultsProvider defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider(test);
			// The return value (above) is never null
			location = defaultsProvider.getDefaultLocation(test);
			if (location == null) {
				// Use the DefaultDeploymentsProvider suggestion
				location = LaunchConfigurationExtensionsManager.getInstance().getDefaultExecutionHistoryDefaultsProvider().getDefaultLocation(test);
			}
			if (location == null) {
				// Use the location derived from the current selection in the Workbench
				location = getContextLocation();
			}
		}
	
		return location;
	}

    /**
     * @return the first test element from the workbench selection
     */
    private static IContainer getContextLocation() {
        try {
            IWorkbenchPage page = TestUIPlugin.getInstance().getWorkbench().getActiveWorkbenchWindow().getActivePage();
            if (page != null) {
                ISelection selection = page.getSelection();
                if (selection instanceof IStructuredSelection) {
                    IStructuredSelection ss = (IStructuredSelection) selection;
                    Object obj = ss.getFirstElement();
                    if (obj instanceof IResource) {
                    	IProject project = ((IResource)obj).getProject();
                    	if (project != null) {
                    		return project;
                    	}
                    }
                }
            }
        } catch (Throwable t) {
            TestUIPlugin.logError(t);
        }
        return null;
    }
    

}
