/*******************************************************************************
 * Copyright (c) 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.test.ui.launch.delegates;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.ILaunchConfigurationDelegate;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.ui.internal.launch.processes.TestExecutionProcess;
import org.eclipse.hyades.test.ui.launch.extensions.IRunHandler;
import org.eclipse.hyades.test.ui.launch.extensions.IRunHandler2;

/**
 * This class is an abstract LaunchConfigurationDelegate that all Hyades Launch Configuration 
 * Delegates should inherit. It handles the common properties and behaviors for launching
 * a Hyades Launch Configuration.
 * Sub-classes should implement <code>getLaunchedElement(ILaunchConfiguration)</code>,
 * <code>invokeTestExecutionHarness(ILaunchConfiguration, String, StringBuffer)</code>.
 * They should usually also override <code>validate(ILaunchConfiguration)</code>.
 * @author jcanches
 * @since 3.1
 */
public abstract class AbstractLaunchConfigurationDelegate2 implements ILaunchConfigurationDelegate {

	private List problems;
	
	private ResourceSet resourceSet;
	
	/**
	 * Get the Test Element that the configuration will launch. This Test Element is used to
	 * determine which runHandler will be invoked before the Execution Harness is called.
	 * @param configuration A Hyades Launch Configuration
	 * @return A typed Test Element (instance of TPFTest or TPFTestComponent)
	 * @throws CoreException
	 */
	protected abstract Object getLaunchedElement(ILaunchConfiguration configuration) throws CoreException;
	
	/**
	 * Validate a given configuration. This method checks that the configuration contains the
	 * necessary information for running a test.
	 * Sub-classes should add more checks by overriding this method, and invoking the protected
	 * method <code>reportProblem(String)</code> (one or more times) for reporting any problems
	 * found in the configuration. If this method is not invoked, the configuration is considered
	 * as valid. Overriding methods should invoke the parent method using
	 * <code>super.validate(configuration, mode)</code>.
	 * @param configuration A Hyades Launch Configuration
	 * @param mode The run mode the Launch Configuration should be running in.
	 * @throws CoreException
	 */
	protected void validate(ILaunchConfiguration configuration, String mode) throws CoreException {
		Object launchedElement = getLaunchedElement(configuration);
		if (launchedElement == null) {
			reportProblem("The configuration does not provide a Launchable Test Element"); //$NON-NLS-1$
		}
	}
	
	/**
	 * Invokes the Test Execution Harness with the appropriate information from the
	 * given Launch Configuration, in the specified mode.
	 * @param configuration A Hyades Launch Configuration
	 * @param mode The run mode the Launch Configuration should be running in.
	 * @param errorMessages This is initially an empty buffer that this method should fill
	 * with the errors reported by the Test Execution Harness. The Test Execution Harness will be
	 * considered as failed if this buffer is not empty after the call to this method.
	 * @return An IExecutor instance. This instance represents the executor of the test. Its state
	 * will be monitored to display its state in the Debug View. null is accepted: in that case,
	 * the Test Execution Progress will not be monitored.
	 * @throws CoreException
	 */
	protected abstract IExecutor invokeTestExecutionHarness(ILaunchConfiguration configuration, String mode, StringBuffer errorMessages, IProgressMonitor monitor) throws CoreException;
	
	/**
	 * This method is used when validating a configuration for reporting problems.
	 * This method may be invoked several times by <code>validate</code> for reporting
	 * several problems.
	 * Invoking this method from another method than validate has no effect.
	 * @param description A description of the problem to report.
	 */
	protected void reportProblem(String description) {
		problems.add(description);
	}
	
	/**
	 * Allows to reset any private field. This method may be overrided, but sub-classes should not
	 * omit the call to super.resetFields().
	 */
	protected void resetFields() {
		this.problems = new ArrayList();
		this.resourceSet = null;
	}
	
	/**
	 * Returns the Delegate's private Resource Set. A delegate instantiates its own 
	 * ResourceSet at each launch. The Resource Set is unique during the launch process.
	 * @return
	 */
	protected ResourceSet getResourceSet() {
		if (this.resourceSet == null) {
			this.resourceSet = new ResourceSetImpl();
		}
		return this.resourceSet;
	}
	
	/**
	 * This method is not meant to be overrided. It is invoked by the Debug UI Plugin.
	 */
	public void launch(ILaunchConfiguration configuration, String mode,	ILaunch launch, IProgressMonitor monitor) throws CoreException {
		monitor.beginTask(TestUIPlugin.getString("BasicTestLaunchConfigurationDelegate.monitorLaunchingTest"), 3); //$NON-NLS-1$
		try {
			// Initialize all fields
			this.resetFields();
			
			// Validate the configuration
			this.validate(configuration, mode);
			if (problems.size() > 0) {
				MultiStatus status = new MultiStatus(TestUIPlugin.getID(), 0, TestUIPlugin.getString("_EXC_AbstractLaunchConfigurationDelegate.configurationProblems"), null); //$NON-NLS-1$
				Iterator it = problems.iterator();
				while (it.hasNext()) {
					String problem = (String) it.next();
					IStatus subStatus = new Status(IStatus.ERROR, TestUIPlugin.getID(), 0, problem, null);
					status.add(subStatus);
				}
				throw new CoreException(status);
			}
			
			// Pre-run extension
	    	IRunHandler runHandler = LaunchConfigurationExtensionsManager.getInstance().getRunHandler(getLaunchedElement(configuration));
	    	if (runHandler != null) {
	    		try {
	    			runHandler.preRun(configuration, mode, launch, getResourceSet(), new SubProgressMonitor(monitor, 1));
	    		} catch (Throwable t) {
	    			IStatus status = new Status(IStatus.ERROR, TestUIPlugin.getID(), 0, TestUIPlugin.getString("_EXC_AbstractLaunchConfigurationDelegate.preRunHandlerException"), t); //$NON-NLS-1$
	    			throw new CoreException(status);
	    		}
	    	}
			
			// Invoke the Test Execution Harness
			StringBuffer errorMessages = new StringBuffer();
			IExecutor executor = null;
			try {
				executor = invokeTestExecutionHarness(configuration, mode, errorMessages, new SubProgressMonitor(monitor, 1, SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
				if (errorMessages.length() > 0) {
					IStatus status = new Status(IStatus.ERROR, TestUIPlugin.getID(), 0, TestUIPlugin.getString("_EXC_AbstractLaunchConfigurationDelegate.testHarnessProblems") + errorMessages, null); //$NON-NLS-1$
		    		throw new CoreException(status);
				}
			} catch (Throwable t) {
				IStatus status = new Status(IStatus.ERROR, TestUIPlugin.getID(), 0, TestUIPlugin.getString("_EXC_AbstractLaunchConfigurationDelegate.testHarnessProblems") + t.getClass().getName(), t); //$NON-NLS-1$
	    		throw new CoreException(status);
			}
			
			if (executor != null) {
				launch.addProcess(new TestExecutionProcess(executor, launch, getResourceSet()));
				
				if (runHandler != null && runHandler instanceof IRunHandler2) {
					((IRunHandler2)runHandler).postLaunch(launch, executor, new SubProgressMonitor(monitor, 1, SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
				}
			}
			
		} finally {
			monitor.done();
		}
	}
	
}
