/**********************************************************************
 * Copyright (c) 2003,2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.sdb.analysis;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.hyades.analysis.engine.IAnalysisMonitor;
import org.eclipse.hyades.analysis.engine.ILogAnalyzer;
import org.eclipse.hyades.analysis.engine.Incident;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.util.LogUIConstants;
import org.eclipse.hyades.models.cbe.CBECommonBaseEvent;
import org.eclipse.hyades.models.cbe.CBEDefaultEvent;
import org.eclipse.hyades.sdb.internal.SDbPlugin;
import org.eclipse.hyades.sdb.internal.preferences.SymptomDBDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.internal.Workbench;

public class LogAnalyzerImpl implements ILogAnalyzer {

	protected String errorMessage;
	protected Map _analysisEngine = new Hashtable();
	private final String PLATFORM_PREFIX = "platform:/resource";
	private Incident incident;
	/**
	 * Constructor for LogAnalyzerImpl.
	 */
	public LogAnalyzerImpl() {
		super();
		incident = new Incident();
	}

	/**
	 * @see org.eclipse.hyades.analysis.engine.ILogAnalyzer#errorMsg()
	 */
	public String errorMsg() {
		return errorMessage;
	}

//	public void importDB(IProgressMonitor monitor) {
//		//imports the DB since it is not yet within the workspace
//		String absolutePath = "";
//		try {
//			String symptomdbPath = SDbPlugin.getDefault().getPreferenceStore().getString(SdUIConstants.SYMPTOM_DB_PATH_KEY);
//			String pName = SDbPlugin.getDefault().getPreferenceStore().getString(SdUIConstants.PROJECT_NAME_KEY);
//			String sdbFileName = SDbPlugin.getDefault().getPreferenceStore().getString(SdUIConstants.SDB_FILE_NAME_KEY);
//
//			int i = symptomdbPath.indexOf(",");
//			if (i != -1 && symptomdbPath.substring(i + 1, i + 2).equals("1")) {
//				absolutePath = symptomdbPath.substring(0, i);
//			}
//
//			SDBLoader l = new SDBLoader();
//			l.loadDatabase(new Path(sdbFileName).toString(), new Path(PLATFORM_PREFIX + "/" + absolutePath).toString(), monitor);
//
//		} catch (Exception e) {
//		}
//
//	}
	
	private int loadDatabases() {
		Map list = getSymptomDBPathList();

		Set keySet = _analysisEngine.keySet();

		for (Iterator iter = keySet.iterator(); iter.hasNext();) {
			AnalysisEngine element = (AnalysisEngine) iter.next();
			if (!databaseExists(element.getSymptomDatabasePath())) {
				element.removeSymptomDatabase();
				iter.remove();
				continue;
			}

			String path = element.getSymptomDatabasePath();
			Long value = (Long) list.get(path);
			if (value == null) {
				element.removeSymptomDatabase();
				iter.remove();
			} else if ((value != null && value.longValue() != ((Long) _analysisEngine.get(element)).longValue())) {
				element.removeSymptomDatabase();
				iter.remove();
			} else {
				list.remove(path);
			}
		}

		Set listSet = list.keySet();
		for (Iterator iter = listSet.iterator(); iter.hasNext();) {
			String path = (String) iter.next();
			_analysisEngine.put(AnalysisEngine.getInstance(path), list.get(path));
		}
		
		return _analysisEngine.size();
	}
	/**
	 * @see org.eclipse.hyades.analysis.engine.ILogAnalyzer#loadDatabase()
	 */
	public void loadDatabase() {

		errorMessage = null;

		int engineSize = loadDatabases();

		if (engineSize == 0) {
			errorMessage = SDbPlugin.getResourceString("STR_NO_SYMPTOM_DB_ERROR");
			Display.getDefault().syncExec(new Runnable(){
				  public void run(){
				  	if (MessageDialog.openQuestion(Workbench.getInstance().getActiveWorkbenchWindow().getShell(),
				  			LogUIPlugin.getResourceString("STR_LOG_MSG"),
							LogUIPlugin.getResourceString("STR_NO_SYMPTOM_DB_ERROR"))) {
				  		SymptomDBDialog dialog = new SymptomDBDialog(Workbench.getInstance().getActiveWorkbenchWindow().getShell(),
				  				LogUIPlugin.getResourceString("STR_SYMPTOM_DB_DLG_TITLE"),
								null);
				  		dialog.open();
				  		if (dialog.getReturnCode() == Window.OK) {
				  			if (loadDatabases()>0) errorMessage = null;
				  		}
				  	}
				  }
			});
		}
	}

	/**
	 * @see org.eclipse.hyades.analysis.engine.ILogAnalyzer#unloadDatabase()
	 */
	public void unloadDatabase() {

		Set analysisSet = _analysisEngine.keySet();
		for (Iterator iter = analysisSet.iterator(); iter.hasNext();) {
			AnalysisEngine engine = (AnalysisEngine) iter.next();
			engine.removeSymptomDatabase();

		}
		_analysisEngine.clear();

	}

	protected Map getSymptomDBPathList() {
		Map list = new HashMap();
		String symptomdbPath = SDbPlugin.getDefault().getPreferenceStore().getString(LogUIConstants.SYMPTOM_DB_PATH_KEY);

		if (symptomdbPath == null || symptomdbPath.equals(""))
			return list;

		int idx = symptomdbPath.indexOf(";");
		while (idx != -1) {
			String data = symptomdbPath.substring(0, idx);
			int i = data.indexOf(",");

			if (i != -1 && data.substring(i + 1).equals("1")) {

				String path = data.substring(0, i);
				IWorkspace workbench = ResourcesPlugin.getWorkspace();
				IPath filePath = new Path(path);
				IResource fres = workbench.getRoot().findMember(filePath);
				if (fres != null && fres.exists()) {
					list.put(PLATFORM_PREFIX + path, new Long(getTimestamp(fres)));
				}
			}

			symptomdbPath = symptomdbPath.substring(idx + 1);
			idx = symptomdbPath.indexOf(";");
		}

		int i = symptomdbPath.indexOf(",");
		if (i != -1 && symptomdbPath.substring(i + 1).equals("1")) {

			String path = symptomdbPath.substring(0, i);
			IWorkspace workbench = ResourcesPlugin.getWorkspace();
			IPath filePath = new Path(path);
			IResource fres = workbench.getRoot().findMember(filePath);
			if (fres != null && fres.exists()) {
				list.put(PLATFORM_PREFIX + path, new Long(getTimestamp(fres)));
			}
		}

		return list;
	}

	protected boolean databaseExists(String path) {
		if (path == null)
			return false;

		path = resourcePath(URI.createURI(path));
		IWorkspace workbench = ResourcesPlugin.getWorkspace();
		IResource res = workbench.getRoot().findMember(path);
		if (res == null || !res.exists())
			return false;

		return true;
	}

	protected String resourcePath(URI uri) {
		String path = uri.path();

		if (path.startsWith("/resource"))
			return path.substring(9);

		if (path.startsWith(PLATFORM_PREFIX))
			return path.substring(18);

		return path;
	}

	public String analyze(Object selection, IAnalysisMonitor monitor) {

		Iterator i = ((List) selection).iterator();
		String[] record = new String[]{"0", String.valueOf(((List) selection).size())};
		int count = 0;
		while (i.hasNext()) {
			Object elem = i.next();
			if (elem instanceof CBECommonBaseEvent)
				//list.append(
				analyzeEvent((CBECommonBaseEvent) elem);
			if (monitor != null) {
				record[0] = String.valueOf(++count);
				monitor.worked(1);
				monitor.subTask(LogUIPlugin.getResourceString("STR_ANALYZE_RECORD", record));
				if (monitor.isCanceled())
					return "";
			}
		}

		return "";
	}

	protected String analyzeEvent(CBECommonBaseEvent logRecord) {

		String symptomString = getSymptomStringFor(logRecord);

		return findMatchesFor(logRecord, symptomString);

	}

	protected String getSymptomStringFor(CBECommonBaseEvent logRecord) {

		String symptomString = logRecord.getExtensionName();

		symptomString = logRecord.getMsg();

		return symptomString;

	}

	private String findMatchesFor(CBECommonBaseEvent logRecord, String symptomString) {

		EList directivesList = logRecord.getSymptoms();

		if (!directivesList.isEmpty())
			directivesList.clear();

		Set keySet = _analysisEngine.keySet();
		incident.setMessageId(symptomString);
		for (Iterator iter = keySet.iterator(); iter.hasNext();) {
			AnalysisEngine engine = (AnalysisEngine) iter.next();
			Object[] directives = engine.analyze(incident);

			for (int idx = 0; idx < directives.length; idx++) {

				logRecord.getSymptoms().add(directives[idx]);
			}

		}

		((CBEDefaultEvent) logRecord).setAnalyzed(true);
		return symptomString;

	}
	private long getTimestamp(IResource resource) {

		long currentTimestamp = 0;
		IPath path = resource.getLocation();
		if (path != null) {
			currentTimestamp = path.toFile().lastModified();
		}
		return currentTimestamp;

	}

}
