/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.remote;

import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.util.Enumeration;
import java.util.Vector;

import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;


public class RemoteComponentSkeleton {

	/* Whether the native impls are available; the value of this is fully
	 * known at class initialization time and is set by a static initializer 
	 * in this class.  Is immutable once set in the static initializer to
	 * avoid potential threading issues across instances of this class.
	 */
    private static final boolean nativesAvailable;

    /* Whether the agnet is being monitored currently */
    private boolean isLogging = false;

    /* Whether the agent is registered with the Agent Controller */
    private boolean isInitialized = false;

    /* Listeners for monitoring events */
    protected final Vector monitorListeners = new Vector();

    /* Listeners for agent controller events */
    protected final Vector serverListeners = new Vector();

	/* Listeners for custom commands */
	protected final Vector commandListeners = new Vector();

    /* Listeners for client events */
    protected final Vector clientListeners = new Vector();

    /* Whether the UUID's for the node and JVM are valid */
    private boolean uuidsValid = false;

    /* The UUID's usedby the Agent Controller */
    private String agentUUID;
    private String nodeUUID;
    private String jvmUUID;

    /* The following fields are used by the native code.  DO NOT MODIFY */
    private String name;
    private String type;
    private boolean isRegistered = false;
    /* This is a instance storage mechanism that the native code uses
       to quickly locate the logging mechanism data that is stored in
       the native library. DO NOT MODIFY THIS VARIABLE
    */
    private long mechanism = -1;   /* bugzilla 65470 - changed initial value from 0 to -1 because 0 is a valid list index */

    /* Whether the Agent Controller is running on the local box.  A single flag allows us to streamline handshaking for
       multiple agent instances.
     */
    private static boolean serviceAvailable = false;

    /* The number of agents */
    private static int agentCount = 0;

    /* Whether we are currently requesting a remote monitor through a peer agent */
    private boolean requestForMonitor = false;

    /* Try and load the native library.  If this fails we will ignore the error;
     * first assign to local variable to determine if natives are loadable and then
     * assign blank final natives available static variable.
     */
    static {
        boolean nativesLoadable = false;
        try {
            System.loadLibrary("hcjbnd");
            nativesLoadable = true;
        }
        catch (Throwable e) {
        }
        nativesAvailable = nativesLoadable;
    }

    /* Default configuration for this IRemoteEntity.*/
    private AgentConfiguration _defaultConfiguration = null;

    /**
     * Create aaAgent with the specified name and type<br>
     *
     * @param name java.lang.String - the remotely identifyable name of the agent
     * @param type java.lang.String - the remotely identifyable type of the agent
     */
    public RemoteComponentSkeleton(String name, String type) {
        super();
        setName(name);
        setType(type);

        /* Create the default configuration for this IRemoteEntity using its name.*/
        _defaultConfiguration = new AgentConfiguration();
        _defaultConfiguration.setAgentName(name);
    }

    /**
     * Add a listener for agent controller events
     * @param listener - org.eclipse.hyades.internal.execution.remote.AgentControllerListener
     */
    public void addAgentControllerListener(AgentControllerListener listener) {
    	synchronized(serverListeners) {
    		 serverListeners.add(listener);
    	}
    }

    /**
     * Add a listener for monitor events.
     * @param listener org.eclipse.hyades.internal.execution.remote.MonitorListener
     */
    public void addMonitorListener(MonitorListener listener) {
    	synchronized(monitorListeners) {
    		monitorListeners.add(listener);
    	}
    }

    /**
     * Add a listener for client events.
     * @param listener - org.eclipse.hyades.internal.execution.remote.RemoteClientListener
     */
    public void addRemoteClientListener(RemoteClientListener listener) {
    	synchronized(clientListeners) {
    		clientListeners.add(listener);
    	}
    	
    }
    
	/**
	 * Add a listener for client events.
	 * @param listener - org.eclipse.hyades.internal.execution.remote.RemoteClientListener
	 */
	public void addCommandListener(CustomCommandHandler listener) {
		synchronized(commandListeners) {
			commandListeners.add(listener);
		}
	
	}

    /**
     * Called by the native implementation to indicate that the agent
     * is now registered with the Agent Controller.
     */
    private void agentControllerActive() {
        serviceAvailable = true;
        uuidsValid = true;

        synchronized (this.getClass()) {
            if (agentCount == 0) {
                this.getClass().notify();
            }
        }

        synchronized(serverListeners) {
        	Enumeration e=serverListeners.elements();
        	while(e.hasMoreElements()) {
        		AgentControllerListener listener=(AgentControllerListener)e.nextElement();
        		listener.agentControllerActive();
        	}
        }
    }

    /**
     * Called by the native implementation to indicate that the agent is no longer
     * registered with the Agent Controller.  At the same time the IRemoteExecutionComponent's default configuration
     * is destroyed as it may be changed by the Agent Controller before the next time we re-register.
     */
    private void agentControllerInactive() {
        serviceAvailable = false;
        setLogging(false);
       	_defaultConfiguration.clear();
        synchronized(serverListeners) {
        	Enumeration e=serverListeners.elements();
        	while(e.hasMoreElements()) {
        		AgentControllerListener listener=(AgentControllerListener)e.nextElement();
        		listener.agentControllerInactive();
        	}
        }
    }

    /**
     * Called by the native implementation to indicate that the agents
     * remotely attached client has either detached from the agent or
     * that the client has died.
     */
    private void remoteClientExited() {
        setLogging(false);
        synchronized(clientListeners) {
        	Enumeration e=clientListeners.elements();
        	while(e.hasMoreElements()) {
        		RemoteClientListener listener=(RemoteClientListener)e.nextElement();
        		listener.clientInactive();
        	}
        }
    }

    /**
     * Deregister from the Agent Controller.
     */
    public synchronized void deregister() {
        if (this.isInitialized) {
            this.deregister0(this.name);
            this.isInitialized = false;
            /**
             * Synchronize decrement of agent count
             */
            synchronized (this.getClass()) {
                agentCount--;
            }
        }
    }
    /**
     * Native delegate of IRemoteExecutionComponent.deregister()
     */
    private native void deregister0(String name);

    /**
     * If the instance is finalized it will be deregistered from the Agent Controller.  Otherwise
     * it is not deregistered until the JVM exits.
     */
    public void finalize() {
        try {
            this.deregister0(name);
        } finally {
            try {
                super.finalize(); 
            } catch (Throwable t) {
            }
        }
    }

    /**
     * Retrieve the name of the IRemoteExecutionComponent instance.
     * @return java.lang.String
     */
    public String getName() {
        return name;
    }

    /**
     * Retrieve the type of this IRemoteExecutionComponent instance.
     * @return java.lang.String
     */
    public String getType() {
        return type;
    }

    /**
     * incommingCommand method comment.
     */
    /*
    public void handleMessage(IMessage message) {
    	
  
        switch ((int) command.getTag()) {
            case (int) Constants.RA_CUSTOM_COMMAND :
                break;
        }
        if (handler != null) {
            handler.incommingCommand(null, command);
        }
    }
    */

    /**
     * Registers the IRemoteExecutionComponent with the Agent Controller.  If the Agent Controller is not running it throws an
     * AgentControllerUnavailableException but the registration is still pending and
     * the AgentControllerListeners will be invoked when the Agent Controller becomes vailabale.
     */
    public synchronized void initialize() throws AgentControllerUnavailableException {
    	/* If we could not load the .dll we need to throw an exception */
    	if(!nativesAvailable) {
    		throw new AgentControllerUnavailableException();
    	}
    	
        if (!isInitialized) {
            this.initializeEngine0(name, type);

            /* We will wait up to two seconds before continuing if this is the first instance. */
            synchronized (this.getClass()) {
                if (agentCount == 0) {
                    try {
                        this.getClass().wait(2000);
                    }
                    catch (InterruptedException e) {
                        /* Allow to continue */
                    }
                }
                agentCount++;

            }
            isInitialized = true;
        }
        if (!serviceAvailable) {
            throw new AgentControllerUnavailableException();
        }
    }

    public synchronized void initializeFast() throws AgentControllerUnavailableException {
    	/* If we could not load the .dll we need to throw an exception */
    	if(!nativesAvailable) {
    		throw new AgentControllerUnavailableException();
    	}
    	
        if (!isInitialized) {
            this.initializeEngine0(name, type);

            synchronized (this.getClass()) {
                agentCount++;
            }
            isInitialized = true;
        }
        if (!serviceAvailable) {
            throw new AgentControllerUnavailableException();
        }	
    }

    /**
     * Native implementation of IRemoteExecutionComponent.initializeEngine().
     */
    private native void initializeEngine0(String name, String uuid) throws AgentControllerUnavailableException;

    /**
     * Returns whether the agent controller is currently active.
     */
    public boolean isAgentControllerActive() {
        return RemoteComponentSkeleton.serviceAvailable;
    }

    /**
     * Returns whether or not this instance of the agent is currently
     * being monitored.<br>
     * ie. Whether the log messages are being collected.<br>
     */
    public boolean isLogging() {
        return isLogging;
    }

    /**
     * Log the specified data.<br>
     *
     * @param msg byte[] - the data to log
     * @param offset int - offset within msg to start logging from
     * @param len int    - the number of bytes to log.
     */
    public synchronized void logMessage(byte[] msg, int offset, int len) {
        if (isLogging)
            this.logMessage0(name, msg, offset, len);
    }

    /**
     * Log the specified message.<br>
     *
     * @param msg java.lang.String - message to log.
     */
    public synchronized void logMessage(String msg) {
        if (isLogging)
            this.logMessage0(name, msg);
    }

    /**
     * Native implementation of IRemoteExecutionComponent.logMessage()
     * @param msg byte[]
     * @param len int
     */
    private native void logMessage0(String agentName, byte[] msg, int offset, int len);

    /**
     * Native implemenation of IRemoteExecutionComponent.logMeggase()
     * @param msg java.lang.String
     */
    private native void logMessage0(String agentName, String msg);


	/**
	 * The following method is used to only log error messages to an active
	 * agent.
	 *
	 * @param message	-	The message to be logged
	 *		  messageId -	The message ID corresponding to the error message
	 *		  severity	-	The level of severity of the message (e.g. default = DEBUG, 1 = INFORMATION, 2 = WARNING, 3 = SEVERE, 4 = MAX ERROR LEVEL)
	 */
    public synchronized void logErrMsg(String message, String messageId, int severity) {
		logErrorMessage(message, messageId, severity);    
    }

 
	/**
	 * Native implementation of logErrorMessage
	 */    
    private native void logErrorMessage(String message, String messageId, int severity);


    /**
     * @param msg java.lang.String
     */
    public synchronized void logMessageUTF8(String msg) {
        if (isLogging)
        
        	/* We will try and log to the correct API for handling
        	 * UTF-8 strings. With old .dll's this will not be resolvable
        	 * and we should use the old version.  The old version
        	 * will only handle ASCII strings
        	 */
        	try {
        		this.logMessageUTF81(name, msg.getBytes("UTF-8"));
        	}
        	catch(Throwable e) {
        		logMessageUTF80(name, msg);
        	}
    }

    /**
     * @param msg java.lang.String
     */
    private native void logMessageUTF81(String agentName, byte[] msg);
    
    /**
     * This does not support UTF-8 encoding properly.
     * @deprecated
     */
    private native void logMessageUTF80(String agentName, String msg);

/*
    public void setMessageHandler(IMessageHandler handler) {
        this.handler = handler;
    }
*/

    /**
     * Sends a message to the remotely attached client if it exists.
     */
    public void sendMessageToAttachedClient(String message, long contextId) {
    	if(nativesAvailable) {
    		try {
    			byte[] bytes = message.getBytes("UTF-8");
        		sendMessageBinary0(bytes, 0, bytes.length, contextId);
    		}
    		catch (UnsupportedEncodingException ex){}
			
    	}
    }

    /**
     * The native implementation of sendMessageToAttachedCleint.
     */
    private native void sendMessage0(String message, long contextId);

    /**
     * Sends a message to the remotely attached client if it exists.
     */
    public void sendMessageToAttachedClient(byte[] message, int offset, int length, long contextId) {
    	if(nativesAvailable) {
    		sendMessageBinary0(message, offset, length, contextId);
    	}
    }

    /**
     * The native implementation of sendMessageToAttachedCleint.
     */
    private native void sendMessage0(byte[] message, int offset, int length, long contextId);
    
	/**
	 * The native implementation of sendMessage that sends the message as binary
	 * data.
	 */
	private native void sendMessageBinary0(byte[] message, int offset, int length, long contextId);

    /**
     * This is called by the native code to turn logging
     * "on" or "off".  When turned "off" there is no stepping
     * through the JNI boundary.
     */
    private synchronized void setLogging(boolean log) {
        isLogging = log;
        synchronized(monitorListeners) {
        	Enumeration e=monitorListeners.elements();
        	while(e.hasMoreElements()) {
        		MonitorListener listener=(MonitorListener)e.nextElement();
        		if (isLogging) {
        			listener.monitorActive();
            		}
            		else {
                		listener.monitorInactive();
            		}
            	}
        }
        if (isLogging && requestForMonitor) {
            synchronized (this) {
                this.notify();
            }
        }

}

    /**
     * Invoked by the native layer to set the agent name to the default name if this
     * instance was created with the default constructor.
     * @param name java.lang.String
     */
    private void setName(String name) {
        this.name = new String(name);
    }

    /**
     * Invoked by the native layer to set the type to the default type if
     * this instance was created with the default condtructor.
     * @param uuid java.lang.String
     */
    private void setType(String type) {
        this.type = new String(type);
    }

    /**
     * Invoked by the native layer to provide the agent UUID.
     */
    private void setAgentUUID(String type) {
        agentUUID = type;
    }

    /**
     * retrieve the agent's UUID as used by the Agent Controller.
     */
    public String getAgentUUID() {
        return agentUUID;
    }

    /**
     * Invoked by the native layer to provide the node UUID.
     *
     */
    private void setNodeUUID(String type) {
        nodeUUID = type;
    }

    /**
     * Retrieve the node's UUID as used by the Agent Controller.  Throws AgentNotRegisteredException if
     * the IRemoteExecutionComponent has not registered with the Agent Controller yet.
     */
    public String getNodeUUID() throws AgentNotRegisteredException {
        if (!uuidsValid) {
            throw new AgentNotRegisteredException();
        }
        return nodeUUID;
    }

    /**
     * Invoked by the native layer to set the JVM's UUID.
     */
    private void setJVMUUID(String type)  {
        jvmUUID = type;
    }

    /**
     * Retrieve the JVM's UUID as used by the Agent Controller.  Throws AgentNotRegisteredException if
     * the IRemoteExecutionComponent has not registered with the Agent Controller yet.
     */
    public String getJVMUUID() throws AgentNotRegisteredException {
        if (!uuidsValid) {
            throw new AgentNotRegisteredException();
        }
        return jvmUUID;
    }

    /**
     * Request for this agent to be monitored by the same client that is monitoring another agent.
     * @param addr - java.net.InetAddress
     * @param agentUUID -
     * @param timeout - how long to wait for the client to start monitoring. Milliseconds
     */
    public void requestMonitorThroughPeer(InetAddress addr, String agentUUID, long timeout) {
    	
    	if(!nativesAvailable) {
    		return;
    	}
    	

        synchronized (this) {
            requestForMonitor = true;
            requestMonitorThroughPeer0(addr.getAddress(), agentUUID);
            try {
                this.wait(timeout);
            }
            catch (InterruptedException e) {
            }
            finally {
                requestForMonitor = false;
            }

        }
    }

    private native void requestMonitorThroughPeer0(byte[] addr, String agentUUID);

    /**
     * Request for this agent to be monitored by the same client that is monitoring another agent.
     * @param addr - java.net.InetAddress
     * @param timeout - how long to wait for the client to start monitoring. Milliseconds
     */
    public void requestMonitorThroughPeer(InetAddress addr, long pid, String agentName, long timeout) {

	if(!nativesAvailable) {
    		return;
    	}
    	
        synchronized (this) {
            requestForMonitor = true;
            requestMonitorThroughPeer0(addr.getAddress(), pid, agentName);
            try {
                this.wait(timeout);
            }
            catch (InterruptedException e) {
            }
            finally {
                requestForMonitor = false;
            }

        }
    }

    private native void requestMonitorThroughPeer0(byte[] addr, long pid, String agentName);

    /**
     * Invoked by the native layer to indicate that the agent requested throug a requestMonitorThroughPeer was
     * not able to be contacted.  This will allow the invoking thread of requestMonitorThroughPeer to continue
     * without waiting the entire timeout period.
     */
    private void peerUnreachable() {
    	synchronized(this) {
    		if(requestForMonitor) {
    			this.notifyAll();	
    		}
    	}
    }

    /**
     * Return the default configuration for this IRemoteExecutionComponent.
     *
     * @return org.eclipse.hyades.internal.execution.remote.AgentConfiguration reference to the default configuration.
     */
    public AgentConfiguration getDefaultConfiguration() {
        return _defaultConfiguration;
    }

    /**
     * This method is called by the native code to add an entry to the default configuration for this IRemoteExecutionComponent.
     *
     * @param entry the new org.eclipse.hyades.internal.execution.remote.AgentConfigurationEntry to be added to the default configuration.
     */
    private void addEntryToDefaultConfiguration(AgentConfigurationEntry entry) {
        _defaultConfiguration.addEntry(entry);

    }

    public void broadcastMessage(String message, long contextId) {
        sendMessageToAttachedClient(message, contextId);
    }

    public void broadcastMessage(byte[] message, int offset, int length, long contextId) {
        sendMessageToAttachedClient(message, offset, length, contextId);
    }

    private Object createObjectInContextOfClassLoader(String objectType) {
    	Object result=null;
    	try {
    		Class clazz=Class.forName(objectType);
    		result=clazz.newInstance();
    	}
    	catch(Throwable e) {
    	}
    	return result;	
    }
    
    private void incomingCommand(CommandElement command) {
    	if(command instanceof CustomCommand) {
	 		synchronized(commandListeners) {
				Enumeration e=commandListeners.elements();
				while(e.hasMoreElements()) {
					CustomCommandHandler listener=(CustomCommandHandler)e.nextElement();
					listener.handleCommand((CustomCommand)command);
				}
	 		}
    	}
    }

	/**
	 * Returns whether the agent registered successfully with Agent Controller
	 * @return Returns isRegistered.
	 */
	public boolean isRegistered() {
		return isRegistered;
	}
}
