package org.eclipse.hyades.logging.core.tests;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;

import javax.xml.transform.TransformerFactory;

import junit.framework.AssertionFailedError;
import junit.framework.TestCase;

import org.eclipse.hyades.internal.logging.core.Constants;
import org.eclipse.hyades.logging.core.XmlUtility;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * All JUnit test cases for the
 * <code>org.eclipse.hyades.logging.core.XmlUtility</code> class.
 * <p>
 * 
 * @author Paul E. Slauenwhite
 * @version December 2, 2004
 * @since July 28, 2004
 */
public class XmlUtilityTest extends TestCase {

    private TestingUtilities utilities = null;

    private String cbeXMLNotFormattedWithSchema = null;

    private StringBuffer cbeXMLFormattedWithSchema = null;

    private StringBuffer symptomDatabaseXMLNotFormattedWithDTD = null;

    private String symptomDatabaseXMLNotFormattedNoDTD = null;

    private StringBuffer symptomDatabaseXMLFormattedWithDTD = null;

    private StringBuffer symptomDatabaseXMLFormattedNoDTD = null;
    
    /**
     * Constructor for XmlUtilityTest.
     * <p>
     * 
     * @param name
     *            The name of the test case.
     */
    public XmlUtilityTest(String name) {
        super(name);
    }

    protected void setUp() throws Exception {

        super.setUp();

        utilities = TestingUtilities.getInstance();
    }
    
    /**
     * Test normalization API.
     */
    public void testNormalization() {

        try {

            String string = null;

            assertTrue(XmlUtility.normalize(string).equals("null"));

            string = "";

            assertTrue(XmlUtility.normalize(string).equals(""));

            string = " < ";

            assertTrue(XmlUtility.normalize(string).equals(" &lt; "));

            string = " > ";

            assertTrue(XmlUtility.normalize(string).equals(" &gt; "));

            string = " & ";

            assertTrue(XmlUtility.normalize(string).equals(" &amp; "));

            string = " \" ";

            assertTrue(XmlUtility.normalize(string).equals(" &quot; "));

            string = " ' ";

            assertTrue(XmlUtility.normalize(string).equals(" &apos; "));

            string = " \t ";

            assertTrue(XmlUtility.normalize(string).equals(" &#x9; "));

            string = " \n ";

            assertTrue(XmlUtility.normalize(string).equals(" &#xA; "));

            string = " \r ";

            assertTrue(XmlUtility.normalize(string).equals(" &#xD; "));

            string = " ";

            assertTrue(XmlUtility.normalize(string).equals(" "));
            //assertTrue(XmlUtility.normalize(string).equals(" &#x20; "));

            for (char character = 0x0020; character <= 0xD7FF; character++) {

                if ((character != '<') && (character != '>') && (character != '&') && (character != '\"') && (character != '\'') && (character != '\t') && (character != '\n') && (character != '\r') && (character != ' ')) {

                    string = " " + character + " ";

                    assertTrue(XmlUtility.normalize(string).equals(" " + character + " "));
                }
            }

            for (char character = 0xE000; character <= 0xFFFD; character++) {

                string = " " + character + " ";

                assertTrue(XmlUtility.normalize(string).equals(" " + character + " "));
            }

            for (char character = 0x0000; character <= 0x0010; character++) {

                if ((character != '<') && (character != '>') && (character != '&') && (character != '\"') && (character != '\'') && (character != '\t') && (character != '\n') && (character != '\r') && (character != ' ')) {

                    string = "" + character;

                    assertTrue(XmlUtility.normalize(string).equals("?"));
                }
            }

            for (char character = 0x0011; character < 0x0020; character++) {

                string = " " + character + " ";

                assertTrue(XmlUtility.normalize(string).equals(" ? "));
            }

            for (char character = 0xD800; character < 0xE000; character++) {

                string = " " + character + " ";

                assertTrue(XmlUtility.normalize(string).equals(" ? "));
            }

            string = " " + ((char) (0xFFFE)) + " ";

            assertTrue(XmlUtility.normalize(string).equals(" ? "));

            string = " " + ((char) (0xFFFF)) + " ";

            assertTrue(XmlUtility.normalize(string).equals(" ? "));

            for (char firstCharacter = 0x0001; firstCharacter <= 0x0010; firstCharacter++) {

                if ((firstCharacter != '<') && (firstCharacter != '>') && (firstCharacter != '&') && (firstCharacter != '\"') && (firstCharacter != '\'') && (firstCharacter != '\t') && (firstCharacter != '\n') && (firstCharacter != '\r') && (firstCharacter != ' ')) {

                    char secondCharacter = 0x0000;

                    while (true) {

                        string = " " + firstCharacter + secondCharacter + " ";

                        assertTrue(XmlUtility.normalize(string).equals(" " + firstCharacter + secondCharacter + " "));

                        if (secondCharacter == 0xFFFF) {
                            break;
                        }

                        secondCharacter++;
                    }
                }
            }
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        }
    }

    /**
     * Test de-normalization API.
     */
    public void testDeNormalization() {

        try {

            String string = null;

            assertTrue(XmlUtility.denormalize(string).equals("null"));

            string = "";

            assertTrue(XmlUtility.denormalize(string).equals(""));

            string = " &lt; ";

            assertTrue(XmlUtility.denormalize(string).equals(" < "));

            string = " &gt; ";

            assertTrue(XmlUtility.denormalize(string).equals(" > "));

            string = " &amp; ";

            assertTrue(XmlUtility.denormalize(string).equals(" & "));

            string = " &quot; ";

            assertTrue(XmlUtility.denormalize(string).equals(" \" "));

            string = " &apos; ";

            assertTrue(XmlUtility.denormalize(string).equals(" ' "));

            string = " &#x9; ";

            assertTrue(XmlUtility.denormalize(string).equals(" \t "));

            string = " &#xA; ";

            assertTrue(XmlUtility.denormalize(string).equals(" \n "));

            string = " &#xD; ";

            assertTrue(XmlUtility.denormalize(string).equals(" \r "));

            string = " ";
            //string = " &#x20; ";

            assertTrue(XmlUtility.denormalize(string).equals(" "));

            string = " &EntityReference; ";

            assertTrue(XmlUtility.denormalize(string).equals(" &EntityReference; "));

            char character = 0x0000;

            while (true) {

                string = " " + character + " ";

                assertTrue(XmlUtility.denormalize(string).equals(" " + character + " "));

                if (character == 0xFFFF) {
                    break;
                }

                character++;
            }
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        }
    }

    /**
     * Test serialize(Document) API.
     */
    public void testSerialize_Document_API() {

        try {

            createXMLDocuments("UTF-8");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMNotFormattedWithSchema), cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMFormattedWithSchema), cbeXMLFormattedWithSchema.toString());

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD), symptomDatabaseXMLFormattedNoDTD.toString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        }
    }

    /**
     * Test serialize(Document, boolean) API.
     */
    public void testSerialize_Document_boolean_API() {

        try {
            
            createXMLDocuments("UTF-8");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMNotFormattedWithSchema, true), cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMFormattedWithSchema, true), cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMNotFormattedWithSchema, false), cbeXMLNotFormattedWithSchema);

            assertEquals(XmlUtility.serialize(cbeDOMFormattedWithSchema, false), cbeXMLNotFormattedWithSchema);

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, true), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, true), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, true), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, true), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, false), symptomDatabaseXMLNotFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, false), symptomDatabaseXMLNotFormattedNoDTD);

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, false), symptomDatabaseXMLNotFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, false), symptomDatabaseXMLNotFormattedNoDTD);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        }
    }

    /**
     * Test serialize(Document, boolean, String) API.
     */
    public void testSerialize_Document_boolean_String_API() {

        try {

            String encoding = "ISO-8859-1";
            
            createXMLDocuments(encoding);

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMNotFormattedWithSchema, true, encoding), cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMFormattedWithSchema, true, encoding), cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMNotFormattedWithSchema, false, encoding), cbeXMLNotFormattedWithSchema);

            assertEquals(XmlUtility.serialize(cbeDOMFormattedWithSchema, false, encoding), cbeXMLNotFormattedWithSchema);

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, true, encoding), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, true, encoding), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, true, encoding), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, true, encoding), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, false, encoding), symptomDatabaseXMLNotFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, false, encoding), symptomDatabaseXMLNotFormattedNoDTD);

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, false, encoding), symptomDatabaseXMLNotFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, false, encoding), symptomDatabaseXMLNotFormattedNoDTD);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        }
    }
    
    /**
     * Test serialize(Document, File) API.
     */
    public void testSerialize_Document_File_API() {

        File tempFile = null;

        try {

            createXMLDocuments("UTF-8");

            tempFile = File.createTempFile("document", "xml");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, tempFile);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMFormattedWithSchema, tempFile);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, tempFile);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, tempFile);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, tempFile);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, tempFile);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }

    /**
     * Test serialize(Document, File, boolean) API.
     */
    public void testSerialize_Document_File_boolean_API() {

        File tempFile = null;

        try {

            createXMLDocuments("UTF-8");

            tempFile = File.createTempFile("document", "xml");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, tempFile, true);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMFormattedWithSchema, tempFile, true);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, tempFile, false);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            XmlUtility.serialize(cbeDOMFormattedWithSchema, tempFile, false);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, tempFile, true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, tempFile, true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, tempFile, true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, tempFile, true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, tempFile, false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, tempFile, false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, tempFile, false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, tempFile, false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }

    /**
     * Test serialize(Document, File, boolean, String) API.
     */
    public void testSerialize_Document_File_boolean_String_API() {

        File tempFile = null;

        try {

            String encoding = "ISO-8859-1";
            
            createXMLDocuments(encoding);

            tempFile = File.createTempFile("document", "xml");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, tempFile, true, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMFormattedWithSchema, tempFile, true, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, tempFile, false, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            XmlUtility.serialize(cbeDOMFormattedWithSchema, tempFile, false, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, tempFile, true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, tempFile, true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, tempFile, true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, tempFile, true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, tempFile, false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, tempFile, false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, tempFile, false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, tempFile, false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }
    
    /**
     * Test serialize(Document, OutputStream) API.
     */
    public void testSerialize_Document_OutputStream_API() {

        File tempFile = null;

        try {

            createXMLDocuments("UTF-8");

            tempFile = File.createTempFile("document", "xml");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, new FileOutputStream(tempFile));

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMFormattedWithSchema, new FileOutputStream(tempFile));

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, new FileOutputStream(tempFile));

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, new FileOutputStream(tempFile));

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, new FileOutputStream(tempFile));

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, new FileOutputStream(tempFile));

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }

    /**
     * Test serialize(Document, OutputStream, boolean) API.
     */
    public void testSerialize_Document_OutputStream_boolean_API() {

        File tempFile = null;

        try {

            createXMLDocuments("UTF-8");

            tempFile = File.createTempFile("document", "xml");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, new FileOutputStream(tempFile), true);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMFormattedWithSchema, new FileOutputStream(tempFile), true);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, new FileOutputStream(tempFile), false);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            XmlUtility.serialize(cbeDOMFormattedWithSchema, new FileOutputStream(tempFile), false);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, new FileOutputStream(tempFile), true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, new FileOutputStream(tempFile), true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, new FileOutputStream(tempFile), true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, new FileOutputStream(tempFile), true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, new FileOutputStream(tempFile), false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, new FileOutputStream(tempFile), false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, new FileOutputStream(tempFile), false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, new FileOutputStream(tempFile), false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }

    /**
     * Test serialize(Document, OutputStream, boolean, String) API.
     */
    public void testSerialize_Document_OutputStream_boolean_String_API() {

        File tempFile = null;

        try {

            String encoding = "ISO-8859-1";
            
            createXMLDocuments(encoding);

            tempFile = File.createTempFile("document", "xml");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, new FileOutputStream(tempFile), true, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMFormattedWithSchema, new FileOutputStream(tempFile), true, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, new FileOutputStream(tempFile), false, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            XmlUtility.serialize(cbeDOMFormattedWithSchema, new FileOutputStream(tempFile), false, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, new FileOutputStream(tempFile), true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, new FileOutputStream(tempFile), true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, new FileOutputStream(tempFile), true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, new FileOutputStream(tempFile), true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, new FileOutputStream(tempFile), false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, new FileOutputStream(tempFile), false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, new FileOutputStream(tempFile), false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, new FileOutputStream(tempFile), false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }

    /**
     * Test serialize(Document, Writer) API.
     */
    public void testSerialize_Document_Writer_API() {

        File tempFile = null;

        try {

            String encoding = "UTF-8";
            
            createXMLDocuments(encoding);

            tempFile = File.createTempFile("document", "xml");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, new OutputStreamWriter(new FileOutputStream(tempFile),encoding));

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMFormattedWithSchema, new OutputStreamWriter(new FileOutputStream(tempFile),encoding));

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding));

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding));

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding));

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding));

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }

    /**
     * Test serialize(Document, Writer, boolean) API.
     */
    public void testSerialize_Document_Writer_boolean_API() {

        File tempFile = null;

        try {

            String encoding = "UTF-8";
            
            createXMLDocuments(encoding);

            tempFile = File.createTempFile("document", "xml");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMFormattedWithSchema, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            XmlUtility.serialize(cbeDOMFormattedWithSchema, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }
    
    /**
     * Test serialize(Document, Writer, boolean, String) API.
     */
    public void testSerialize_Document_Writer_boolean_String_API() {

        File tempFile = null;

        try {

            String encoding = "ISO-8859-1";
            
            createXMLDocuments(encoding);

            tempFile = File.createTempFile("document", "xml");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMFormattedWithSchema, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLFormattedWithSchema.toString());

            XmlUtility.serialize(cbeDOMNotFormattedWithSchema, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            XmlUtility.serialize(cbeDOMFormattedWithSchema, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false, encoding);

            assertEquals(utilities.getFile(tempFile), cbeXMLNotFormattedWithSchema);

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), true, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLFormattedNoDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);

            XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedWithDTD.toString());

            XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD, new OutputStreamWriter(new FileOutputStream(tempFile),encoding), false, encoding);

            assertEquals(utilities.getFile(tempFile), symptomDatabaseXMLNotFormattedNoDTD);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }

    /**
     * Test serializeAsByteArray(Document) API.
     */
    public void testSerializeAsByteArray_Document_API() {

        try {

            createXMLDocuments("UTF-8");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(cbeDOMNotFormattedWithSchema)), cbeXMLFormattedWithSchema.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(cbeDOMFormattedWithSchema)), cbeXMLFormattedWithSchema.toString());

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMNotFormattedWithDTD)), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMNotFormattedNoDTD)), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMFormattedWithDTD)), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMFormattedNoDTD)), symptomDatabaseXMLFormattedNoDTD.toString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        }
    }

    /**
     * Test serializeAsByteArray(Document, boolean) API.
     */
    public void testSerializeAsByteArray_Document_boolean_API() {

        try {

            createXMLDocuments("UTF-8");

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(cbeDOMNotFormattedWithSchema, true)), cbeXMLFormattedWithSchema.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(cbeDOMFormattedWithSchema, true)), cbeXMLFormattedWithSchema.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(cbeDOMNotFormattedWithSchema, false)), cbeXMLNotFormattedWithSchema);

            assertEquals(new String(XmlUtility.serializeAsByteArray(cbeDOMFormattedWithSchema, false)), cbeXMLNotFormattedWithSchema);

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMNotFormattedWithDTD, true)), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMNotFormattedNoDTD, true)), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMFormattedWithDTD, true)), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMFormattedNoDTD, true)), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMNotFormattedWithDTD, false)), symptomDatabaseXMLNotFormattedWithDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMNotFormattedNoDTD, false)), symptomDatabaseXMLNotFormattedNoDTD);

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMFormattedWithDTD, false)), symptomDatabaseXMLNotFormattedWithDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMFormattedNoDTD, false)), symptomDatabaseXMLNotFormattedNoDTD);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        }
    }


    /**
     * Test serializeAsByteArray(Document, boolean, String) API.
     */
    public void testSerializeAsByteArray_Document_boolean_String_API() {

        try {

            String encoding = "ISO-8859-1";
            
            createXMLDocuments(encoding);

            Document cbeDOMNotFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = utilities.createDocumentObjectModel(cbeXMLFormattedWithSchema.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(cbeDOMNotFormattedWithSchema, true, encoding)), cbeXMLFormattedWithSchema.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(cbeDOMFormattedWithSchema, true, encoding)), cbeXMLFormattedWithSchema.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(cbeDOMNotFormattedWithSchema, false, encoding)), cbeXMLNotFormattedWithSchema);

            assertEquals(new String(XmlUtility.serializeAsByteArray(cbeDOMFormattedWithSchema, false, encoding)), cbeXMLNotFormattedWithSchema);

            Document symptomDatabaseDOMNotFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = utilities.createDocumentObjectModel(symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMNotFormattedWithDTD, true, encoding)), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMNotFormattedNoDTD, true, encoding)), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMFormattedWithDTD, true, encoding)), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMFormattedNoDTD, true, encoding)), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMNotFormattedWithDTD, false, encoding)), symptomDatabaseXMLNotFormattedWithDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMNotFormattedNoDTD, false, encoding)), symptomDatabaseXMLNotFormattedNoDTD);

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMFormattedWithDTD, false, encoding)), symptomDatabaseXMLNotFormattedWithDTD.toString());

            assertEquals(new String(XmlUtility.serializeAsByteArray(symptomDatabaseDOMFormattedNoDTD, false, encoding)), symptomDatabaseXMLNotFormattedNoDTD);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        }
    }

    /**
     * Test deserialize(String) API.
     */
    public void testDeserialize_String_API() {

        try {

            createXMLDocuments("UTF-8");

            Document cbeDOMNotFormattedWithSchema = XmlUtility.deserialize(cbeXMLNotFormattedWithSchema);

            Document cbeDOMFormattedWithSchema = XmlUtility.deserialize(cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMNotFormattedWithSchema), cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMFormattedWithSchema), cbeXMLFormattedWithSchema.toString());

            Document symptomDatabaseDOMNotFormattedWithDTD = XmlUtility.deserialize(symptomDatabaseXMLNotFormattedWithDTD.toString());

            Document symptomDatabaseDOMNotFormattedNoDTD = XmlUtility.deserialize(symptomDatabaseXMLNotFormattedNoDTD);

            Document symptomDatabaseDOMFormattedWithDTD = XmlUtility.deserialize(symptomDatabaseXMLFormattedWithDTD.toString());

            Document symptomDatabaseDOMFormattedNoDTD = XmlUtility.deserialize(symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD), symptomDatabaseXMLFormattedNoDTD.toString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        }
    }

    /**
     * Test deserialize(File) API.
     */
    public void testDeserialize_File_API() {

        File tempFile = null;

        try {

            createXMLDocuments("UTF-8");

            tempFile = File.createTempFile("document", "xml");

            utilities.createFile(cbeXMLNotFormattedWithSchema, tempFile);

            Document cbeDOMNotFormattedWithSchema = XmlUtility.deserialize(tempFile);

            utilities.createFile(cbeXMLFormattedWithSchema.toString(), tempFile);

            Document cbeDOMFormattedWithSchema = XmlUtility.deserialize(tempFile);

            assertEquals(XmlUtility.serialize(cbeDOMNotFormattedWithSchema), cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMFormattedWithSchema), cbeXMLFormattedWithSchema.toString());

            utilities.createFile(symptomDatabaseXMLNotFormattedWithDTD.toString(), tempFile);

            Document symptomDatabaseDOMNotFormattedWithDTD = XmlUtility.deserialize(tempFile);

            utilities.createFile(symptomDatabaseXMLNotFormattedNoDTD, tempFile);

            Document symptomDatabaseDOMNotFormattedNoDTD = XmlUtility.deserialize(tempFile);

            utilities.createFile(symptomDatabaseXMLFormattedWithDTD.toString(), tempFile);

            Document symptomDatabaseDOMFormattedWithDTD = XmlUtility.deserialize(tempFile);

            utilities.createFile(symptomDatabaseXMLFormattedNoDTD.toString(), tempFile);

            Document symptomDatabaseDOMFormattedNoDTD = XmlUtility.deserialize(tempFile);

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD), symptomDatabaseXMLFormattedNoDTD.toString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }

    /**
     * Test deserialize(InputSource) API.
     */
    public void testDeserialize_InputSource_API() {

        File tempFile = null;

        try {
            
            createXMLDocuments("UTF-8");

            tempFile = File.createTempFile("document", "xml");

            utilities.createFile(cbeXMLNotFormattedWithSchema, tempFile);

            Document cbeDOMNotFormattedWithSchema = XmlUtility.deserialize(new InputSource(new FileInputStream(tempFile)));

            utilities.createFile(cbeXMLFormattedWithSchema.toString(), tempFile);

            Document cbeDOMFormattedWithSchema = XmlUtility.deserialize(new InputSource(new FileInputStream(tempFile)));

            assertEquals(XmlUtility.serialize(cbeDOMNotFormattedWithSchema), cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMFormattedWithSchema), cbeXMLFormattedWithSchema.toString());

            utilities.createFile(symptomDatabaseXMLNotFormattedWithDTD.toString(), tempFile);

            Document symptomDatabaseDOMNotFormattedWithDTD = XmlUtility.deserialize(new InputSource(new FileInputStream(tempFile)));

            utilities.createFile(symptomDatabaseXMLNotFormattedNoDTD, tempFile);

            Document symptomDatabaseDOMNotFormattedNoDTD = XmlUtility.deserialize(new InputSource(new FileInputStream(tempFile)));

            utilities.createFile(symptomDatabaseXMLFormattedWithDTD.toString(), tempFile);

            Document symptomDatabaseDOMFormattedWithDTD = XmlUtility.deserialize(new InputSource(new FileInputStream(tempFile)));

            utilities.createFile(symptomDatabaseXMLFormattedNoDTD.toString(), tempFile);

            Document symptomDatabaseDOMFormattedNoDTD = XmlUtility.deserialize(new InputSource(new FileInputStream(tempFile)));

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD), symptomDatabaseXMLFormattedNoDTD.toString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }

    /**
     * Test deserialize(InputStream) API.
     */
    public void testDeserialize_InputStream_API() {

        File tempFile = null;

        try {
       
            createXMLDocuments("UTF-8");

            tempFile = File.createTempFile("document", "xml");

            utilities.createFile(cbeXMLNotFormattedWithSchema, tempFile);

            Document cbeDOMNotFormattedWithSchema = XmlUtility.deserialize(new FileInputStream(tempFile));

            utilities.createFile(cbeXMLFormattedWithSchema.toString(), tempFile);

            Document cbeDOMFormattedWithSchema = XmlUtility.deserialize(new FileInputStream(tempFile));

            assertEquals(XmlUtility.serialize(cbeDOMNotFormattedWithSchema), cbeXMLFormattedWithSchema.toString());

            assertEquals(XmlUtility.serialize(cbeDOMFormattedWithSchema), cbeXMLFormattedWithSchema.toString());

            utilities.createFile(symptomDatabaseXMLNotFormattedWithDTD.toString(), tempFile);

            Document symptomDatabaseDOMNotFormattedWithDTD = XmlUtility.deserialize(new FileInputStream(tempFile));

            utilities.createFile(symptomDatabaseXMLNotFormattedNoDTD, tempFile);

            Document symptomDatabaseDOMNotFormattedNoDTD = XmlUtility.deserialize(new FileInputStream(tempFile));

            utilities.createFile(symptomDatabaseXMLFormattedWithDTD.toString(), tempFile);

            Document symptomDatabaseDOMFormattedWithDTD = XmlUtility.deserialize(new FileInputStream(tempFile));

            utilities.createFile(symptomDatabaseXMLFormattedNoDTD.toString(), tempFile);

            Document symptomDatabaseDOMFormattedNoDTD = XmlUtility.deserialize(new FileInputStream(tempFile));

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedWithDTD), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMNotFormattedNoDTD), symptomDatabaseXMLFormattedNoDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedWithDTD), symptomDatabaseXMLFormattedWithDTD.toString());

            assertEquals(XmlUtility.serialize(symptomDatabaseDOMFormattedNoDTD), symptomDatabaseXMLFormattedNoDTD.toString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            fail(t.toString());
        } finally {

            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }   
    private void createXMLDocuments(String encoding){
        
          //Xerces uses the Web line separator character (e.g. \n) in CDATA
          // elements whereas
          //JAXP uses the platform dependent line separator character(s) (e.g.
          // \n\r on Windows).
          String cdataElementNewLine = Constants.LINE_SEPARATOR;

          //JAXP uses the decimal representation of entity reference values (e.g.
          // &#10; for
          //the Web line separator character) in attribute values (e.g. CDATA)
          // whereas
          //Xerces uses the hexadecimal representation of entity reference values
          // (e.g.
          //&#xa; for the Web line separator character).
          //Xerces resolves entity references whereas JAXP persists reference
          // values.
          String newLineEntityReference = "&#10;";
                  
          //Xerces does NOT format CDATA elements relative to their parent
          // element whereas JAXP
          //does format CDATA elements relative to their parent element.
          String cdataElementFormatting = Constants.LINE_SEPARATOR + "    ";
          
          //Test is JAXP is available:
          try {
              TransformerFactory.newInstance();
          } 
          catch (Throwable e) {
              
              cdataElementNewLine = "\n";
              
              //Xerces resolves entity references whereas JAXP persists reference
              //values so test a space.
              newLineEntityReference = " ";
              
              cdataElementFormatting = "";
          }

          //Test is Crimson is available:
          boolean isCrimsonAvailable = false;
          
          try {
              Thread.currentThread().getContextClassLoader().loadClass("org.apache.crimson.parser.XMLReaderImpl");
              
              isCrimsonAvailable = true;
          } 
          catch (ClassNotFoundException c) {
              //Ignore since Crimson is not available.
          }   
          
          cbeXMLNotFormattedWithSchema = "<?xml version=\"1.0\" encoding=\"" + encoding + "\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\"><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A1\" version=\"1.0.1\"><contextDataElements name=\"myContext\" type=\"String\"><contextValue>contextValue</contextValue></contextDataElements><extendedDataElements name=\"ede1\" type=\"noValue\"><values>values</values><children name=\"chname1\" type=\"noValue\"><values>values</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"noValue\"><values>values</values><children name=\"chname2\" type=\"noValue\"><values>values</values></children></extendedDataElements><reporterComponentId application=\"appl1\" component=\"comp1\" componentIdType=\"compIdType1\" componentType=\"compType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/><sourceComponentId application=\"app2\" component=\"comp2\" componentIdType=\"compIdType2\" componentType=\"compType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/><msgDataElement msgLocale=\"EN\"><msgCatalogTokens value=\"value\"/><msgId>msgId</msgId><msgIdType>Name</msgIdType><msgCatalogId>msgCatalogId</msgCatalogId><msgCatalogType>msgCatalogType</msgCatalogType><msgCatalog>msgCatalog</msgCatalog></msgDataElement><situation categoryName=\"StartSituation\"><situationType reasoningScope=\"EXTERNAL\" situationQualifier=\"START COMPLETED\" successDisposition=\"SUCCESSFUL\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:type=\"StartSituation\"/></situation></CommonBaseEvent><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A1\" version=\"1.0.1\"><contextDataElements name=\"myContext\" type=\"String\"><contextValue>contextValue</contextValue></contextDataElements><extendedDataElements name=\"ede1\" type=\"noValue\"><values>values</values><children name=\"chname1\" type=\"noValue\"><values>values</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"noValue\"><values>values</values><children name=\"chname2\" type=\"noValue\"><values>values</values></children></extendedDataElements><reporterComponentId application=\"appl1\" component=\"comp1\" componentIdType=\"compIdType1\" componentType=\"compType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/><sourceComponentId application=\"app2\" component=\"comp2\" componentIdType=\"compIdType2\" componentType=\"compType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/><msgDataElement msgLocale=\"EN\"><msgCatalogTokens value=\"value\"/><msgId>msgId</msgId><msgIdType>Name</msgIdType><msgCatalogId>msgCatalogId</msgCatalogId><msgCatalogType>msgCatalogType</msgCatalogType><msgCatalog>msgCatalog</msgCatalog></msgDataElement><situation categoryName=\"StartSituation\"><situationType reasoningScope=\"EXTERNAL\" situationQualifier=\"START COMPLETED\" successDisposition=\"SUCCESSFUL\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:type=\"StartSituation\"/></situation></CommonBaseEvent><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A1\" version=\"1.0.1\"><contextDataElements name=\"myContext\" type=\"String\"><contextValue>contextValue</contextValue></contextDataElements><extendedDataElements name=\"ede1\" type=\"noValue\"><values>values</values><children name=\"chname1\" type=\"noValue\"><values>values</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"noValue\"><values>values</values><children name=\"chname2\" type=\"noValue\"><values>values</values></children></extendedDataElements><reporterComponentId application=\"appl1\" component=\"comp1\" componentIdType=\"compIdType1\" componentType=\"compType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/><sourceComponentId application=\"app2\" component=\"comp2\" componentIdType=\"compIdType2\" componentType=\"compType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/><msgDataElement msgLocale=\"EN\"><msgCatalogTokens value=\"value\"/><msgId>msgId</msgId><msgIdType>Name</msgIdType><msgCatalogId>msgCatalogId</msgCatalogId><msgCatalogType>msgCatalogType</msgCatalogType><msgCatalog>msgCatalog</msgCatalog></msgDataElement><situation categoryName=\"StartSituation\"><situationType reasoningScope=\"EXTERNAL\" situationQualifier=\"START COMPLETED\" successDisposition=\"SUCCESSFUL\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:type=\"StartSituation\"/></situation></CommonBaseEvent></CommonBaseEvents>";

          cbeXMLFormattedWithSchema = new StringBuffer();

          cbeXMLFormattedWithSchema.append("<?xml version=\"1.0\" encoding=\"" + encoding + "\"?>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("  <CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A1\" version=\"1.0.1\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <contextDataElements name=\"myContext\" type=\"String\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <contextValue>contextValue</contextValue>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </contextDataElements>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <extendedDataElements name=\"ede1\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <children name=\"chname1\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("        <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      </children>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </extendedDataElements>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <extendedDataElements name=\"ede2\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <children name=\"chname2\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("        <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      </children>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </extendedDataElements>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <reporterComponentId application=\"appl1\" component=\"comp1\" componentIdType=\"compIdType1\" componentType=\"compType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <sourceComponentId application=\"app2\" component=\"comp2\" componentIdType=\"compIdType2\" componentType=\"compType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <msgDataElement msgLocale=\"EN\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalogTokens value=\"value\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgId>msgId</msgId>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgIdType>Name</msgIdType>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalogId>msgCatalogId</msgCatalogId>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalogType>msgCatalogType</msgCatalogType>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalog>msgCatalog</msgCatalog>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </msgDataElement>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <situation categoryName=\"StartSituation\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <situationType reasoningScope=\"EXTERNAL\" situationQualifier=\"START COMPLETED\" successDisposition=\"SUCCESSFUL\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:type=\"StartSituation\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </situation>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("  </CommonBaseEvent>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("  <CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A1\" version=\"1.0.1\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <contextDataElements name=\"myContext\" type=\"String\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <contextValue>contextValue</contextValue>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </contextDataElements>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <extendedDataElements name=\"ede1\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <children name=\"chname1\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("        <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      </children>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </extendedDataElements>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <extendedDataElements name=\"ede2\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <children name=\"chname2\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("        <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      </children>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </extendedDataElements>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <reporterComponentId application=\"appl1\" component=\"comp1\" componentIdType=\"compIdType1\" componentType=\"compType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <sourceComponentId application=\"app2\" component=\"comp2\" componentIdType=\"compIdType2\" componentType=\"compType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <msgDataElement msgLocale=\"EN\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalogTokens value=\"value\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgId>msgId</msgId>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgIdType>Name</msgIdType>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalogId>msgCatalogId</msgCatalogId>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalogType>msgCatalogType</msgCatalogType>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalog>msgCatalog</msgCatalog>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </msgDataElement>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <situation categoryName=\"StartSituation\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <situationType reasoningScope=\"EXTERNAL\" situationQualifier=\"START COMPLETED\" successDisposition=\"SUCCESSFUL\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:type=\"StartSituation\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </situation>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("  </CommonBaseEvent>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("  <CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A1\" version=\"1.0.1\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <contextDataElements name=\"myContext\" type=\"String\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <contextValue>contextValue</contextValue>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </contextDataElements>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <extendedDataElements name=\"ede1\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <children name=\"chname1\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("        <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      </children>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </extendedDataElements>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <extendedDataElements name=\"ede2\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <children name=\"chname2\" type=\"noValue\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("        <values>values</values>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      </children>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </extendedDataElements>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <reporterComponentId application=\"appl1\" component=\"comp1\" componentIdType=\"compIdType1\" componentType=\"compType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <sourceComponentId application=\"app2\" component=\"comp2\" componentIdType=\"compIdType2\" componentType=\"compType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <msgDataElement msgLocale=\"EN\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalogTokens value=\"value\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgId>msgId</msgId>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgIdType>Name</msgIdType>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalogId>msgCatalogId</msgCatalogId>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalogType>msgCatalogType</msgCatalogType>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <msgCatalog>msgCatalog</msgCatalog>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </msgDataElement>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    <situation categoryName=\"StartSituation\">");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("      <situationType reasoningScope=\"EXTERNAL\" situationQualifier=\"START COMPLETED\" successDisposition=\"SUCCESSFUL\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:type=\"StartSituation\"/>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("    </situation>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("  </CommonBaseEvent>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);
          cbeXMLFormattedWithSchema.append("</CommonBaseEvents>");
          cbeXMLFormattedWithSchema.append(Constants.LINE_SEPARATOR);

          symptomDatabaseXMLNotFormattedWithDTD = new StringBuffer();

          symptomDatabaseXMLNotFormattedWithDTD.append("<?xml version=\"1.0\" encoding=\"" + encoding + "\"?>");

          //NOTE: The Sun JDK v.1.4.x that includes the Crimson XML parser
          // implementation
          //does not support Document Type Definition (DTD) in the Document
          // Object Model (DOM).
          if (!isCrimsonAvailable) {
              symptomDatabaseXMLNotFormattedWithDTD.append("<!DOCTYPE symptomDatabase [<!ELEMENT symptomDatabase (runtime+)><!ELEMENT runtime (symptom+,solution+,directive*)><!ATTLIST runtime id ID #REQUIRED><!ATTLIST runtime name CDATA #IMPLIED><!ATTLIST runtime symptomUrl CDATA #IMPLIED><!ATTLIST runtime localExternalFileLocation CDATA #IMPLIED><!ELEMENT symptom (matchPattern+)><!ATTLIST symptom id ID #REQUIRED><!ATTLIST symptom description CDATA #IMPLIED><!ATTLIST symptom solutions IDREFS #REQUIRED><!ELEMENT matchPattern EMPTY><!ATTLIST matchPattern name CDATA #IMPLIED><!ATTLIST matchPattern value CDATA #REQUIRED><!ELEMENT solution EMPTY><!ATTLIST solution id ID #REQUIRED><!ATTLIST solution description CDATA #IMPLIED><!ATTLIST solution directives IDREFS #IMPLIED><!ELEMENT directive EMPTY><!ATTLIST directive id ID #REQUIRED><!ATTLIST directive description CDATA #IMPLIED><!ATTLIST directive directiveString CDATA #REQUIRED>]>");
          }

          symptomDatabaseXMLNotFormattedWithDTD.append("<!--  Created on Wednesday, July 28, 2004 at 8:17:54:0963 AM EDT --><symptomDatabase><runtime id=\"Runtime_0\" localExternalFileLocation=\"\" name=\"platform:/resource/LogAnalyzerProject/symptomdb.trcdbxmi\" symptomUrl=\"ftp://ftp.software.ibm.com/software/websphere/info/tools/loganalyzer/symptoms/std/symptomdb.xml\"><!-- Symptoms: --><symptom description=\"ADMR3028E\" id=\"Symptom_0\" solutions=\"Solution_0\"><matchPattern name=\"0\" value=\"ADMR3028E\"/></symptom><!-- Solutions: --><![CDATA[");
          symptomDatabaseXMLNotFormattedWithDTD.append(cdataElementNewLine);
          symptomDatabaseXMLNotFormattedWithDTD.append("        This is a CDATA element");
          symptomDatabaseXMLNotFormattedWithDTD.append(cdataElementNewLine);
          symptomDatabaseXMLNotFormattedWithDTD.append("        	with white space!");
          symptomDatabaseXMLNotFormattedWithDTD.append(cdataElementNewLine);
          symptomDatabaseXMLNotFormattedWithDTD.append("    ]]><solution description=\"This error can occur when an application that uses a jdbc source is copied or cloned from a server that has the jdbc driver installed onto one that does not.  This means that WebSphere does not know where to look for the physical jdbc driver file on the failing node.\" directives=\"Directive_0\" id=\"Solution_0\"/><!-- Directives: --><directive description=\"The resolution is to select the specified data source (listed in the error message after &quot;provider named&quot;) in the WebSphere topology under &quot;jdbc providers&quot;, then select the &quot;nodes&quot; tab, click &quot;Install new&quot;, select the problem node, and use the file dialog to specify the jdbc driver file (such as db2java.zip or other).");
          symptomDatabaseXMLNotFormattedWithDTD.append(newLineEntityReference);
          symptomDatabaseXMLNotFormattedWithDTD.append("ADMR3028Edbdrvr wsae40\" directiveString=\"\" id=\"Directive_0\"/></runtime></symptomDatabase>");

          symptomDatabaseXMLNotFormattedNoDTD = "<?xml version=\"1.0\" encoding=\"" + encoding + "\"?><!--  Created on Wednesday, July 28, 2004 at 8:17:54:0963 AM EDT --><symptomDatabase><runtime id=\"Runtime_0\" localExternalFileLocation=\"\" name=\"platform:/resource/LogAnalyzerProject/symptomdb.trcdbxmi\" symptomUrl=\"ftp://ftp.software.ibm.com/software/websphere/info/tools/loganalyzer/symptoms/std/symptomdb.xml\"><!-- Symptoms: --><symptom description=\"ADMR3028E\" id=\"Symptom_0\" solutions=\"Solution_0\"><matchPattern name=\"0\" value=\"ADMR3028E\"/></symptom><!-- Solutions: --><![CDATA[" + cdataElementNewLine + "        This is a CDATA element" + cdataElementNewLine + "        	with white space!" + cdataElementNewLine + "    ]]><solution description=\"This error can occur when an application that uses a jdbc source is copied or cloned from a server that has the jdbc driver installed onto one that does not.  This means that WebSphere does not know where to look for the physical jdbc driver file on the failing node.\" directives=\"Directive_0\" id=\"Solution_0\"/><!-- Directives: --><directive description=\"The resolution is to select the specified data source (listed in the error message after &quot;provider named&quot;) in the WebSphere topology under &quot;jdbc providers&quot;, then select the &quot;nodes&quot; tab, click &quot;Install new&quot;, select the problem node, and use the file dialog to specify the jdbc driver file (such as db2java.zip or other)." + newLineEntityReference + "ADMR3028Edbdrvr wsae40\" directiveString=\"\" id=\"Directive_0\"/></runtime></symptomDatabase>";

          symptomDatabaseXMLFormattedWithDTD = new StringBuffer();

          symptomDatabaseXMLFormattedWithDTD.append("<?xml version=\"1.0\" encoding=\"" + encoding + "\"?>");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);

          //NOTE: The Sun JDK v.1.4.x that includes the Crimson XML parser
          // implementation
          //does not support Document Type Definition (DTD) in the Document
          // Object Model (DOM).
          if (!isCrimsonAvailable) {

              symptomDatabaseXMLFormattedWithDTD.append("<!DOCTYPE symptomDatabase [");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ELEMENT symptomDatabase (runtime+)>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ELEMENT runtime (symptom+,solution+,directive*)>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST runtime id ID #REQUIRED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST runtime name CDATA #IMPLIED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST runtime symptomUrl CDATA #IMPLIED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST runtime localExternalFileLocation CDATA #IMPLIED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ELEMENT symptom (matchPattern+)>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST symptom id ID #REQUIRED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST symptom description CDATA #IMPLIED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST symptom solutions IDREFS #REQUIRED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ELEMENT matchPattern EMPTY>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST matchPattern name CDATA #IMPLIED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST matchPattern value CDATA #REQUIRED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ELEMENT solution EMPTY>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST solution id ID #REQUIRED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST solution description CDATA #IMPLIED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST solution directives IDREFS #IMPLIED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ELEMENT directive EMPTY>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST directive id ID #REQUIRED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST directive description CDATA #IMPLIED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("  <!ATTLIST directive directiveString CDATA #REQUIRED>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
              symptomDatabaseXMLFormattedWithDTD.append("]>");
              symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          }

          symptomDatabaseXMLFormattedWithDTD.append("<!--  Created on Wednesday, July 28, 2004 at 8:17:54:0963 AM EDT -->");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("<symptomDatabase>");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("  <runtime id=\"Runtime_0\" localExternalFileLocation=\"\" name=\"platform:/resource/LogAnalyzerProject/symptomdb.trcdbxmi\" symptomUrl=\"ftp://ftp.software.ibm.com/software/websphere/info/tools/loganalyzer/symptoms/std/symptomdb.xml\">");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("    <!-- Symptoms: -->");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("    <symptom description=\"ADMR3028E\" id=\"Symptom_0\" solutions=\"Solution_0\">");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("      <matchPattern name=\"0\" value=\"ADMR3028E\"/>");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("    </symptom>");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("    <!-- Solutions: -->");
          symptomDatabaseXMLFormattedWithDTD.append("<![CDATA[");
          symptomDatabaseXMLFormattedWithDTD.append(cdataElementNewLine);
          symptomDatabaseXMLFormattedWithDTD.append("        This is a CDATA element");
          symptomDatabaseXMLFormattedWithDTD.append(cdataElementNewLine);
          symptomDatabaseXMLFormattedWithDTD.append("        	with white space!");
          symptomDatabaseXMLFormattedWithDTD.append(cdataElementNewLine);
          symptomDatabaseXMLFormattedWithDTD.append("    ]]>");
          symptomDatabaseXMLFormattedWithDTD.append(cdataElementFormatting);
          symptomDatabaseXMLFormattedWithDTD.append("<solution description=\"This error can occur when an application that uses a jdbc source is copied or cloned from a server that has the jdbc driver installed onto one that does not.  This means that WebSphere does not know where to look for the physical jdbc driver file on the failing node.\" directives=\"Directive_0\" id=\"Solution_0\"/>");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("    <!-- Directives: -->");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("    <directive description=\"The resolution is to select the specified data source (listed in the error message after &quot;provider named&quot;) in the WebSphere topology under &quot;jdbc providers&quot;, then select the &quot;nodes&quot; tab, click &quot;Install new&quot;, select the problem node, and use the file dialog to specify the jdbc driver file (such as db2java.zip or other).");
          symptomDatabaseXMLFormattedWithDTD.append(newLineEntityReference);
          symptomDatabaseXMLFormattedWithDTD.append("ADMR3028Edbdrvr wsae40\" directiveString=\"\" id=\"Directive_0\"/>");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("  </runtime>");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedWithDTD.append("</symptomDatabase>");
          symptomDatabaseXMLFormattedWithDTD.append(Constants.LINE_SEPARATOR);

          symptomDatabaseXMLFormattedNoDTD = new StringBuffer();

          symptomDatabaseXMLFormattedNoDTD.append("<?xml version=\"1.0\" encoding=\"" + encoding + "\"?>");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("<!--  Created on Wednesday, July 28, 2004 at 8:17:54:0963 AM EDT -->");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("<symptomDatabase>");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("  <runtime id=\"Runtime_0\" localExternalFileLocation=\"\" name=\"platform:/resource/LogAnalyzerProject/symptomdb.trcdbxmi\" symptomUrl=\"ftp://ftp.software.ibm.com/software/websphere/info/tools/loganalyzer/symptoms/std/symptomdb.xml\">");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("    <!-- Symptoms: -->");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("    <symptom description=\"ADMR3028E\" id=\"Symptom_0\" solutions=\"Solution_0\">");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("      <matchPattern name=\"0\" value=\"ADMR3028E\"/>");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("    </symptom>");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("    <!-- Solutions: -->");
          symptomDatabaseXMLFormattedNoDTD.append("<![CDATA[");
          symptomDatabaseXMLFormattedNoDTD.append(cdataElementNewLine);
          symptomDatabaseXMLFormattedNoDTD.append("        This is a CDATA element");
          symptomDatabaseXMLFormattedNoDTD.append(cdataElementNewLine);
          symptomDatabaseXMLFormattedNoDTD.append("        	with white space!");
          symptomDatabaseXMLFormattedNoDTD.append(cdataElementNewLine);
          symptomDatabaseXMLFormattedNoDTD.append("    ]]>");
          symptomDatabaseXMLFormattedNoDTD.append(cdataElementFormatting);
          symptomDatabaseXMLFormattedNoDTD.append("<solution description=\"This error can occur when an application that uses a jdbc source is copied or cloned from a server that has the jdbc driver installed onto one that does not.  This means that WebSphere does not know where to look for the physical jdbc driver file on the failing node.\" directives=\"Directive_0\" id=\"Solution_0\"/>");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("    <!-- Directives: -->");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("    <directive description=\"The resolution is to select the specified data source (listed in the error message after &quot;provider named&quot;) in the WebSphere topology under &quot;jdbc providers&quot;, then select the &quot;nodes&quot; tab, click &quot;Install new&quot;, select the problem node, and use the file dialog to specify the jdbc driver file (such as db2java.zip or other).");
          symptomDatabaseXMLFormattedNoDTD.append(newLineEntityReference);
          symptomDatabaseXMLFormattedNoDTD.append("ADMR3028Edbdrvr wsae40\" directiveString=\"\" id=\"Directive_0\"/>");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("  </runtime>");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
          symptomDatabaseXMLFormattedNoDTD.append("</symptomDatabase>");
          symptomDatabaseXMLFormattedNoDTD.append(Constants.LINE_SEPARATOR);
      }
}