package org.eclipse.hyades.logging.adapter;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import org.eclipse.hyades.logging.adapter.impl.Controller;
import org.eclipse.hyades.logging.adapter.impl.Status;
import org.eclipse.hyades.logging.adapter.util.Messages;

/**
 * Adapter is a simple main entry point to run an adapter controller.
 */
public class Adapter {

    private Controller rootController = null;
    private String contextFile = null;
    private String componentFile = null;
    // Adapter internal log outputter
    private IOutputter logOutputter = null;
    
    /*
     * Status object representing the status of this adapter instance.
     */
    private Status status = new Status();
    /**
     * Sets the file path of the context configuration portion of the 
     * adapter configuration. This method must be called before the start
     * method in order for the configuration to be used.
     * @param configFilePath The file path of the context configuration
     */
    public void setContextConfigPath(String configFilePath) {
        contextFile = configFilePath;
    }

    /**
     * Sets the file path of the component configuration portion of the 
     * adapter configuration. This method must be called before the start
     * method in order for the configuration to be used.
     * @param configFilePath The file path of the component configuration
     */
    public void setComponentConfigPath(String configFilePath) {
        componentFile = configFilePath;
    }
    
    /**
     * Setup the adapter and validate the configuration file.  This method
     * will not run the adapter and stops after validation.
     * @throws AdapterException if the configuration is invalid
     */
    public void validate() throws AdapterException {
    	rootController = new Controller();

        if (componentFile == null) {
            componentFile = Messages.getString("HyadesGADefaultComponentConfigurationsFile"); //$NON-NLS-1$
        }

        if (contextFile == null) {
            contextFile = Messages.getString("HyadesGADefaultContextConfigurationFile"); //$NON-NLS-1$
        }

        rootController.setContextConfigPath(contextFile);
        rootController.setComponentConfigPath(componentFile);
        
        /* try and prepare the adapter for running.  This may fail */
        rootController.prepareAdapter();
    	
    }
    
    /**
     * Starts the generic log adapter with the default configuration file SimpleComponentConfigurations.xml
     * unless the configuration files were specified with setContextConfigPath and setComponentConfigPath methods.
     * @param separateThread Run the adapter in a separate thread
     * @param daemon Run the adapter as a daemon thread.
     * @throws AdapterException If an error occurs during execution.
     */
    public void start(boolean separateThread, boolean daemon) throws AdapterException {

        rootController = new Controller();

        if (componentFile == null) {
            componentFile = Messages.getString("HyadesGADefaultComponentConfigurationsFile"); //$NON-NLS-1$
        }

        if (contextFile == null) {
            contextFile = Messages.getString("HyadesGADefaultContextConfigurationFile"); //$NON-NLS-1$
        }

        rootController.setContextConfigPath(contextFile);
        rootController.setComponentConfigPath(componentFile);

        rootController.setLogOutputter(logOutputter);
        
        /* try and prepare the adapter for running.  This may fail */
        rootController.prepareAdapter();

        if (!separateThread) {
            rootController.run();
            // All contexts have stopped so set the active status to false
            status.setActive(false);
        }
        else {
            Thread serverThread = new Thread(rootController);
            serverThread.setDaemon(daemon);
            serverThread.start();
            // Started the controller so set the active status ot true
            status.setActive(true);
        }
    }
    
    /**
     * Stops all of the adapter contexts.  Note any data still being processed
     * by the contexts will be flushed to the last component of each context.
     *
     */
    public void stop() {
        if (rootController != null) {
            rootController.stop();
        }
        status.setActive(false);
    }
	/**
	 * Stops all of the adapter contexts.  This stop method will still cause the contexts to be 
	 * flushed but will not call the context listener methods during flushing.
	 */
	public void hardStop() {
		if (rootController != null) {
			rootController.hardStop();
		}
		status.setActive(false);
	}
	
	/**
	 * Adapter main method to run the adapter as a standalone program.
	 * @param args -cc contextConfigurationPath -ac componentConfigurationPath
	 */
    public static void main(String[] args) {

        Adapter adapter = new Adapter();

        if (adapter.parseCommandLine(args)) {
            try {
                adapter.start(false, false);
            }
            catch (AdapterException e) {
            	/* RKD:  If we get an exception at this point we likely do not have a valid configuration and hense
            	 * the log file is likely not flushed.  We will need to write this to stderr.
            	 */
            	System.err.println(Messages.getString("HyadesGAAdapterFatal_ERROR_"));
            	System.err.println(e.getMessage());
                adapter.stop();
            }
        }
        else {
            System.out.println(Messages.getString("HyadesGACommand_line_usage"));
        }
    }

    /**
     * Parse the command line arguments
     * @param args command line arguments
     * @return Whether the command line was parsed successfully
     */
    private boolean parseCommandLine(String[] args) {

        if ((args != null) && (args.length > 0)) {

            String ccArgName = Messages.getString("HyadesGA-cc");
            String acArgName = Messages.getString("HyadesGA-ac");

            //Process all arguements:
            for (int i = 0; i < args.length; i++) {

                //If this is the -cc arguement, derive the context configuration:
                if (ccArgName.equalsIgnoreCase(args[i])) {

                    if ((++i < args.length) && (args[i] != null)) {
                    	setContextConfigPath(args[i]);
                    }
					else{
						return false;
					}
                }

                //If this is the -ac arguement, derive the the component configuration:
                else if (acArgName.equalsIgnoreCase(args[i])) {

                    if ((++i < args.length) && (args[i] != null)) {
                    	setComponentConfigPath(args[i]);   
                    }
                    else{
                    	return false;
                    }
                }

                //If there is any other argument, fail:
                else {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Gets the status of the adapter instance.
     * @return the org.eclipse.hyades.logging.adapter.IStatus object representing the status.
     */
    public IStatus getStatus() {
    	status.setChildrenStatus(rootController.getStatus());
    	return status;
    }
    
    /**
     * Set the Adapter log outputter for GLA messages
     * @param outputter component
     */
    public void setLogOutputter(IOutputter outputter) {
    	logOutputter = outputter;
    }
}