/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.internal.util;

import java.lang.reflect.Array;

/**
 * Please see description in the IEventQueue interface.
 */
public class EventQueue {
	public static int DEFAULT_QUEUE_LENGTH = 10; // can queue max 10 records
	private Object lock;
	private Object[] items;
	private int queueLength; // total queue length
	private int firstFreeSlot; // pointer to the first available slot for storage
	private boolean shuttingDown = false; // indicated whether a user-initiated shutdown has occurred

	/**
	 * Constructs a EventQueue with a default length
	 *
	 */
	public EventQueue() {
		this(DEFAULT_QUEUE_LENGTH);
	}

	/**
	 * Constructs a EventQueue with a specified length
	 * @param length
	 */
	public EventQueue(int length) {
		lock = new Object();
		//items = new Object[length]; /* do not create the array here because the class type is unknown */
		queueLength = length;
		firstFreeSlot = 0;
	}

	/**
	 * Add new items into the queue
	 * @param newItem Array of new items to be added to the queue
	 * @throws InterruptedException
	 */
	public void addItem(Object[] newItem) throws InterruptedException {
		int entriesToCopy = 0;

		if(newItem != null) {
			entriesToCopy = newItem.length;
		}

		synchronized(lock) {
			// Create the array if it is null/empty
			if(items == null) {
				items = (Object[])Array.newInstance(newItem[0].getClass(), queueLength); // we can get the class type here
				firstFreeSlot = 0;
			}

			// Make sure all entries are added to the queue
			while(!shuttingDown && (entriesToCopy > 0)) {
				// Wait until there is at least one free slot
				if(freeSpace() == 0) {
					try {
						lock.wait();
					} catch(InterruptedException e) {
						throw new InterruptedException("Queue is full");
					}
				}
				else {
					// Copy the item into the slot
					items[firstFreeSlot] = newItem[newItem.length - entriesToCopy];
					firstFreeSlot++; // move the free slot pointer
					entriesToCopy--; // finished one item
				}
			}

			// Notify any blocking get
			lock.notifyAll();
		}
	}

	/**
	 * Get all the items in the queue
	 * @return Array of objects stored in the queue
	 * @throws InterruptedException
	 */
	public Object[] getItem() throws InterruptedException {
		Object[] rc = null;

		synchronized(lock) {
			// Wait until there is some data to retrieve
			while(!shuttingDown && isEmpty()) {
				try {
					lock.wait(); // wait indefinitely
				} catch(InterruptedException e) {
					throw new InterruptedException("Queue is empty");
				}
			}

			if(!shuttingDown) {
				rc = (Object[])Array.newInstance(items[0].getClass(), size()); // only return non-null entries
				for(int i = 0; i < size(); i++) {
					rc[i] = items[i];
				}
				clear(); // empty the queue
			}

			// Notify any blocking add
			lock.notifyAll();

			return rc;
		}
	}

	public boolean isFull() {
		return (firstFreeSlot == queueLength);
	}

	public boolean isEmpty() {
		return (firstFreeSlot == 0);
	}

	public int size() {
		return firstFreeSlot;
	}

	public int freeSpace() {
		return (queueLength - firstFreeSlot);
	}

	public void flush() throws InterruptedException {
		synchronized(lock) {
			if(!isEmpty()) {
				lock.wait(); // do not need a loop because the get operation will empty the whole array
			}
			shuttingDown = true;
			lock.notifyAll();
		}
	}

	private void clear() {
		for(int i = 0; i < items.length; i++) {
			items[i] = null;
		}
		firstFreeSlot = 0;
	}
}
