package org.eclipse.hyades.logging.adapter.outputters;

import java.io.FileWriter;
import java.io.IOException;
import java.util.Hashtable;

import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * A specialization of the <code>CBEFileOutputter</code> for atomically
 * writing Common Base Events from multiple contexts to a single output file.
 * <p>
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @version July 14, 2004
 * @since April 23, 2004
 * @see org.eclipse.hyades.logging.adapter.outputters.CBEFileOutputter
 */
public class CBEConvergentFileOutputter extends CBEFileOutputter {
    
    /**
     * Static cache (e.g. <code>java.util.Hashtable</code>) of file 
     * writers for atomically writing to output files.
     * <p>
     * <code>FileWriter</code> values are keyed by the absolute path 
     * of their physical output file.
     */
    private static Hashtable fileWriters = new Hashtable();
    
    /**
     * Static platform-dependent line separator string.
     */
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");

    /**
     * @see org.eclipse.hyades.logging.adapter.IComponent#update()
     */
    public void update() throws AdapterInvalidConfig {
        
        synchronized(fileWriters) {

            try {

                //Attempt to configure the outputter:
                super.update();
            }
            catch(AdapterInvalidConfig a) {
                
                //"Whether or not a file is available or may be created depends upon the underlying 
                //platform.  Some platforms, in particular, allow a file to be opened for writing by 
                //only one FileWriter (or other file-writing object) at a time. In such situations 
                //the constructors in this class will fail if the file involved is already open." 
                //(<code>java.io.FileWriter</code> JavaDocs) 

                //If a file writer is NOT cached for this output file (see above comment), 
                //re-throw the exception since an error occurred during the configuration 
                //of the outputter:
                if(!fileWriters.containsKey(rawLogFileName)){
                    throw a;
    		    }
            }
	                    
            //If a file writer is cached for this output file, set
            //the current <code>FileWriter</code> handle to the 
            //cached instance to ensure atomic file writing:
		    if(fileWriters.containsKey(rawLogFileName)){
		        fw = ((FileWriter)(fileWriters.get(rawLogFileName)));
		    }
		    
		    //Otherwise, cache the newly created <code>FileWriter</code> 
		    //handle for future use:
		    else{
		        fileWriters.put(rawLogFileName,fw);
		    }
        }
    }
	
    /**
     * @see org.eclipse.hyades.logging.adapter.outputters.CBEFileOutputter#writeToFile(java.lang.String)
     */
    protected void writeToFile(String event) throws IOException {
		
        //Ensure writing to the single output file is atomic:
	    synchronized(fw) {
		    		   
		    fw.write(event);
			fw.write(LINE_SEPARATOR);
			fw.flush();
		}		
	}
}
