/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.outputters;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Hashtable;

import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.IOutputter;
import org.eclipse.hyades.logging.adapter.impl.Outputter;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
/**
 * @author smith
 *
 * This Outputter writes CBE's to a file 
 * 
 */
public class CBEFileOutputter extends Outputter implements IOutputter
{
	/**
	 * directory address to start with, maybe relative or absolute 
	 */
	private String directory = null;
	/**
	 * relative file name (just name), maybe just regular expression 
	 */
	private String fileName = null;
	protected String rawLogFileName = null;
	protected  FileWriter fw = null;
		


	public Object[] processEventItems(Object[] msgs)
	{	
		if (msgs == null)
			return null;
			
		return processCBEs((CommonBaseEvent[]) msgs);
	}
	
	protected void writeToFile(String event) throws IOException,AdapterException {
		/* If the file writer does not exist then create it */
		if (fw == null) {
			setFileWriter();
		}	

		fw.write(event);
		fw.write("\n");
		fw.flush();
	}
	
	
	/**
	 * Process CBE's passed in - write them to a file
	 * @param events CBE's to write to file
	 * @return CommonBaseEvent[]
	 */
	private CommonBaseEvent[] processCBEs(CommonBaseEvent[] events)
	{
		if (events == null)
			return events;
			
		for (int i = 0; i < events.length; i++)
		{
			if (events[i] != null) {
				
				// Write CBE to file
				try {
				    
					String outstr = EventFormatter.toCanonicalXMLString(events[i],false);
					
					if (outstr != null) {
						writeToFile(outstr);
						incrementItemsProcessedCount();
					}

				} catch (IOException ioe) {
				    
				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

				    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_File_Outputter_Write_ERROR_");
				    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),rawLogFileName});							    

				    event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);
					
				    log(event);
				} catch (NullPointerException npe) {

				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

				    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_File_Outputter_Write_ERROR_");
				    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),rawLogFileName});							    

				    event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);
					
				    log(event);
				} catch (AdapterException e) {
					/* The output file could not be created so stop processing */
					return null;
				}
			}
		}
		
		return events;
	}
	public Object[] testProcessEventItems(Object[] msgs) throws AdapterInvalidConfig
	{
		if(!(msgs instanceof CommonBaseEvent[])) {
			throw new AdapterInvalidConfig("This outputter will only accept arrays of CommonBaseEvent");
		}
		return testProcessCBEs((CommonBaseEvent[]) msgs);
	}
	private CommonBaseEvent[] testProcessCBEs(CommonBaseEvent[] events)
	{
		return events;
	}
	
	/**
	 * Returns the directory.
	 * @return String
	 */
	final public String getDirectory()
	{
		return directory;
	}
	/**
	 * Returns the fileName.
	 * @return String
	 */
	final public String getFileName()
	{
		return fileName;
	}
	/**
	 * Sets the directory.
	 * @param directory The directory to set
	 */
	final public void setDirectory(String directory)
	{
		this.directory = directory;
	}
	/**
	 * Sets the fileName.
	 * @param fileName The fileName to set
	 */
	final public void setFileName(String fileName)
	{
		this.fileName = fileName;
	}
	
	/**
	 * update the configuration based on the configuration Element
	 */
	public void update() throws AdapterInvalidConfig
	{
		// first get the basic configuration set
		super.update();

		String directory = null;
		String fileName = null;
		
//		Get configuration

		Element element = getConfiguration();
		
		Element outputterTypeInstance = null;
		Element outputterNode;

		// Get the outputter parameters from the outputter properties
		Hashtable outputterProperties = getProperties();		
		if (outputterProperties != null && !outputterProperties.isEmpty()) {
			directory = (String)outputterProperties.get(Messages.getString("HyadesGAdirectoryAttributeName"));

			fileName = (String)outputterProperties.get(Messages.getString("HyadesGAfileNameAttributeName"));				
		}
		else {
			/**
			 * If there are no properties then get the outputter parameters from the old SingleFileOutputterType instance
			 */
			NodeList outputterNodes = element.getChildNodes();
			for (int i = 0; i < outputterNodes.getLength(); i++) {
				if (outputterNodes.item(i).getNodeType() == Node.ELEMENT_NODE) {
					outputterNode = (Element) outputterNodes.item(i);
	
					if (outputterNode.getTagName().equals(Messages.getString("HyadesGASingleFileOutputterTypeTagName"))) {
						// Get the outputter parameters from the outputter type instance attributes
						outputterTypeInstance = outputterNode;
						// Get attributes				
						if (outputterTypeInstance.hasAttribute(Messages.getString("HyadesGAdirectoryAttributeName")))
							directory = outputterTypeInstance.getAttribute(Messages.getString("HyadesGAdirectoryAttributeName"));
						if (outputterTypeInstance.hasAttribute(Messages.getString("HyadesGAfileNameAttributeName")))
							fileName = outputterTypeInstance.getAttribute(Messages.getString("HyadesGAfileNameAttributeName"));
					}
				}
			}
		}
			
		// We must have a directory and fileName
		if (directory == null || directory.length() == 0 || fileName == null || fileName.length() == 0) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_SingleFileOutputter_Invalid_Config_File_ERROR_"));
		}
		
		setDirectory(directory);
		setFileName(fileName);

		prepareFile();
	}
	
	/**
	 * Prepare the file for the log CBE output
	 */
	protected void prepareFile() throws AdapterInvalidConfig 
	{
		/* If the directory does not end with the proper file
		 * separator then we need to add one.  RKD:  We probably
		 * need to make sure we convert incorrect path separators
		 * to the proper platform.
		 */
		String pathSeparator=System.getProperty("file.separator");
		if(getDirectory().endsWith(pathSeparator)) {
			rawLogFileName = getDirectory() + getFileName();
		}
		else {
			rawLogFileName = getDirectory() + pathSeparator + getFileName();
		}

		try {
			setFileWriter();
		}
		catch (AdapterException ae) {
			throw new AdapterInvalidConfig(ae);
		}
	}
		
	/**
	 * Set the FileWriter for this File Outputter
	 */
	protected void setFileWriter() throws AdapterException 
	{
		try {
			fw = new FileWriter(rawLogFileName);
		}
		catch (IOException ioe) {
		    
		    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
		    
		    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_File_Outputter_Create_ERROR_");
		    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),rawLogFileName});							    

		    event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);

		    log(event);

			/* RKD: At this point there is no point if allowing this context to continue as we cannot access this outputter file.
			 */
		    throw new AdapterException(Messages.getString("HyadesGA_CBE_File_Outputter_Create_ERROR_", getUniqueID(),rawLogFileName));
		}
	}	
	
	
	/**
	 * Closes the file we are writing to.
	 * @see org.eclipse.hyades.logging.adapter.IComponent#stop()
	 */
	public void stop()
	{
		super.stop();
		if(fw!=null) {
			try {
				fw.close();
			}
			catch(IOException e) {
				/* We cannot close the file.  The file must already be closed.  Perhaps
				 * we should log an event in this scenario.
				 */
			}
		}
		
	}
}
