/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.sensors;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Hashtable;

import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.IComponent;
import org.eclipse.hyades.logging.adapter.IContext;
import org.eclipse.hyades.logging.adapter.impl.Context;
import org.eclipse.hyades.logging.adapter.impl.Sensor;
import org.eclipse.hyades.logging.adapter.util.BufferedPeriodicReader;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.adapter.util.MultipleFilesReader;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
/**
 * Sample Implementation of an OS File Sensor.
 *
 *
 * @version 0.1
 * @since 0.1
 * @see org.eclipse.hyades.logging.adapter.impl.Sensor
 * @see org.eclipse.hyades.logging.adapter.ISensor
 */
public class SingleOSFileSensor extends Sensor
{
	/* The relative or absolute directory containing the log file. */
	private String directory = null;
	
	/* The name including extension of the log file.
	 *
	 * Note: This value may contain a regular expression.
	 */
	private String fileName = null;
	
	/* The full (e.g. absolute paths) command string of the converter application. */
	private String converterCommand = null;
	
	/* The shell to run the converter in */
	private String converterShell = null;
	
	/* The logfile we are processing */
	private String rawLogFileName = null;
	
	/* bThe mechanism for reading the file */
	private BufferedPeriodicReader input = null;
	
	
	/* allocate the max blocking size  */
	private String[] lastLine = new String[maximumBlocking];
		
	
	/* If the directory does not end with the proper file
	 * separator then we need to add one.  RKD:  We probably
	 * need to make sure we convert incorrect path separators
	 * to the proper platform.
	 */
	private String pathSeparator=System.getProperty("file.separator");
	
	/* character set of the file we are dealing with */
	private String charset;

	/* Support for multiple files */
	private boolean multi = false; // specify whether multi files are being monitored
	private MultipleFilesReader mfr = null;
	private RandomAccessFile tmpLog = null;
	private File tmpFile = null;
	private static String prefix = "hgla";

	/**
	 * No-arguement constructor to create a SingleOSFileSensor.
	 */
	public SingleOSFileSensor()
	{
		super();
		try {
			tmpFile = File.createTempFile(prefix, null);
			tmpLog = new RandomAccessFile(tmpFile, "rws");
		} catch (Exception e) {
			tmpLog = null;
		}
	}
	/**
	 * Stops the sensor from reading the log file.  All object
	 * references are cleaned, file handles are closed and temporary
	 * files deleted.
	 *
	 * @see org.eclipse.hyades.logging.adapter.IComponent#stop()
	 */
	public void stop()
	{
		super.stop();
		clean();
	}
	/**
	 * Update the configuration based on the configuration Element.
	 *
	 * @see org.eclipse.hyades.logging.adapter.IComponent#update()
	 */
	public void update() throws AdapterInvalidConfig
	{
		// first get the basic configuration set
		super.update();
		// maximumBlocking is set by the sensor config
		lastLine = new String[maximumBlocking];
		String directory = null;
		String fileName = null;
		String converterCmdAttribute = null;

		Element sensorTypeInstance = null;
		Element sensorNode;
		Element element = getConfiguration();

		// Get the sensor parameters from the sensor properties
		Hashtable sensorProperties = getProperties();		
		if (sensorProperties != null && !sensorProperties.isEmpty()) {
			directory = (String)sensorProperties.get(Messages.getString("HyadesGAdirectoryAttributeName"));

			fileName = (String)sensorProperties.get(Messages.getString("HyadesGAfileNameAttributeName"));
			
			converterCmdAttribute = (String)sensorProperties.get(Messages.getString("HyadesGAconverterCmdAttributeName"));
			
			converterShell = (String)sensorProperties.get(Messages.getString("HyadesGAconverterShellAttributeName"));										
		}
		else {
			// If there are no properties then get sensor parameters from the old SingelFileSensor type instance
			NodeList sensorNodes = element.getChildNodes();
			for (int i = 0; i < sensorNodes.getLength(); i++) {
				if (sensorNodes.item(i).getNodeType() == Node.ELEMENT_NODE) {
					sensorNode = (Element) sensorNodes.item(i);
	
					/* Else the configuration is the old style SingleFileSensor type element */
					if (sensorNode.getTagName().equals(Messages.getString("HyadesGASingleFileSensorTagName"))) {
						// Get the sensor parameters from the sensor type instance attributes
						sensorTypeInstance = sensorNode;
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAdirectoryAttributeName"))) {
							directory = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAdirectoryAttributeName"));
						}
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAfileNameAttributeName"))) {
							fileName = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAfileNameAttributeName"));
						}
	
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAconverterCmdAttributeName"))) {
							converterCmdAttribute = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAconverterCmdAttributeName"));
						}
	
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAconverterShellAttributeName"))) {
							converterShell = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAconverterShellAttributeName"));
						}					
					}
				}
			}
		}
		
		if (converterCmdAttribute != null) {
			converterCmdAttribute = converterCmdAttribute.trim();
			if (converterCmdAttribute.length() == 0) {
				converterCmdAttribute = null;
			}
		}
		
		if (converterShell != null) {
			converterShell = converterShell.trim();
			if (converterShell.length() == 0) {
				converterShell = null;
			}
		}

		// Get the parent context so we can get the charset and
		// continuous operation flag
		boolean continuousContext = false;
		for(IComponent current=this; current!=null; current=current.getParent()) {
			if(current instanceof IContext) {
				IContext context = (IContext)current;
				charset=context.getCharset();
				if (context instanceof Context) {
					continuousContext = ((Context)context).getContinuousOperation();
				}
				break;
			}
		}
		// Check if the directory and fileName properties are non null
		if (directory == null || directory.trim().length() == 0 ||
				fileName == null || fileName.trim().length() == 0) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_SingleFileSensor_Invalid_Config_File_ERROR_"));
		}
		
		mfr = new MultipleFilesReader(directory, fileName, tmpLog);
		mfr.init();

		// If only one file matches the regex, use the original SingleOSFileSensor
		if(mfr.size() <= 1) {
			multi = false;
			// Close the temporary file since it won't be used
			if (tmpLog != null) {
				try {
					tmpLog.close();
				}
				catch (IOException e) {
					// Ignore this exception
				}
			}
			if(mfr.size() == 1) {
				String fn = mfr.getNext();
				//We must have a vaild directory and fileName that exists on the local file system:
				if (fn == null || fn.trim().length() == 0 ||
					(converterCmdAttribute == null && !(new File(fn).canRead()))) {
					throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_SingleFileSensor_Invalid_Config_File_ERROR_"));
				}
				setFileName(new File(fn).getName()); // Because the name is fully-qualified			
			}
			else { // no match at all
				// If there was no converter command to generate the log file and the sensor is not running in a continuous context then throw an exception
				if (converterCmdAttribute == null && !continuousContext) {
					throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_SingleFileSensor_Invalid_Config_File_ERROR_"));
				}
				else {
					// use the original file name, assuming the converter command will generate the file
					setFileName(fileName.trim());
				}
			}
		}
		// Otherwise, use MultipleOSFileSensor
		else {
			multi = true;

			// Create a single log file for SingleOSFileSensor to read
			setFileName(tmpFile.getAbsolutePath());
			mfr.start();
		}
		
		setDirectory(directory.trim());
		setConverterCommand(converterCmdAttribute);
		
	    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
	    event.setMsg(Messages.getString("HyadesGASensor_SingleOSFileSensor_Configuration_INFO_",getFileName(), getDirectory(), getConverterCommand()));
	    event.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
	    log(event);
	    
		/* If we do not have a charset we will default to UTF-8.  THIS SHOULD NEVER HAPPEN */
		if(charset==null) {
			if (System.getProperty("os.name", "Windows").equals("z/OS") || System.getProperty("os.name", "Windows").equals("OS/390")) {
				/* Default to EBCDIC charset on z/OS */
				charset="IBM-1047";
			}
			else {
				charset="UTF-8";
			}
		}
	}

	/**
	 * 	simulates a getNext
	 * @see com.ibm.acad.general.sensor.ISensor#testGetNext()
	 */
	public Object[] testGetNext()
	{
		return testGetNextLine();
	}
	/**
	 * implements the testGetNext behaviour by returning a test string
	 * @return String[]
	 */
	public String[] testGetNextLine()
	{
		String[] lineArray = new String[2];
		lineArray[0] = "test string";
		return lineArray;
	}
	
	/**
	 * returns last read line
	 * @see com.ibm.acad.general.sensor.ISensor#getNext()
	 */
	public Object[] getNext() {
		
		
		/* On our first pass through getNext() and subsequent calls after we have reached the end of the file
		 * we will need to do any pre-processing before reading the data.  If we are flushing do not run the converter again
		 */
		if (rawLogFileName == null  && !flushingMode) {
			if(multi) {
				rawLogFileName = getFileName();
			}
			else {
				if(getDirectory().endsWith(pathSeparator)) {
					rawLogFileName = getDirectory() + getFileName();
				}
				else {
					rawLogFileName = getDirectory() + pathSeparator + getFileName();
				}
			}
			
			if(rawLogFileName != null) {
				try {
					if(input==null) {
						input = new BufferedPeriodicReader(rawLogFileName,converterCommand, converterShell);
						input.setCharset(charset);
						input.setConfidenceBufferSize(confidenceBufferSize);
						input.setFileFooterSize(fileFooterSize);
						input.setMultiFile(multi);
					}
					input.prepare();
				}
				catch (Exception e) {

				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
				    event.setMsg(e.toString());
				    event.setSeverity(CommonBaseEvent.SEVERITY_CRITICAL);

				    log(event);
				}    			
			}
		}

		/* Clean up the array */
		for(int i = 0; i < lastLine.length; i++) {
			lastLine[i] = null;
		}

		/* Retrieve the contents of the file up to maximumBlocking lines */
		if (input != null) {	
			for (int i = 0; i < maximumBlocking; i++) {
				try {
					lastLine[i] = input.readLine();
				}
				catch (Exception e) {

				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
				    event.setMsg(e.toString());
				    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

				    log(event);
				}
				if (lastLine[i] == null) {
					i = maximumBlocking;
				    rawLogFileName=null;
				}
				else {
					incrementItemsProcessedCount();
				}
			}
		}
		
		/* Was there any data that we retrieved?  if so, return it, otherwise indicate there is no more data. */
		if (lastLine[0] != null) {
			return lastLine;
		}
		else {
			return null;
		}
	}
	
	/**
	 * do all cleaning here
	 */
	public void clean() {
		try	{
			if (input != null)
			{
				input.close();
			}
			mfr.interrupt();

			if(tmpLog != null) {
				try {
					tmpLog.close();
				} catch (IOException e) {
				}
			}
			tmpFile.delete();
		}
		catch (Exception e) {
		
		    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
		    event.setMsg(e.toString());
		    event.setSeverity(CommonBaseEvent.SEVERITY_HARMLESS);

		    log(event);
		}
	}
	
	/**
	 * Returns the directory.
	 * @return String
	 */
	final public String getDirectory()
	{
		return directory;
	}
	
	/**
	 * Returns the fileName.
	 * @return String
	 */
	final public String getFileName()
	{
		return fileName;
	}
	
	/**
	 * Returns the converter command.
	 * @return String
	 */
	final public String getConverterCommand()
	{
		return converterCommand;
	}
	/**
	 * Sets the directory.
	 * @param directory The directory to set
	 */
	final public void setDirectory(String directory)
	{
		this.directory = directory;
	}
	/**
	 * Sets the fileName.
	 * @param fileName The fileName to set
	 */
	final public void setFileName(String fileName)
	{
		this.fileName = fileName;
	}

	/**
	 * Sets the converter command.
	 * @param converterCmd The converter command to set
	 */
	final public void setConverterCommand(String converterCmd)
	{
		this.converterCommand = converterCmd;
	}

}
