/**********************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.http.internal.importWizard;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;

import org.eclipse.hyades.models.common.facades.behavioral.ILoop;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestInvocation;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.facades.behavioral.impl.HyadesFactory;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.common.TestCommon;
import org.eclipse.hyades.test.common.TestCommonPlugin;
import org.eclipse.hyades.test.common.internal.util.TestCommonUtil;
import org.eclipse.hyades.test.http.TestHttpPlugin;
import org.eclipse.hyades.test.http.internal.preferences.HttpPreferenceUtil;
import org.eclipse.hyades.test.http.internal.util.RequestHelper;
import org.eclipse.hyades.test.java.TestJavaPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.ui.internal.util.CoreUtil;
import org.eclipse.hyades.ui.internal.util.JavaUtil;
import org.eclipse.hyades.ui.internal.util.XMLUtil;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * @author marcelop
 * @since 1.0.2
 */
public class HttpRequestFromXML
implements IDisposable
{
	private String testSuiteBaseName;
	private String scriptFile;
	private IPath outputFolder;
	private List resources;
	private RequestHelper requestHelper;

	private int resourceNameIndex;
	private int requestCounter;
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		if(requestHelper != null)
			requestHelper.dispose();
		if(resources != null)
			resources.clear();
	}

	/**
	 * @return
	 */
	public IPath getOutputFolder()
	{
		return outputFolder;
	}

	/**
	 * @param outputFolder
	 */
	public void setOutputFolder(IPath outputFolder)
	{
		this.outputFolder = outputFolder;
	}

	/**
	 * @return
	 */
	public String getScriptFile()
	{
		return scriptFile;
	}

	/**
	 * @param scriptFile
	 */
	public void setScriptFile(String scriptFile)
	{
		this.scriptFile = scriptFile;
	}

	/**
	 * @return
	 */
	public String getTestSuiteBaseName()
	{
		return testSuiteBaseName;
	}

	/**
	 * @param testSuiteBaseName
	 */
	public void setTestSuiteBaseName(String testSuiteBaseName)
	{
		this.testSuiteBaseName = testSuiteBaseName;
	}

	public boolean load()
	{
		Element element = null;
		try
		{
			element = XMLUtil.loadDom(new FileInputStream(new File(scriptFile)), "pageSet");
		}
		catch (FileNotFoundException e)
		{
			openErrorMessage(TestHttpPlugin.getString("import.error.UnableToParse"), CoreUtil.getStackTrace(e));
			return false;
		}
		
		if(element == null)
		{
			openErrorMessage(TestHttpPlugin.getString("import.error.UnableToParse"), null);
			return false;
		}
			
		return loadPageSet(element);
	}
	
	protected boolean loadPageSet(Element pageSet)
	{
		if("5.0.0".equals(XMLUtil.getValue(pageSet,"version")))
			return loadPageSet500(pageSet);
		
		openErrorMessage(TestHttpPlugin.getString("import.error.UnableToParse"), TestHttpPlugin.getString("import.error.UnableToParseDetail2"));
		return false;					
	}

	protected boolean loadPageSet500(Element pageSet)
	{
		NodeList pages = XMLUtil.getChildrenByName(pageSet, "page");
		if((pages == null) || (pages.getLength() == 0))
		{
			openErrorMessage(TestHttpPlugin.getString("import.error.UnableToParse"), TestHttpPlugin.getString("import.error.UnableToParseDetail3"));
			return false;
		}
			
		requestHelper = new RequestHelper();
		boolean eachPageIsTestSuite = HttpPreferenceUtil.getInstance().getImportAsSeparate();
		ITestSuite testSuite = null;
		int pageCount = 0;
		for(int i=0, max=pages.getLength(); i<max; i++)
		{
			if(eachPageIsTestSuite || (testSuite == null))
			{
				testSuite = createTestSuite();
				pageCount = 1;
			}
			else
				pageCount++;
				
			parsePage(testSuite, (Element)pages.item(i), pageCount);
		}	
		requestHelper.dispose();
		
		saveResources();
		return true;
	}
	
	protected void parsePage(ITestSuite testSuite, Element page, int pageCount)
	{
		NodeList requests = XMLUtil.getChildrenByName(page, "request");
		if(requests != null)
		{   
			ILoop loop = HyadesFactory.INSTANCE.createLoop();
			loop.setName(TestJavaPlugin.getString("W_LOOP") + " " + pageCount);
			loop.getCondition().setConstraint("1");
			testSuite.getImplementor().getBlock().getActions().add(loop);			
			
			for(int i=0, max=requests.getLength(); i<max; i++)
			{
				ITestCase testCase = createTestCase(testSuite, loop);
				parseRequest((Element)requests.item(i), testCase);
			}	
		}		
	}
	
	protected void parseRequest(Element request, ITestCase testCase)
	{
		requestHelper.setTestCase(testCase);

		requestHelper.setAttribute(RequestHelper.ATT_METHOD, XMLUtil.getValue(request, "method"));
		requestHelper.setAttribute(RequestHelper.ATT_HOST, XMLUtil.getValue(request, "host"));
		requestHelper.setAttribute(RequestHelper.ATT_PORT, XMLUtil.getValue(request, "port"));
		requestHelper.setAttribute(RequestHelper.ATT_ABS_PATH, XMLUtil.getValue(request, "abs_path"));
		requestHelper.setAttribute(RequestHelper.ATT_BODY, XMLUtil.getValue(request, "body"));
		requestHelper.setAttribute(RequestHelper.ATT_VERSION, XMLUtil.getValue(request, "version"));
		
		NodeList headers = XMLUtil.getChildrenByName(request, "header");
		if(headers != null)
		{
			for(int i=0, max=headers.getLength(); i<max; i++)
			{				
				Element header = (Element)headers.item(i);
				requestHelper.setHeader(XMLUtil.getValue(header, "name"), XMLUtil.getValue(header, "value"));
			}	
		}
	}
	
	protected void openErrorMessage(String text, String detail)
	{
		if(detail != null)
			text = text + " \n\n" + detail;
		MessageDialog.openError(Display.getDefault().getActiveShell(), TestHttpPlugin.getString("W_ERROR"), text);
	}
	
	protected ITestSuite createTestSuite()
	{
		if(resources == null)
			resources = new ArrayList();
			
		IPath file = null;
		do
		{
			file = outputFolder.append(testSuiteBaseName + (++resourceNameIndex)).addFileExtension(ICommonConstants.TEST_SUITE_FILE_EXTENSION);
		}
		while(ResourcesPlugin.getWorkspace().getRoot().getFile(file).exists());
		
		URI uri = URI.createPlatformResourceURI(file.toString());
		Resource resource = EMFUtil.getResourceFactory(ICommonConstants.TEST_SUITE_FILE_EXTENSION).createResource(uri);
		resources.add(resource);
		
		ITestSuite testSuite = HyadesFactory.INSTANCE.createTestSuite(resource);
		testSuite.setType(TestCommon.HTTP_JUNIT_TEST_SUITE_TYPE);
		testSuite.setName(testSuiteBaseName + " "+ resourceNameIndex);		
		if(testSuite.getImplementor() == null)
			HyadesFactory.INSTANCE.createImplementor(testSuite, false);
			
		String clsName = JavaUtil.getValidClassName(testSuite.getName(), true);
		if(clsName == null)
			clsName = "Test";
		testSuite.getImplementor().setResource("test." + clsName);

		return testSuite;
	}
	
	protected ITestCase createTestCase(ITestSuite testSuite, ILoop loop)
	{
		ITestCase testCase = TestCommonUtil.createTestCase(testSuite, TestCommon.HTTP_JUNIT_TEST_CASE_TYPE, false, TestHttpPlugin.getString("W_HTTP_REQUEST").toLowerCase() + " " + (++requestCounter));

		ITestInvocation testInvocation = HyadesFactory.INSTANCE.createTestInvocation(testCase);
		loop.getBlock().getActions().add(testInvocation);

		String baseName = TestJavaPlugin.getString("W_LWR_INV");
		String testName = testCase.getName();
		if(testName == null)
			testInvocation.setName(TestCommonUtil.getUniqueName(baseName, loop.getBlock().getActions()));
		else
			testInvocation.setName(TestCommonPlugin.getString("NME_CONCAT_SEP", new String[]{testName, baseName}));
		
		return testCase;
	}
	
	protected void saveResources()
	{
		for (Iterator i = resources.iterator(); i.hasNext();)
		{
			try
			{
				EMFUtil.save((Resource)i.next());
			}
			catch (Exception e)
			{
				TestHttpPlugin.logError(e);
			}
		}
	}
	
	public List getResources()
	{
		if(resources == null)
			return Collections.EMPTY_LIST;
			
		return resources;
	}
}
