/**********************************************************************
 * Copyright (c) 2003-2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.util.internal;

import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCFilter;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TRCSourceInfo;

/**
 * Here are some utilities related to Coverage feature
 */
public class CoverageUtil {

	/**
	 * @param e
	 * @return source location signature if any
	 */
	private static String getSourceSig(TRCSourceInfo e) {
		return (e != null ? e.getLocation()+":" : "");
	}
	
	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCMethod e) {
		return  getSourceSig(e.getSourceInfo())+
				e.getLineNo()+":"+
				getSig(e.getDefiningClass())+"."+
				e.getName()+
				e.getSignature();
	}

	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCClass e) {
		return  getSourceSig(e.getSourceInfo())+
				e.getLineNo()+":"+
				getSig(e);
	}

	/**
	 * @param e
	 * @return
	 */
	public static String getFullSig(TRCPackage e) {
		return e.getName();
	}
	
	/**
	 * @param e
	 * @return qualified name of the class
	 */
	private static String getSig(TRCClass e) {
		return (e.getPackage().getName() == null ? "" : (e.getPackage().getName()+"."))+
				e.getName();
	}
	
	/**
	 * This is for post-filtering needed because of the boundary classes collected even if excluded
	 * by the filter set.<br>
	 * The empty packages are filtered out and
	 * those listed in the active filters for this agent configuration
	 * @param pac
	 * @return
	 */
	public static boolean isFiltered(TRCPackage pac) {
		if (pac.getClasses() == null ||
			pac.getClasses().size() == 0) {
			//if empty package
			return true;
		}

		Object[] conflist = pac.getProcess().getAgent().getAgentProxy().getConfigurations().toArray();
		int idxmax = conflist.length;
		for(int idx = 0; idx < idxmax; idx++){
			TRCConfiguration c = (TRCConfiguration) conflist[idx];
			if(!c.isActive())continue;
			
			Object[] filtlist = c.getFilters().toArray();
			for (int k = 0; k < filtlist.length; ++k) {
				TRCFilter f = (TRCFilter)filtlist[k];
				if(!f.getActive().booleanValue())
					continue;
				if(isFilteredPackage(f,pac)){
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * The empty classes are filtered out too
	 * @param c
	 * @return
	 */
	public static boolean isFiltered(TRCClass cl) {
		if (cl.getMethods() == null ||
			cl.getMethods().size() == 0) {
			//if empty class
			return true;
		}
		Object[] conflist = cl.getPackage().getProcess().getAgent().getAgentProxy().getConfigurations().toArray();
		int idxmax = conflist.length;
		for(int idx = 0; idx < idxmax; idx++){
			TRCConfiguration c = (TRCConfiguration) conflist[idx];
			if(!c.isActive())continue;
			
			Object[] filtlist = c.getFilters().toArray();
			for (int k = 0; k < filtlist.length; ++k) {
				TRCFilter f = (TRCFilter)filtlist[k];
				if(!f.getActive().booleanValue())
					continue;
				if(isFilteredClass(f,cl)){
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * @param filter
	 * @param pack
	 * @return 
	 */
	private static boolean isFilteredPackage(TRCFilter filter, TRCPackage pack)
	{
		String pattern = filter.getPattern();
		String mode = filter.getMode();
		String type = filter.getType();
		if(type.equals(mode)){
			type = parseFilterPattern(pattern);
		}
		String methodPattern = filter.getOperation();
		if(methodPattern!=null && (methodPattern.length()>0 && !methodPattern.equals("*"))){
			return false;
		}
		String name = pack.getName();
		boolean exclude = true;
		if(mode.equals("INCLUDE")){//$NON-NLS-1$
			exclude = false;
		}
		if(type.equals("NONE")){
			if (name.equals(pattern)) {
				return exclude;
			}
		}else if(type.equals("PREFIX")){
			if (name.endsWith(pattern)) {
				return exclude;
			}
		}else if(type.equals("SUFFIX")){
			if (name.startsWith(pattern)) {
				return exclude;
			}
		}
		return false;
	}

	/**
	 * @param filter
	 * @param cl
	 * @return 
	 */
	private static final int Path_Separator = '.';
	private static boolean isFilteredClass(TRCFilter filter, TRCClass cl)
	{
		String pattern = filter.getPattern();
		String mode = filter.getMode();
		String type = filter.getType();
		if(type.equals(mode)){
			type = parseFilterPattern(pattern);
		}
		String methodPattern = filter.getOperation();
		if(methodPattern!=null && methodPattern.length()>0 && !methodPattern.equals("*")){
			return false;
		}
		String name = cl.getName();
		if(pattern.indexOf(Path_Separator)!=-1){
			name = cl.getPackage().getName() + Path_Separator + name;
		}
		boolean exclude = true;
		if(mode.equals("INCLUDE")){//$NON-NLS-1$
			exclude = false;
		}
		if(type.equals("NONE")){
			if (name.equals(pattern)) {
				return exclude;
			}
		}else if(type.equals("PREFIX")){
			if (name.endsWith(pattern)) {
				return exclude;
			}
		}else if(type.equals("SUFFIX")){
			if (name.startsWith(pattern)) {
				return exclude;
			}
		}
		return false;
	}
	
	private static String parseFilterPattern(String filterPattern)
	{
	 String genericPattern = "NONE";
	 
	 if(filterPattern.length()==0){
	 	return genericPattern;
	 }

	 int p = filterPattern.indexOf('*');
	 if (p == 0)
	 {
	  genericPattern = "PREFIX";
	  filterPattern = filterPattern.substring(1);
	 }
	 else if (p == (filterPattern.length()-1))
	 {
	  genericPattern = "SUFFIX";
	  filterPattern = filterPattern.substring(0,(filterPattern.length()-2));
	 }
	 return genericPattern;
	}


}
