/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.ui.internal.extension;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.ui.model.IWorkbenchAdapter;
import org.eclipse.ui.model.WorkbenchAdapter;

/**
 * Hyades internal representation of the navigatorFilterSet
 * element in the navigatorExtensions extension point. Contains
 * accessors for all the attributes.
 *
 * @since 3.0
 */
public class NavigatorFilterSet implements IAdaptable {

	private static final IPreferenceStore _prefs = HyadesUIPlugin.getInstance().getPreferenceStore();
	
	private ViewerFilter _filter;
	private ImageDescriptor _icon;
	private String _id;
	private String _label;
	private String _navigatorID;
	private boolean _recursive;
	private boolean _common;
	private boolean _defaultEnabled;
	
	/**
	 * Constructs a new element from the given configuration. 
	 */
	public NavigatorFilterSet(IConfigurationElement e)
			throws ClassNotFoundException, IllegalAccessException, InstantiationException {
		IPluginDescriptor descriptor = e.getDeclaringExtension().getDeclaringPluginDescriptor();

		_id = e.getAttribute("id");
		_label = e.getAttribute("label");
		_common = "true".equals(e.getAttribute("common"));
		_defaultEnabled = "true".equals(e.getAttribute("defaultEnabled"));
		_recursive = !"false".equals(e.getAttribute("recursive"));
		_icon = ImageDescriptor.createFromURL(descriptor.find(new Path(e.getAttribute("icon"))));

		String className = e.getAttribute("class");
		Class clazz = descriptor.getPluginClassLoader().loadClass(className);
		_filter = (ViewerFilter)clazz.newInstance();
	}

	/**
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter) {
		if (IWorkbenchAdapter.class.equals(adapter)) {
			return new WorkbenchAdapter() {
				public ImageDescriptor getImageDescriptor(Object object) {
					return get_icon();
				}

				public String getLabel(Object o) {
					return get_label();
				}
			};
		}
		return null;
	}
	
	/**
	 * Returns this filter set's unique id.
	 * 
	 * @return this filter set's unique id.
	 */
	public String getID() {
		return _id;
	}
	
	/**
	 * Returns this filter set's image icon.
	 * 
	 * @return this filter set's image icon.
	 */
	public ImageDescriptor getImageDescriptor() {
		return _icon;
	}
	
	/**
	 * Returns this filter set's label.
	 * 
	 * @return this filter set's label.
	 */
	public String getLabel() {
		return _label;
	}
	
	/**
	 * Returns this filter set's preference key, which is a
	 * boolean flag stating whether or not this filter set is
	 * enabled.
	 * 
	 * @return this filter set's preference key.
	 */
	public String getPreferenceKey() {
		return _id + ".enabled";
	}
	
	/**
	 * Returns the filter set's <code>ViewerFilter</code>.
	 * 
	 * @return the filter set's <code>ViewerFilter</code>.
	 */
	public ViewerFilter getViewerFilter() {
		return _filter;
	}
	
	/**
	 * Indicates whether or not this filter set is a common
	 * filter set. Common filter sets appear directly in the
	 * navigator menu, as well as the filters dialog.
	 * 
	 * @return whether or not this is a common filter.
	 */
	public boolean isCommon() {
		return _common;
	}
	
	/**
	 * Indicates whether or not this filter set is enabled by
	 * default.
	 * 
	 * @return whether or not this filter set is enabled by
	 *    default.
	 */
	public boolean isDefaultEnabled() {
		return _defaultEnabled;
	}
	
	/**
	 * Indicates whether or not this filter set is currently
	 * enabled.
	 *  
	 * @return whether or not this filter set is currently enabled.
	 */
	public boolean isEnabled() {
		return _prefs.getBoolean(getPreferenceKey());
	}
	
	/**
	 * Indicates whether or not this filter set is recursive.
	 * Recursive filter sets filter out any children (recursively)
	 * of a filtered element.
	 * 
	 * @return whether or not this filter set is recursive.
	 */
	public boolean isRecursive() {
		return _recursive;
	}
	
	/**
	 * Enables or disables this filter set.
	 * 
	 * @param value <code>true</code> if this filter set should
	 *    be enabled, <code>false</code> otherwise.
	 */
	public void setEnabled(boolean value) {
		_prefs.setValue(getPreferenceKey(), value);
	}
    /**
     * This method is not intended to be overridden.
     * @return Returns the _icon.
     */
    protected ImageDescriptor get_icon() {
        return _icon;
    }
    /**
     * This method is not intended to be overridden.
     * @return Returns the _label.
     */
    protected String get_label() {
        return _label;
    }
}
