/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.util.Vector;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jdt.core.search.SearchMatch;
import org.eclipse.jdt.core.search.SearchParticipant;
import org.eclipse.jdt.core.search.SearchPattern;
import org.eclipse.jdt.core.search.SearchRequestor;
import org.eclipse.jdt.ui.actions.OpenAction;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.hyades.ui.HyadesUIPlugin;

/**
 * A utility class to open a Java file selecting a specific element in the file.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class OpenJavaSource
{
	/**
	 * Eclipse's Java perspective ID.
	 */
	private final static String JAVA_PERSPECTIVE_ID = "org.eclipse.jdt.ui.JavaPerspective";

	/**
	 * Opens the source for a selected object in the text editor of the Java Perspective.
	 * @param String pattern - the object that we wish to display the source for.  The pattern should be the full
	 * package name of the class or method.  For a method, the signature is also part of the pattern.  If the method
	 * is a constructor, the class name is followed by the signature.  
	 * Eg., a typical method would be com.ibm.etools.pd.core.util.OpenSourceUtil.openSource(String, int) void
	 * and a constructor would be com.ibm.ibm.etools.pd.core.util.OpenSourceUtil()
	 * @param javaType The java type identifies whether the pattern represents a Type, a method, or a constructor.
	 * The values are defined in org.eclipse.jdt.core.search.IJavaSearchConstants and can be TYPE, METHOD, or CONSTRUCTOR 
	 * respectively.	
	 * @param searchScope The search scope for the source to be opened.  If null, the WorkspaceScope is be used;
	 * @param switchToJavaPerspective If <code>true</code> the Java perspective is be opened.
	 * @return <code>true</code> if the editor was opened or <code>false</code> otherwise
	 */
	public static boolean openSource(String pattern, int javaType, IJavaSearchScope searchScope, boolean switchToJavaPerspective)
	{
		OpenSourceJavaSearchResultRequestor requestor = getSearchResults(pattern, javaType, searchScope);
		if (requestor != null && requestor.getNumberOfMatches() > 0) {
			return openWbSource(requestor.getMatchForPattern(pattern), switchToJavaPerspective);
		}
		return false;
	}

	public static IFile getSourceFile(String pattern, int javaType, IJavaSearchScope searchScope) {
		OpenSourceJavaSearchResultRequestor requestor = getSearchResults(pattern, javaType, searchScope);
		if (requestor != null) {
			return OpenSourceJavaSearchResultRequestor.getFile(requestor.getMatchForPattern(pattern));
		}
		return null;
	}
	
	private static String fixPatternForInnerClasses(String pattern, int javaType)
	{
		// Handle patterns for inner classes
		int bIdx = pattern.indexOf("(");
		int idx;
		
		if (bIdx >= 0)
		{
			idx = pattern.substring(0, bIdx).lastIndexOf("$");
		}
		else
		{
			idx = pattern.lastIndexOf("$");
		}
		
		if (idx >= 0)
		{
			pattern = pattern.replace('$', '.');			

			if (bIdx >= 0 && javaType == IJavaSearchConstants.CONSTRUCTOR)
			{
				String outerClass = pattern.substring(0, idx);				
				String outerClassParamater = outerClass+","; 
				
				if (pattern.indexOf(outerClassParamater) >= 0)
					pattern = pattern.replaceAll(outerClassParamater, "");
				else
					pattern = outerClass + pattern.replaceAll(outerClass, ""); 
			}
		}
		return pattern;
	}
	
	private static SearchPattern getSearchPattern(String pattern, int javaType)
	{
		if (pattern != null && !"".equals(pattern))
		{
			pattern = fixPatternForInnerClasses(pattern, javaType);
			
			return SearchPattern.createPattern(pattern, javaType, IJavaSearchConstants.DECLARATIONS,
					SearchPattern.R_EXACT_MATCH | SearchPattern.R_CASE_SENSITIVE);
		}
		
		return null;
	}
		
	private static IJavaSearchScope getJavaSearchScope(IJavaSearchScope searchScope)
	{
		if (searchScope == null)
		{
			IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
			
			IJavaElement[] javaElements = new IJavaElement[projects.length];
			
			for (int i = 0; i < projects.length; i++)
			{
				javaElements[i] = JavaCore.create(projects[i]);
			}			
			
			searchScope = SearchEngine.createJavaSearchScope(javaElements, false);
		}
		
		return searchScope;
	}
	
	private static OpenSourceJavaSearchResultRequestor getSearchResults(String pattern, int javaType, IJavaSearchScope searchScope) {
		SearchPattern searchPattern = getSearchPattern(pattern, javaType);
		
		if (searchPattern != null)
		{
			OpenSourceJavaSearchResultRequestor requestor = new OpenSourceJavaSearchResultRequestor();
			
			// Create a search engine and search the workspace for the selected object.
			SearchEngine searchEngine = new SearchEngine();
			
			// Perform the search
			try
			{
				searchEngine.search(searchPattern, new SearchParticipant[] {SearchEngine.getDefaultSearchParticipant()},
						getJavaSearchScope(searchScope), requestor, new NullProgressMonitor());				
			}
			catch (CoreException e) {
				// This error will result in falling into the openSourceFailed code below
			}
			return requestor;
		}
		return null;
	}
	
	/**
	  * OpenSourceJavaSearchResultCollector is responsible for handling the search results for finding 
	  * the source and source range for the given selected object.  An instance of this class is created and
	  * passed to the SearchEngine.
	  */
	private static class OpenSourceJavaSearchResultRequestor extends SearchRequestor
	{
		private Vector _matches;

		public OpenSourceJavaSearchResultRequestor()
		{
			super();
			_matches = new Vector();
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jdt.core.search.SearchRequestor#acceptSearchMatch(org.eclipse.jdt.core.search.SearchMatch)
		 */
		public void acceptSearchMatch(SearchMatch match) throws CoreException {
			_matches.addElement(match);
		}
		
		public int getNumberOfMatches()
		{
			return _matches.size();
		}
		
		public SearchMatch getMatch(int i)
		{
			return (SearchMatch)_matches.elementAt(i);
		}
		
		public SearchMatch getFirstMatch()
		{
			if (getNumberOfMatches() > 0)
				return getMatch(0);
			
			return null;
		}
		
		public SearchMatch getMatchForPattern(String pattern)
		{
			if (getNumberOfMatches() > 1)
			{
				try
				{
					IJavaElement je = null;
					IMethod me = null;		
					
					for (int i = 0; i < getNumberOfMatches(); i++)
					{
						je = (IJavaElement)getMatch(i).getElement();
						
						if (je instanceof IMethod)
						{
							me = (IMethod)je;
							
							if (!me.isConstructor() && pattern.indexOf(me.getDeclaringType().getFullyQualifiedName()) >= 0)
								return getMatch(i);
						}
					}
				}
				catch (Exception e)
				{
					HyadesUIPlugin.logError(e);
				}
			}
			
			return getFirstMatch();
		}		
		
		public static IFile getFile(SearchMatch match)
		{
			if (match != null && match.getResource() instanceof IFile)
				return (IFile) match.getResource();
			
			return null;
		}
	}


	/**
	 * openWbSource is used to open the specified source in the text editor of the Java Perspective.
	 * In addition, it moves the source to the start of the specified source region and highlights
	 * the region.
	 * @param match  org.eclipse.jdt.core.search.SearchMatch - A SearchMatch object that this method will open the source on.
	 * @return boolean - true if the source was opened successfully, false otherwise.
	 */
	private static boolean openWbSource(SearchMatch match, boolean switchToJavaPerspective)
	{
		IWorkbenchPage page = null;
	
		if(switchToJavaPerspective)
			page = getJavaActivePage();
		else
			page = UIUtil.getActiveWorkbenchPage();
		
		if(page == null)
			return false;
		
		IJavaElement je = (IJavaElement)match.getElement();

		StructuredSelection ss = new StructuredSelection(je);
		
		try
		{
			OpenAction openAction = new OpenAction(page.getActivePart().getSite());
			openAction.run(ss);
			
			return true;
		}catch (Exception e)
		{
			HyadesUIPlugin.logError(e);
		}
				
		return false;
	}

	/**
	 * getJavaActivePage is used to locate and return the workbench page for the Java Perspective.
	 */
	private static IWorkbenchPage getJavaActivePage()
	{
		IWorkbenchPage page = UIUtil.getActiveWorkbenchPage();
		if (page == null || !page.getPerspective().getId().equals(JAVA_PERSPECTIVE_ID))
		{
			final IWorkbenchWindow dwindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
			page = null;

			try
			{
				IWorkbenchPage[] persps = dwindow.getPages();
				for (int idx = 0; idx < persps.length; idx++)
				{
					if (persps[idx].getPerspective().getId().equals(JAVA_PERSPECTIVE_ID))
					{
						//trace perspective
						page = persps[idx];

						dwindow.setActivePage(page);
						break;
					}
				}

				if (page == null)
				{
					IAdaptable element = ResourcesPlugin.getWorkspace().getRoot();
					IWorkbench workBench = dwindow.getWorkbench();
					if (workBench != null && element != null)
						page = workBench.showPerspective(JAVA_PERSPECTIVE_ID, dwindow, element);
				}
			}
			catch (Exception exc)
			{
				HyadesUIPlugin.logError(exc);
			}
		}

		return page;
	}	
}