/**********************************************************************
 * Copyright (c) 2003,2004 Scapa Technologies Limited and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.widgets.table.internal;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColorCellEditor;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;

public abstract class TableAdapter extends Composite implements IStructuredContentProvider, ITableLabelProvider, ICellModifier
{
	public static final int LABEL = 0;
	public static final int TEXT = 1;
	public static final int CHOICE = 2;
	public static final int INTEGER = 3;
	public static final int BOOLEAN = 4;
	public static final int FLOAT = 5;
	public static final int DOUBLE = 6;	
	public static final int COLOR = 7;	
	public static final int CUSTOM = 8;	//the entire element is passed in
	
	public static int EXTRA_COLUMN_WIDTH = 23;
	//public static int MAX_COLUMN_WIDTH = 500;
	
	int [] columnWidths = new int[0];
	
	protected boolean [] autoSizeOnContents = new boolean[0];
	protected boolean [] enforceMinWidths = new boolean[0];
	protected boolean [] columnExpands = new boolean[0];
	protected int [] columnTypes = new int[0];
	
	protected TableViewer viewer;
	
	boolean resized = false;
	
	TableAdapterControlListener controlListener = new TableAdapterControlListener();
	
	public TableAdapter(Composite parent)
	{
		super(parent, SWT.BORDER);

		setLayout(new FillLayout());
		setBackground(parent.getBackground());
		
		viewer = new TableViewer(new Table(this, SWT.FULL_SELECTION | SWT.SINGLE | SWT.HIDE_SELECTION));		
		viewer.getTable().setHeaderVisible(true); 
		viewer.getTable().setLinesVisible(true);
		
		viewer.setContentProvider(this);
		viewer.setLabelProvider(this);
		viewer.setCellModifier(this);
		
		addControlListener(controlListener);
	}
	
	public TableViewer getViewer()
	{
		return viewer;
	}
	
	public void configure()
	{
		viewer.setInput("");
		
		GC gc = new GC(this);
		for (int i=0; i<viewer.getTable().getColumnCount(); i++)
		{
			int w = gc.stringExtent(viewer.getTable().getColumn(i).getText()).x + EXTRA_COLUMN_WIDTH;

			if (w > columnWidths[i]) columnWidths[i] = w;
			
			if (autoSizeOnContents[i])
			{
				w = gc.stringExtent(getLongestText(i)).x + getMaxImageWidth(i) + EXTRA_COLUMN_WIDTH;
				
				if (w > columnWidths[i]) columnWidths[i] = w;
			}
			
			viewer.getTable().getColumn(i).setWidth(columnWidths[i]);
		}
		gc.dispose();
		
		resized = false;
	}
	
	protected String getLongestText(int columnIndex)
	{
		String longestString = "";
		int maxStringLength = 0;
		
		Object [] elements = getElements(null);
	
		for (int i=0; i<elements.length; i++)
		{
			String columnText = getColumnText(elements[i], columnIndex);
			if (columnText.length() > maxStringLength)
			{
				maxStringLength = columnText.length();
				longestString = columnText;
			} 
		}
		
		return longestString;
	}
	
	protected int getMaxImageWidth(int columnIndex)
	{
		int maxImageWidth = 0;
		
		Object [] elements = getElements(null);
	
		for (int i=0; i<elements.length; i++)
		{
			Image image = getColumnImage(elements[i], columnIndex);
					
			if (image != null)
			{
				if (image.getImageData().width > maxImageWidth)
				{
					maxImageWidth = image.getImageData().width;
				} 	
			}
		}
		
		return maxImageWidth;
	}
	
	public void setColumnNames(String [] columnNames)
	{
		viewer.setColumnProperties(columnNames);
		
		TableColumn [] columns = new TableColumn[columnNames.length];
		
		autoSizeOnContents = new boolean[columnNames.length];
		enforceMinWidths = new boolean[columnNames.length];
		columnWidths = new int[columnNames.length];
		columnExpands = new boolean[columnNames.length];
		columnTypes = new int[columnNames.length];
		
		for (int i=0; i<columns.length; i++)
		{
			columns[i] = new TableColumn(viewer.getTable(), SWT.LEFT);
			
			columns[i].setText(columnNames[i]);
			
			columnWidths[i] = 23;
			enforceMinWidths[i] = true;
			columnExpands[i] = true;
			autoSizeOnContents[i] = false;
			columnTypes[i] = LABEL;
			
			columns[i].addControlListener(controlListener);
		}
		
		resized = false;
	}
	
	public String [] getColumnNames()
	{
		String [] columnNames = new String[viewer.getColumnProperties().length];
		for (int i=0; i<columnNames.length; i++) columnNames[i] = (String)viewer.getColumnProperties()[i];
		return columnNames;
	}

	public void setColumnEditor(CellEditor editor, int column) {
		CellEditor[] editors = viewer.getCellEditors();
//		System.out.println(editors.length);
		editors[column] = editor;
		viewer.setCellEditors(editors);
	}
	
	public void setColumnTypes(int [] columnTypes)
	{
		this.columnTypes = columnTypes;
		
		CellEditor[] editors = new CellEditor[columnTypes.length];

		for (int i=0; i<columnTypes.length; i++)
		{
			switch (columnTypes[i])
			{
				case LABEL:
					editors[i] = new TextCellEditor(viewer.getTable());
					break;
				case TEXT:
					editors[i] = new TextCellEditor(viewer.getTable());
					break;
				case CHOICE:
					editors[i] = new ComboBoxCellEditor(viewer.getTable(), getComboChoices(null, i));
					
					int w = editors[i].getControl().computeSize(SWT.DEFAULT, SWT.DEFAULT).x;
					if (w > columnWidths[i]) columnWidths[i] = w;

					break;
				case INTEGER:
					editors[i] = new TextCellEditor(viewer.getTable());
					((Text) editors[i].getControl()).addVerifyListener(
						new VerifyListener() 
						{
							public void verifyText(VerifyEvent e)
							{
								if (e.text.length() == 0)
								{
									e.doit = true;
								}
								else
								{
									try
									{
										Integer.parseInt(e.text);
										e.doit = true;
									}
									catch (NumberFormatException ex)
									{
										e.doit = false;
									}
								}
							}
						});
					
					break;
				case FLOAT:
					editors[i] = new TextCellEditor(viewer.getTable());
					((Text) editors[i].getControl()).addVerifyListener(new FloatVerifyListener((Text)editors[i].getControl())); 
					
					break;					
				case DOUBLE:
					editors[i] = new TextCellEditor(viewer.getTable());
					((Text) editors[i].getControl()).addVerifyListener(new DoubleVerifyListener((Text)editors[i].getControl())); 
					
					break;					
				case BOOLEAN:
					editors[i] = new CheckboxCellEditor(viewer.getTable());
					break;
				case COLOR:
					editors[i] = new ColorCellEditor(viewer.getTable());
					break;
			}
		}

		viewer.setCellEditors(editors);
	}
	
	public String [] getComboChoices(Object element, int columnIndex)
	{
		return new String[0];
	}

	public abstract Object[] getElements(Object inputObject);
	
	public int getRow(Object element)
	{
		Object [] elements = getElements(null);
		for (int i=0; i<elements.length; i++) if (element == elements[i]) return i;
		
		return -1; 
	}
	
	public int getColumn(String property)
	{
		for (int i=0; i<viewer.getColumnProperties().length; i++)
		{
			if (property.equals(viewer.getColumnProperties()[i]))
			{
				return i;
			}
		}
		return -1;
	}
		
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput){}
		
	public Image getColumnImage(Object element, int columnIndex)
	{
		return null;
	}
		
	public String getColumnText(Object element, int columnIndex)
	{
		return element.toString() + "[" + columnIndex + "]";
	}
	
	public RGB getColumnRGB(Object element, int columnIndex)
	{
		return null;
	}
	
	// Add a 0 to the end of a float if necessary
	public String appendFloat(String text)
	{
		if (text.charAt(text.length()-1) == '.')
			text = text + "0";				
		return text;
	}
		
	public Object getValue(Object element, String property)
	{
		int column = getColumn(property);
		switch (columnTypes[column])
		{
			case LABEL:
			case TEXT:
			case INTEGER:
				return getColumnText(element, column);
			case FLOAT:
			case DOUBLE:
				return appendFloat(getColumnText(element,column));
			case CHOICE:
				String [] choices = getComboChoices(element, column);
				for (int j=0; j<choices.length; j++)
				{
					if (getColumnText(element, column).equals(choices[j]))
					{
						return new Integer(j);
					}   
				}
				return new Integer(0);
			case BOOLEAN:
				return new Boolean(viewer.getTable().getItem(getRow(element)).getChecked());
			case COLOR:
				return getColumnRGB(element,column);
			case CUSTOM:
				return element;
			default:
				return null;
		}
	}
		
	public boolean canModify(Object element, String property) 
	{
		int column = getColumn(property);
		
		int possibleNewType = getColumnType(element, column);
		if (possibleNewType != columnTypes[column])
		{
			columnTypes[column] = possibleNewType;
			setColumnTypes(columnTypes);
		}
				
		switch (columnTypes[column])
		{
			case LABEL:
				return false;
			case TEXT:
				return true;
			case INTEGER:
				return true;
			case CHOICE:
				((ComboBoxCellEditor)viewer.getCellEditors()[column]).setItems(getComboChoices(element, column));
				return true;
			case BOOLEAN:
				return true;
			case FLOAT:
				return true;
			case DOUBLE:
				return true;
			case COLOR:
				return true;
			case CUSTOM:
				return true;
			default:
				return false;
		}			
	}
	
	protected int getColumnType(Object element, int column)
	{
		return columnTypes[column];
	}
	
	public void modify(Object element, String property, Object value)
	{
		int column = getColumn(property);
		try
		{
			switch (columnTypes[column])
			{
				case LABEL:
				case TEXT:
					valueChanged(((TableItem)element).getData(), column, (String)value);
					break;
				case INTEGER:
					if (!((String)value).equals(""))
						valueChanged(((TableItem)element).getData(), column, new Integer((String)value));
					break;
				case CHOICE:
					valueChanged(((TableItem)element).getData(), column, (Integer)value);
					break;					
				case BOOLEAN:
					valueChanged(((TableItem)element).getData(), column, (Boolean)value);
					break;
				case FLOAT:
					if (!((String)value).equals(""))				
						valueChanged(((TableItem)element).getData(),column, new Float((String)value));
					break;
				case DOUBLE:
					if (!((String)value).equals(""))				
						valueChanged(((TableItem)element).getData(),column, new Double((String)value));
					break;
				case COLOR:
					if (value != null)
						valueChanged(((TableItem)element).getData(),column, (RGB)value);
					break;
				case CUSTOM:
					if (value != null)
						valueChanged(((TableItem)element).getData(),column);
				default:
					break;
			}
			viewer.update(((TableItem)element).getData(), null);
		}
		catch (InvalidTableValueException e)
		{
			MessageDialog.openError(getShell(), "Cannot Update Value", "Could not change " + property + " value.\n\n" + e.getClass().getName() + "\n" + e.getMessage());
		}
	}
	
	public void valueChanged(Object element, int column, String newValue) throws InvalidTableValueException
	{}
	
	public void valueChanged(Object element, int column, Float newValue) throws InvalidTableValueException
	{}
	
	public void valueChanged(Object element, int column, Double newValue) throws InvalidTableValueException
	{}

	public void valueChanged(Object element, int column, Boolean newValue) throws InvalidTableValueException
	{}
		
	public void valueChanged(Object element, int column, Integer newValue) throws InvalidTableValueException
	{}

	public void valueChanged(Object element, int column, RGB rgb) throws InvalidTableValueException
	{}

	public void valueChanged(Object element, int column) throws InvalidTableValueException
	{}
	
	public void addListener(ILabelProviderListener listener){}
 
	public void removeListener(ILabelProviderListener listener){}
		
	public boolean isLabelProperty(Object element, String property){return false;}
		
	public void dispose(){}
	
	public void expandColumns()
	{
		for (int i=0; i<((TableViewer)viewer).getTable().getColumnCount(); i++)
			((TableViewer)viewer).getTable().getColumn(i).removeControlListener(controlListener);
			
		int fixedColumnWidths = 0;
		int expandableColumnWidths = 0;
		int expandableColumnCount = 0;
		for (int i=0; i<viewer.getTable().getColumnCount(); i++)
		{
			if (columnExpands[i])
			{
				expandableColumnWidths += ((TableViewer)viewer).getTable().getColumn(i).getWidth();
				expandableColumnCount++;
			}
			else
			{
				fixedColumnWidths += ((TableViewer)viewer).getTable().getColumn(i).getWidth();
			}
		}
			
		int diff = (this.getSize().x - 4) - (expandableColumnWidths + fixedColumnWidths); 				
			
		if ((diff > 0 || !resized) && expandableColumnCount > 0)
		{
			for (int i=0; i<viewer.getTable().getColumnCount(); i++)
			{
				int extra = diff;
				if (expandableColumnCount > 0) extra = diff/expandableColumnCount;
				
				if (columnExpands[i])
				{
					int newWidth = ((TableViewer)viewer).getTable().getColumn(i).getWidth() + extra;
					
					if (enforceMinWidths[i] && newWidth < columnWidths[i])
					{
						expandableColumnCount--;
						diff -= columnWidths[i] - newWidth;
						newWidth = columnWidths[i];
					}
					
					((TableViewer)viewer).getTable().getColumn(i).setWidth(newWidth);
				}
			}
		}
				
		for (int i=0; i<((TableViewer)viewer).getTable().getColumnCount(); i++)
			((TableViewer)viewer).getTable().getColumn(i).addControlListener(controlListener);
		
	}

	class TableAdapterControlListener extends ControlAdapter
	{
		public void controlResized(ControlEvent event)
		{
			Object source = event.getSource();
			
			if (source == TableAdapter.this)
			{
				expandColumns();
			}
			else
			{
				for (int i=0; i<viewer.getTable().getColumnCount(); i++)
				{
					if (source == ((TableViewer)viewer).getTable().getColumn(i))
					{
						resized = true;
						columnWidths[i] = ((TableViewer)viewer).getTable().getColumn(i).getWidth();
						break;
					}
				}
			}
		}
	}
}
