package org.eclipse.hyades.logging.log4j;

import org.apache.log4j.Layout;
import org.apache.log4j.spi.LoggingEvent;
import org.eclipse.hyades.internal.logging.core.XmlGenerator;
import org.eclipse.hyades.logging.core.IExternalizableToXml;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/** 
 * Extension of the Log4J <code>org.apache.log4j.Layout</code> abstract
 * class used by <code>org.apache.log4j.Appender</code> instances to format 
 * <code>org.apache.log4j.spi.LoggingEvent</code> instances to XML.
 * <p>
 * If the <code>org.apache.log4j.spi.LoggingEvent</code> instance contains an 
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> as its
 * <code>message</code> (e.g. instance of the 
 * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code> interface), 
 * the encapsulated Common Base Event is serialized to XML based on the Common Base 
 * Event v.1.0.1 specification.  Otherwise, the <code>org.apache.log4j.spi.LoggingEvent</code>'s
 * <code>message</code> is serialized to XML based on a generic serialization algorithm as 
 * implemented in <code>org.eclipse.hyades.internal.logging.core.XmlGenerator</code>.
 * <p>
 * An instance of this class will be returned from the <code>getLayout()</code> 
 * API on <code>org.apache.log4j.Appender</code> instances.
 * <p>  
 * Users may configure a <code>org.apache.log4j.Appender</code> to use the 
 * <code>XmlLayout</code> by adding the following the entry to the 
 * <code>&lt;appender&gt;</code> element in the <code>log4j.xml</code> 
 * properties file:
 * <p>
 * <code>
 * &lt;!-- XmlLayout --&gt;
 * &lt;!-- Serializes events to XML: --&gt;
 * &lt;layout class="org.eclipse.hyades.logging.log4j.XmlLayout" /&gt;
 * </code>
 * <p>
 * Alternatively, an instantiation of this layout class may be set directly on
 * <code>org.apache.log4j.Appender</code> instances by using the 
 * <code>setLayout()</code> API.
 * <p>
 * NOTE: The Log4J classes must be on the CLASSPATH at run-time to
 * utilize this appender class.
 * <p>
 *  
 * 
 * @author    Paul E. Slauenwhite
 * @version   December 7, 2004
 * @since     July 20, 2004
 * @see       org.apache.log4j.Layout
 * @see       org.apache.log4j.spi.LoggingEvent
 */
public class XmlLayout extends Layout {

    /**
     * Private reference to the generic XML serialization class.
     */
    private XmlGenerator xmlGenerator = null;

    /**
     * Private static reference to the platform-dependent line separator character.
     */
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");

    /**
     * Generates the XML representation of the parameter 
     * <code>org.apache.log4j.spi.LoggingEvent</code> instance.
     * <p>
     * If the <code>org.apache.log4j.spi.LoggingEvent</code> instance contains an 
	 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> as its
	 * <code>message</code> (e.g. instance of the 
	 * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code> interface), 
	 * the encapsulated Common Base Event is serialized to XML based on the Common Base 
	 * Event v.1.0.1 specification.  Otherwise, the <code>org.apache.log4j.spi.LoggingEvent</code>'s
	 * <code>message</code> is serialized to XML based on a generic serialization algorithm as 
	 * implemented in <code>org.eclipse.hyades.internal.logging.core.XmlGenerator</code>.
     * <p>
     * 
     * @param loggingEvent The <code>org.apache.log4j.spi.LoggingEvent</code> instance to be formatted to XML.
     * @return The XML representation of the parameter <code>org.apache.log4j.spi.LoggingEvent</code> instance.
     * @see org.apache.log4j.Layout#format(LoggingEvent)
     */
    public String format(LoggingEvent loggingEvent) {

        if(loggingEvent != null){
            
            Object message = loggingEvent.getMessage();

            if (message instanceof IExternalizableToXml){
                return ((((IExternalizableToXml) message).externalizeCanonicalXmlString()).concat(LINE_SEPARATOR));
            }
            
            if(xmlGenerator == null){
                xmlGenerator = new XmlGenerator();
            }
            
            xmlGenerator.reset(null, true, 4);
            
            return (xmlGenerator.objectToXML(message).concat(LINE_SEPARATOR));
        }

        return "<null/>";
    }

    /**
     * If the layout handles the throwable object contained within LoggingEvent, 
     * then the layout should return false. Otherwise, if the layout ignores 
     * throwable object, then the layout should return true. 
     * 
     * @return boolean true, if the layout handles the throwable object contained within LoggingEvent, other false.
     * @see org.apache.log4j.Layout#ignoresThrowable()
     */
    public boolean ignoresThrowable() {
        return true;
    }

    /**
     * Activate the options that were previously set with calls to option setters. 
     * This allows to defer activiation of the options until all options have been set. 
     * This is required for components which have related options that remain ambigous 
     * until all are set.  For example, the FileAppender has the File and Append options 
     * both of which are ambigous until the other is also set.
     * 
     * @see org.apache.log4j.spi.OptionHandler#activateOptions()
     */
    public void activateOptions() {
    }
}
