/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.manual.runner;

import java.io.OutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;

import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.remote.CustomCommandHandler;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;
import org.eclipse.hyades.test.common.agent.ComptestAgent;
import org.eclipse.hyades.test.common.event.ExecutionEvent;
import org.eclipse.hyades.test.common.event.MessageEvent;
import org.eclipse.hyades.test.common.util.BaseString;
import org.eclipse.hyades.test.manual.runner.model.util.EventLogger;
import org.eclipse.hyades.test.manual.runner.model.util.ModelUtil;
import org.eclipse.hyades.test.manual.runner.ui.RemoteAppFrame;

/**
 * @author marcelop
 * @since 1.0.2
 */
public class HyadesManualRunner
{
	/**
	 * This class provides a OutputStrema that can be used to override stdout and stderr.
	 * @author rduggan
	 */
	private static class AgentConsoleStream 
	extends OutputStream
	{
		/* The types of streams we may have */
		public static final int OUT = 0;
		public static final int ERR = 1;

		/* Our mode (type) */
		private int mode;

		/* Agent to log the message to */
		private RemoteComponentSkeleton agent;
		private String hierarchyId;

		/* Buffer to append data to until we get a newline character */
		private StringBuffer buffer = new StringBuffer();

		public AgentConsoleStream(RemoteComponentSkeleton agent, int mode, String hierarchyId)
		{
			this.mode = mode;
			this.agent = agent;
			this.hierarchyId = hierarchyId;
		}

		/**
		 * Provide the write function.  We will be invoked by the PrintWriter here.
		 * RKD:  Does this need to be synchonized?  Will we be invoked by the PrintWriter in
		 * a thread-safe manner?  I will assume so but this may need to change.
		 * @see java.io.OutputStream#write(int)
		 */
		public void write(int b)
		{
			buffer.append((char)b);
			if(b == '\n')
			{
				MessageEvent messageEvent = new MessageEvent();
				messageEvent.setText(buffer.toString());
				messageEvent.setSeverity(MessageEvent.INFO);
				messageEvent.setOwnerId(hierarchyId);

				if (mode == OUT)
				{
					messageEvent.setText("System.out:\n" + buffer.toString());
				}
				else if (mode == ERR)
				{
					messageEvent.setText("System.err:\n" + buffer.toString());
				}

				/* Log the message */
				agent.logMessageUTF8(messageEvent.toString());

				/* Create the next line */
				buffer = new StringBuffer();
			}
		}
	}

	private boolean isOKToStart = false;
	private ComptestAgent comptestAgent;
	private String testScript;

	public static void main(String[] args)
	{
				
		/* Create our agent and wait for start command */
		final RemoteComponentSkeleton agent = new RemoteComponentSkeleton("Executor$" + args[args.length-1], "tester");

		/* Instanciate an instance of this class */
		final HyadesManualRunner runner = new HyadesManualRunner(agent);

		// Add a command listener that can okay the starting of the test.
		agent.addCommandListener(new CustomCommandHandler()
		{
			public void handleCommand(CustomCommand command)
			{
				if(command.getData().equals(IControlMessage.START))
				{
					runner.setOKToStart(true);
				}
				else
				{
					runner.setTestScript(command.getDataBinary());
				}
			}
		});

		/* Handshake with the Agent controller */
		try
		{
			agent.initialize();
		}
		catch (Throwable e)
		{
		}

		/* Wait for our start.  We will never wait more that 3 minutes */
		while (!runner.isOKToStart())
		{
			synchronized (runner)
			{
				try
				{
					runner.wait(3 * 60000);
					if (!runner.isOKToStart())
					{
						/* Try and tell the client we are timing out */
						agent.sendMessageToAttachedClient("Test was not started after 3 minutes.  Exiting testcase.", 0);

						/* We have timed out.  exit */
						System.exit(-1);
					}
				}
				catch (InterruptedException e)
				{

				}
			}
		}

		try
		{
			if((runner.getTestScript() == null) && (args.length > 1))
			{
				StringBuffer sb = new StringBuffer();
				for (int i = 1, maxi = args.length; i < maxi; i++)
					sb.append(" ").append(args[i]);
				runner.setTestScript(sb.toString().trim());
			}
						
			/* Setup stdout and stder */
			String hierarchyId = args[0] + "(1)";
			System.setOut(new PrintStream(new AgentConsoleStream(agent, AgentConsoleStream.OUT, hierarchyId)));
			System.setErr(new PrintStream(new AgentConsoleStream(agent, AgentConsoleStream.ERR, hierarchyId)));

			runner.initializeRemoteApp();

			RemoteAppFrame remoteAppFrame = new RemoteAppFrame()
			{
				/**
				 * @see org.eclipse.hyades.test.manual.runner.ui.RemoteAppFrame#doClose(boolean)
				 */
				public void doClose(boolean confirmEnd)
				{
					super.doClose(confirmEnd);
					agent.sendMessageToAttachedClient("Testcase completed successfuly", 0);
					runner.exit();
					System.exit(0);
				}
			};
			remoteAppFrame.show();
		}
		catch(Throwable e)
		{
			/* We could not find the class (or something else, report the error */
			MessageEvent messageEvent = new MessageEvent();
			messageEvent.setText(BaseString.getStackTrace(e));
			messageEvent.setSeverity(MessageEvent.ERROR);
			messageEvent.setOwnerId(args[0]);
			messageEvent.setId("ROOT");
			agent.logMessageUTF8(messageEvent.toString());
		}
	}

	/**
	 * Constructor for HyadesManualRunner
	 * @param agent
	 */
	public HyadesManualRunner(RemoteComponentSkeleton agent)
	{
		comptestAgent = new ComptestAgent(agent);
		ModelUtil.setEventLogger(new EventLogger()
		{
			public void log(ExecutionEvent executionEvent)
			{
				logEvent(executionEvent);
			}
		});
	}

	/**
	 * Is it okay to continue starting the test?
	 * @return <code>true</code> if it is okay to start the test,
	 * <code>false</code> otherwise
	 */
	public synchronized boolean isOKToStart()
	{
		return isOKToStart;
	}

	// Set the "okay to start" flag and notify any threads that are waiting
	// on this object's monitor.
	protected synchronized void setOKToStart(boolean isOK)
	{
		isOKToStart = isOK;
		notifyAll();
	}

	protected void start()
	{
		comptestAgent.write("<EXECUTION>");
	}

	protected void exit()
	{
		comptestAgent.write("</EXECUTION>");
	}

	/**
	 * @see org.eclipse.hyades.test.common.junit.HyadesTestRunner#writeEvent(org.eclipse.hyades.test.common.event.ExecutionEvent)
	 */
	protected void logEvent(ExecutionEvent executionEvent)
	{
		if((comptestAgent != null) && (executionEvent != null))
			comptestAgent.write(executionEvent.toString());
	}
	
	protected void initializeRemoteApp()
	throws Exception
	{
		start();
		new RemoteApp().initialize(getTestScript());
	}
	
	protected void setTestScript(byte[] data)
	{
		try
		{
			testScript = new String(data, "UTF-8");
		}
		catch (UnsupportedEncodingException e)
		{
		}
	}
	
	protected void setTestScript(String testScript)
	{
		this.testScript = testScript;
	}
	
	protected String getTestScript()
	{
		return testScript;
	}
}
