/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.manual.runner.model.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.eclipse.hyades.test.common.util.XMLUtil;
import org.eclipse.hyades.test.manual.runner.model.Action;
import org.eclipse.hyades.test.manual.runner.model.IActionOwner;
import org.eclipse.hyades.test.manual.runner.model.Loop;
import org.eclipse.hyades.test.manual.runner.model.NamedElement;
import org.eclipse.hyades.test.manual.runner.model.Test;
import org.eclipse.hyades.test.manual.runner.model.TestCase;
import org.eclipse.hyades.test.manual.runner.model.TestInvocation;
import org.eclipse.hyades.test.manual.runner.model.TestSuite;

/**
 * @author marcelop
 * @since 1.0.2
 */
public class ScriptParser
{
	private Map namedElementById;
	private List verdicts;
	private Map elementByTestSuite;
	private TestSuite root;
	private List testInvocations;
	
	private int executionId;
	
	public void dispose()
	{
		if(namedElementById != null)
		{
			namedElementById.clear();
			namedElementById = null;
		}
		
		if(verdicts != null)
		{
			verdicts.clear();
			verdicts = null;
		}
		
		if(elementByTestSuite != null)
		{
			elementByTestSuite.clear();
			elementByTestSuite = null;
		}
		
		if(testInvocations != null)
		{
			testInvocations.clear();
			testInvocations = null;
		}
		
		root = null;
	}
	
	public TestSuite parse(byte[] xmlBytes)
	throws Exception
	{
		Element testScriptElement = XMLUtil.loadDom(new String(xmlBytes, "UTF-8"), "testScript");
		if(testScriptElement != null)
		{
			NodeList nodeList = XMLUtil.getChildrenByName(testScriptElement, "verdicts");
			if(nodeList != null)
			{
				for(int i=0; i<nodeList.getLength(); i++)
				{
					if(nodeList.item(i) instanceof Element)
					{
						Element verdictsElement = (Element)nodeList.item(i);
						if(verdictsElement != null)
						{
							verdicts = new ArrayList();
							parseVerdicts(verdictsElement);
							break;
						}
					}
				}
			}

			nodeList = XMLUtil.getChildrenByName(testScriptElement, "testSuite");
			if(nodeList != null)
			{
				for(int i=0; i<nodeList.getLength(); i++)
				{
					if(nodeList.item(i) instanceof Element)
					{
						Element testSuiteElement = (Element)nodeList.item(i);
						if(testSuiteElement != null)
						{
							root = new TestSuite();
							root.setExecutionId(0);
							
							namedElementById = new HashMap();
							elementByTestSuite = new HashMap();
							testInvocations = Collections.EMPTY_LIST;
							
							parse(root, testSuiteElement);
							adjust(root);
							testInvocations = getTestInvocations(root);
							
							break;
						}
					}
				}
			}
		}
		
		return root;			
	}
	
	public List getTestInvocations()
	{
		return testInvocations;
	}
	
	private List getTestInvocations(IActionOwner actionOwner)
	{
		int initialExecutionId = executionId;
		List list = new ArrayList();
		for(Iterator i = actionOwner.getActions().iterator(); i.hasNext();)
		{
			Action action = (Action)i.next();
			action.setExecutionId(executionId);
			if(action.isSynchronous())
				executionId++;
				
			if(action instanceof IActionOwner)
			{
				list.addAll(getTestInvocations((IActionOwner)action));
			}
			else if(action instanceof TestInvocation)
			{
				TestInvocation testInvocation = (TestInvocation)action;
				list.add(testInvocation);

				if(testInvocation.getTest() == null)
					continue;
					
				if(testInvocation.getTest() instanceof IActionOwner)
				{
					testInvocation.getTest().setExecutionId(testInvocation.getExecutionId());
					list.addAll(getTestInvocations((IActionOwner)testInvocation.getTest()));
				}
			}
		}
		
		if(actionOwner instanceof Action)
		{ 
			if(!((Action)actionOwner).isSynchronous())
				executionId = initialExecutionId;
		}
		else if(actionOwner instanceof TestSuite)
		{
			TestInvocation testInvocation = ((TestSuite)actionOwner).getTestInvocation();
			if((testInvocation != null) && (!testInvocation.isSynchronous()))
				executionId = initialExecutionId;
		}
				
		return list;		
	}
	
	public List getVerdics()
	{
		if(verdicts == null)
			return Collections.EMPTY_LIST;
		return new ArrayList(verdicts);
	}
	
	protected void adjust(IActionOwner actionOwner)
	{
		for (Iterator i = actionOwner.getActions().iterator(); i.hasNext();)
		{
			Action action = (Action)i.next();
			if(action instanceof IActionOwner)
			{
				adjust((IActionOwner)action);
			}
			else if(action instanceof TestInvocation)
			{
				TestInvocation testInvocation = (TestInvocation)action;
				if(testInvocation.getTestId() == null)
					continue;

				if(testInvocation.getTest() != null)
					continue;
					
				Object object = namedElementById.get(testInvocation.getTestId());
				if(object instanceof TestCase)
				{
					testInvocation.setTest((Test)object);
				}
				else if(object instanceof TestSuite)
				{
					TestSuite ts = new TestSuite();
					testInvocation.setTest(ts);
					ts.setTestInvocation(testInvocation);
					
					parse(ts, (Element)elementByTestSuite.get(object));
					adjust(ts);
				}
				else
					continue;
				
				if(testInvocation.getTest().getDescription() != null)
				{
					if(testInvocation.getDescription() != null)
						testInvocation.setDescription(testInvocation.getTest().getDescription() + "\n\n" + testInvocation.getDescription());
					else
						testInvocation.setDescription(testInvocation.getTest().getDescription());
				}
			}
		}
	}
	
	protected void parseVerdicts(Element element)
	{
		NodeList nodeList = XMLUtil.getChildrenByName(element, "verdict");
		if(nodeList != null)
		{
			for(int i=0, maxi=nodeList.getLength(); i<maxi; i++)
			{
				Element verdictElement = (Element)nodeList.item(i);
				try
				{
					verdicts.add(new Verdict(XMLUtil.getValue(verdictElement, "label"), Integer.parseInt(XMLUtil.getValue(verdictElement, "value")), new Boolean(XMLUtil.getValue(verdictElement, "isDefault")).booleanValue()));				
				}
				catch (NumberFormatException e)
				{
				}
			}
		}
	}
	
	protected void parse(TestSuite testSuite, Element element)
	{
		parse((Test)testSuite, element);
		elementByTestSuite.put(testSuite, element);

		testSuite.setLocation(XMLUtil.getValue(element, "location"));

		if(testSuite.getTestInvocation() == null)
		{
			NodeList referencedTestSuites = XMLUtil.getChildrenByName(element, "referencedTestSuites");
			if(referencedTestSuites != null)
			{
				for(int i=0, maxi=referencedTestSuites.getLength(); i<maxi; i++)
				{
					Node node = referencedTestSuites.item(i);
					if(!(node instanceof Element))
						continue;
				
					NodeList referencedTestSuite = XMLUtil.getChildrenByName((Element)node, "testSuite");
					if(referencedTestSuite != null)
					{
						for(int j=0, maxj=referencedTestSuite.getLength(); j<maxj; j++)
						{
							TestSuite ts = new TestSuite();
							parse(ts, (Element)referencedTestSuite.item(j));
						}
					}
				}
			}
		}
		
		NodeList testCaseGroups = XMLUtil.getChildrenByName(element, "testCases");
		if(testCaseGroups != null)
		{
			for(int i=0, maxi=testCaseGroups.getLength(); i<maxi; i++)
			{
				Node node = testCaseGroups.item(i);
				if(!(node instanceof Element))
					continue;
					
				NodeList testCases = XMLUtil.getChildrenByName((Element)node, "testCase");
				if(testCases != null)
				{
					for(int j=0, maxj=testCases.getLength(); j<maxj; j++)
					{
						TestCase testCase = new TestCase();
						testCase.setTestSuite(testSuite);
						parse(testCase, (Element)testCases.item(j));
					}
				}
			}
		}

		NodeList behaviorNodeList = XMLUtil.getChildrenByName(element, "behavior");
		if(behaviorNodeList != null)
		{
			for(int i=0, maxi=behaviorNodeList.getLength(); i<maxi; i++)
			{
				NodeList actionNodeList = behaviorNodeList.item(i).getChildNodes();
				if(actionNodeList != null)
					parse(testSuite, actionNodeList);
			}
		}
	}
	
	protected void parse(IActionOwner actionOwner, NodeList nodeList)
	{
		int syncCount = 0;
		for(int i=0, maxi=nodeList.getLength(); i<maxi; i++)
		{
			Node node = nodeList.item(i);
			if(!(node instanceof Element))
				continue;
									
			Element actionElement = (Element)node;
			String tagName = actionElement.getTagName();
			
			if("testInvocation".equals(tagName))
			{
				TestInvocation testInvocation = new TestInvocation();
				testInvocation.setOwner(actionOwner);
				parse(testInvocation, actionElement);
			}
			else if("loop".equals(tagName))
			{
				Loop loop = new Loop();
				loop.setOwner(actionOwner);
				parse(loop, actionElement);
				
				if(loop.getIterations() > 1)
				{
					for(int j=2; j<=loop.getIterations(); j++)
					{
						loop = new Loop();
						loop.setOwner(actionOwner);
						loop.setIteration(j);
						parse(loop, actionElement);
					}
				}
			}
		}
	}
	
	protected void parse(TestCase testCase, Element element)
	{
		parse((Test)testCase, element);
	}
	
	protected void parse(Test test, Element element)
	{
		parse((NamedElement)test, element);
	}

	protected void parse(Loop loop, Element element)
	{
		parse((Action)loop, element);
		
		loop.setIterations(XMLUtil.getIntValue(element, "iterations"));
		NodeList nodeList = element.getChildNodes();
		if(nodeList != null)
			parse(loop, nodeList);
	}

	protected void parse(TestInvocation testInvocation, Element element)
	{
		parse((Action)testInvocation, element);
			
		testInvocation.setTestId(XMLUtil.getValue(element, "testId"));		
	}

	protected void parse(Action action, Element element)
	{
		parse((NamedElement)action, element);
	
		action.setSynchronous(Boolean.valueOf(XMLUtil.getValue(element, "synchronous")).booleanValue());
	}

	protected void parse(NamedElement namedElement, Element element)
	{
		namedElement.setId(XMLUtil.getValue(element, "id"));
		namedElement.setName(XMLUtil.getValue(element, "name"));
		namedElement.setDescription(XMLUtil.getValue(element, "description"));
		
		namedElementById.put(namedElement.getId(), namedElement);
	}
}