/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.loaders.util.LoadersUtils;

/**
 * @author slavescu
 */
public abstract class EObjectsTraverser {
	protected static final int AFTER_CHILDREN = 3;
	protected static final int BEFORE_CHILDREN = 1;

	private Collection contents;
	private List visitors;

	public EObjectsTraverser(Collection contents) {
		this.contents = contents;
		visitors = new FastList();
	}

	public void dispose() {
		visitors.clear();
		contents = null;
	}

	protected boolean afterChildren(EObject element) {
		return callVisitors(AFTER_CHILDREN, element);
	}

	protected boolean beforeChildren(EObject element) {
		return callVisitors(BEFORE_CHILDREN, element);
	}

	public EObjectsTraverser registerVisitors(EObjectVisitor visitor) {
		if (visitor != null)
			visitors.add(visitor);
		return this;
	}

	public EObjectsTraverser deregisterVisitor(EObjectVisitor visitor) {
		if (visitor != null)
			visitors.remove(visitor);
		return this;
	}

	public EObjectsTraverser deregisterAllVisitors() {
		visitors.clear();
		return this;
	}

	protected boolean callVisitors(int flag, EObject element) {
		boolean ret = true;
		for (int i = visitors.size() - 1; i >= 0; i--) {
			try {
				switch (flag) {
					case BEFORE_CHILDREN :
						ret = ((EObjectVisitor) visitors.get(i)).beforeChildren(element) && ret;
						break;

					case AFTER_CHILDREN :
						ret = ((EObjectVisitor) visitors.get(i)).afterChildren(element) && ret;
						break;
				}
			} catch (RuntimeException re) {
				LoadersUtils.log(re);
			}
		}
		return ret;
	}

	public boolean traverse() {
		return traverse(contents.iterator());
	}

	protected boolean traverse(Iterator iter) {
		while (iter.hasNext()) {
			EObject traversedElement = (EObject) iter.next();
			EObject visitorElement = getElement(traversedElement);
			if (!beforeChildren(visitorElement))
				return false;

			traverseEObject(traversedElement);

			if (!afterChildren(visitorElement))
				return false;
		}
		return true;
	}

	protected boolean traverse(List children) {
		for (Iterator iter = children.iterator(); iter.hasNext();) {
			EObject traversedElement = (EObject) iter.next();
			EObject visitorElement = getElement(traversedElement);
			if (!beforeChildren(visitorElement))
				return false;

			traverseEObject(visitorElement);

			if (!afterChildren(visitorElement))
				return false;

		}
		return true;
	}

	protected abstract boolean traverseEObject(EObject object);
	protected abstract EObject getElement(EObject traversedElement);
}
