/**********************************************************************
Copyright (c) 2004 Hyades project.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Common Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/cpl-v10.html

Contributors:
 IBM Rational - initial implementation
**********************************************************************/
package org.eclipse.hyades.internal.config.generator;

import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.util.Enumeration;
import java.util.Hashtable;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

public class ConfigFile extends DefaultHandler {
	protected static String TAG = ConfigUtility.getString("Config.AgentControllerConfig.Tag");
	private static String lineSeparator = System.getProperty("line.separator");

	protected DocumentBuilderFactory docBuildFactory = null;
	protected DocumentBuilder docBuild = null;
	protected Document doc = null;
	protected Element acConfig = null;
	protected Element holder = null;

	private int size = 100;
	private Hashtable hash = new Hashtable(size);

	private String sr = PlatformObject.sr;
	private String filename = null;

	public ConfigFile(String name) {
		docBuildFactory = DocumentBuilderFactory.newInstance();
		try {
			docBuild = docBuildFactory.newDocumentBuilder();
			createNewDocument();
		} catch (ParserConfigurationException e) {
			Logger.err("Cannot create document builder");
		}

		filename = name;
		File f = new File(filename);
		if(f.exists()) {
			loadExistingDocument(filename);
		}
	}

	public void saveToFile() {
		if(filename != null) {
			saveToFile(filename);
		}
	}

	public void saveToFile(String fileName) {
		PrintWriter writer = null;

		try {
			writer = new PrintWriter(new FileWriter(fileName));
		}
		catch(Exception e) {
			Logger.err(ConfigUtility.getString("Config.Writer.Error.CannotWriteFile") + " " + fileName);
			Logger.err(e.getMessage());
		}

		if(writer != null) {
			writer.print(toString());
			writer.flush();
			writer.close();
		}
	}

	public String toString() {
		StringBuffer strbuf = new StringBuffer();
		String encoding = PlatformObject.encoding; // Set a default value
		String version = "1.0"; // Set a default value

		//
		// XML header information
		//
		strbuf.append("<?xml");

		//
		// Version and encoding
		//
		strbuf.append(" version=\"" + version + "\"");
		strbuf.append(" encoding=\"" + encoding + "\"");

		//
		// Closing the header
		//
		strbuf.append("?>" + ConfigFile.lineSeparator);

		//
		// Print each child node
		//
		NodeList nodes = holder.getChildNodes();
		if(nodes != null) {
			for(int i = 0; i < nodes.getLength(); i++) {
				strbuf.append(ConfigUtility.print(nodes.item(i)));
			}
		}

		return strbuf.toString();
	}

	public void generateConfiguration() {
		generateHyadesConfiguration();
	}

	/**
	 * Generate the basic Hyades Data Collection Engine configuration
	 */
	public void generateHyadesConfiguration() {
	}

	public void init(Hashtable hash) {
		Enumeration keys = hash.keys();
		while(keys.hasMoreElements()) {
			String key = (String)keys.nextElement();
			String value = (String)hash.get(key);
			setValue(key, value);
		}
	}

	public void createNewDocument() {
		DOMImplementation dom = docBuild.getDOMImplementation();
		doc = dom.createDocument(null, TAG, null);
		holder = doc.createElement("ElementHolder");
	}

	public void loadExistingDocument(String file) {
		SAXParser saxp = null;
		XMLReader parser = null;

		//
		// Create the parser
		//
		try {
			saxp = SAXParserFactory.newInstance().newSAXParser();
			parser = saxp.getXMLReader();
			parser.setContentHandler(this);
			parser.setErrorHandler(this);
		} catch (Exception e) {
			Logger.err(ConfigUtility.getString("Config.Parser.Error.CreateParser") + " " + parser);
			Logger.err(e.getMessage());
		}

		if(parser != null) {
			try {
				parser.parse(file);
			} catch (Exception e) {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.ParseFile") + " " + file);
				Logger.err(e.getMessage());
				if(e instanceof SAXException) {
					Exception e1 = ((SAXException)e).getException();
					if(e1 != null) {
						Logger.err(e1.getMessage());
					}
				}
			}
		}
	}

	public void startElement(String uri, String local, String raw, Attributes attrs) throws SAXException {
		//
		// UserDefinition
		//
		if(raw.equals(UserDefinition.TAG)) {
			String authenticationType = null;
			String name = null;

			for(int i = 0; i < attrs.getLength(); i++) {
				if(attrs.getQName(i).equals(ConfigUtility.getString("Config.UserDefinition.AuthenticationType.Tag"))) {
					authenticationType = attrs.getValue(i);
				}
				else if(attrs.getQName(i).equals(ConfigUtility.getString("Config.UserDefinition.Name.Tag"))) {
					name = attrs.getValue(i);
				}
				else {
					Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getQName(i));
				}
			}
			if((authenticationType != null) && (name != null)) {
				setValue("SECURITY", "true");
				if(getValue("USERS") == null) {
					setValue("USERS", name);
				}
				else {
					setValue("USERS", getValue("USERS") + "," + name);
				}

			}
		}

		//
		// Variable
		//
		else if(raw.equals(Variable.TAG)) {
			String name = null;
			String value = null;
			String position = null;

			for(int i = 0; i < attrs.getLength(); i++) {
				if(attrs.getQName(i).equals(ConfigUtility.getString("Config.Variable.Name.Tag"))) {
					name = attrs.getValue(i);
				}
				else if(attrs.getQName(i).equals(ConfigUtility.getString("Config.Variable.Value.Tag"))) {
					value = attrs.getValue(i);
				}
				else if(attrs.getQName(i).equals(ConfigUtility.getString("Config.Variable.Position.Tag"))) {
					position = attrs.getValue(i);
				}
				else {
					Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getQName(i));
				}
			}

			if((name != null) && (value != null) && (position != null)) {
				//
				// Populate the hash table, do not add CLASSPATH or PATH
				//
				if(!name.equals("CLASSPATH") && !name.equals("PATH")) {
					setValue(name, value);
				}
			}
		}

		//
		// Allow
		//
		else if(raw.equals(Allow.TAG)) {
			String name = null;

			for(int i = 0; i < attrs.getLength(); i++) {
				if(attrs.getQName(i).equals(ConfigUtility.getString("Config.Allow.Host.Tag"))) {
					name = attrs.getValue(i);
				}
				else {
					Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getQName(i));
				}
			}

			if(name != null) {
				if(name.toLowerCase().equals("all") || name.toLowerCase().equals("local")) {
					setValue("ALLOW", name.toUpperCase());
					removeKey("HOSTS");
				}
				else {
					setValue("ALLOW", "CUSTOM");
					if(getValue("HOSTS") == null) {
						setValue("HOSTS", name);
					}
					else {
						setValue("HOSTS", getValue("HOSTS") + "," + name);
					}
				}
			}
		}

		//
		// Unrecognized element name
		//
		else {
			Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedElement") + " " + local);
		}
	}

	/**
	 * @return Returns the acConfig.
	 */
	public Element getAcConfig() {
		return acConfig;
	}

	public String getValue(String tag) {
		return (String)hash.get(tag);
	}

	public void setValue(String tag, String value) {
		if((tag != null) && (value != null)) {
			hash.put(tag, value);
		}
	}

	public void removeKey(String tag) {
		hash.remove(tag);
	}

	public boolean isExist(String tag) {
		return hash.containsKey(tag);
	}
	/**
	 * @return Returns the holder.
	 */
	public Element getHolder() {
		return holder;
	}
	/**
	 * @return Returns the doc.
	 */
	public Document getDoc() {
		return doc;
	}
}
