/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.extensions.DBCommandFactory;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;
/**
 * This command deletes the given objects from the database, including all
 * contained objects.
 */
public class DeleteObjects extends DBCommand {
	protected DatabaseType type;
	protected EReference reference;
	protected List classesAndIds;
	protected Map classesToIds;
	protected Map classesToMetadata;

	public DeleteObjects(JDBCHelper helper, DBMap map, List classesAndIds) {
		super(helper, map);
		this.type = helper.getDatabaseType();
		this.classesAndIds = classesAndIds;
	}

	public Object execute() throws Exception {
		try {
			PerfUtil p = new PerfUtil("DeleteObjects.execute()",true);
			computeObjectsToDelete();
			deleteObjects();
			p.stopAndPrintStatus();
			helper.commitTransaction();
		} catch (Exception e) {
			helper.rollbackTransaction();
			throw e;
		}
		return null;
	}

	protected void computeObjectsToDelete() throws Exception {
		HashMap classesToIds = new HashMap();

		for (int i = 0, l = classesAndIds.size(); i < l; i += 2) {
			EClass eClass = (EClass) classesAndIds.get(i);
			List ids = (List) classesAndIds.get(i + 1);
			addToMap(eClass, ids);
			computeObjectsToDelete(eClass, ids);
		}
	}

	protected void addToMap(EClass eClass, List ids) {
		if (ids.isEmpty())
			return;

		if (classesToIds == null)
			classesToIds = new HashMap();

		List classIds = (List) classesToIds.get(eClass);

		if (classIds == null) {
			classIds = new ArrayList();
			classesToIds.put(eClass, classIds);
		}

		classIds.addAll(ids);
	}

	protected void computeObjectsToDelete(EClass eClass, List ids) throws Exception {
		if (ids.isEmpty())
			return;

		ClassMetadata metadata = getMetadata(eClass);
		EReference[] containments = metadata.getContainmentReferences();
		int[] sourceIds = getArray(ids);

		for (int i = 0, l = containments.length; i < l; i++) {
			EReference reference = containments[i];
			DBCommand getIds = DBCommandFactory.INSTANCE.createGetIdsOfReferencedObjects(helper, dbMap, type, sourceIds, reference, false);
			List containedIds = (List) getIds.execute();
			EClass referenceType = reference.getEReferenceType();
			addToMap(referenceType, containedIds);
			computeObjectsToDelete(referenceType, containedIds);
		}
	}

	protected ClassMetadata getMetadata(EClass eClass) {
		if (classesToMetadata == null)
			classesToMetadata = new HashMap();

		ClassMetadata metadata = (ClassMetadata) classesToMetadata.get(eClass);

		if (metadata == null) {
			metadata = new ClassMetadata(eClass, dbMap, new RDBHelper());
			classesToMetadata.put(eClass, metadata);
		}

		return metadata;
	}

	protected int[] getArray(List integers) {
		int[] ints = new int[integers.size()];

		for (int i = 0, l = integers.size(); i < l; i++) {
			ints[i] = ((Integer) integers.get(i)).intValue();
		}

		return ints;
	}

	protected void deleteObjects() throws Exception {
		Iterator entries = classesToIds.entrySet().iterator();

		while (entries.hasNext()) {
			Map.Entry entry = (Map.Entry) entries.next();
			EClass eClass = (EClass) entry.getKey();
			List ids = (List) entry.getValue();
			deleteObjectsFromClassTables(eClass, ids);
		}
	}

	protected void deleteObjectsFromClassTables(EClass eClass, List ids) throws Exception {
		deleteObjectsFromClassTable(eClass, ids);
		ClassMetadata metadata = getMetadata(eClass);
		EClass[] superClasses = metadata.getAllSuperclasses();

		for (int i = 0; i < superClasses.length; i++)
			deleteObjectsFromClassTable(superClasses[i], ids);

		EClass[] subClasses = metadata.getAllSubclasses();

		for (int i = 0; i < subClasses.length; i++)
			deleteObjectsFromClassTable(subClasses[i], ids);
	}

	protected void deleteObjectsFromClassTable(EClass eClass, List ids) throws Exception {
		SQLStatement delete = StatementFactory.INSTANCE.createDeleteStatement(dbType,dbMap, eClass, ids, type);
		String statement = delete.getStatement();
		String[] statements;

		if (statement == null) {
			statements = ((DeleteStatement) delete).getStatements();
		} else {
			statements = new String[1];
			statements[0] = statement;
		}

		for (int i = 0; i < statements.length; i++) {
			if (debug)
				System.out.println(statements[i]);

			helper.executeUpdateStatement(statements[i]);
		}
	}
} // DeleteObjects
