/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.NoResourceTableException;
import org.eclipse.hyades.resources.database.internal.ResourceNotFoundException;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;
/**
 * This class obtains the ids of the top level objects of the resource with the
 * given URI. It returns a list of pairs of EClasses and lists of ids of top
 * level objects that are instances of that EClass.
 */
public class GetTopLevelObjects extends DBCommand {
	protected URI uri;
	protected Map classesToIds;

	public GetTopLevelObjects(JDBCHelper helper, DBMap map, URI uri) {
		super(helper, map);
		this.uri = uri;
	}

	/**
	 * @see org.eclipse.hyades.resources.database.internal.DBCommand#execute()
	 */
	public Object execute() throws Exception {
		Table resourceTable = dbMap.getResourceTable();

		if (resourceTable == null)
			throw new NoResourceTableException();

		String query = createResourceTableQuery(resourceTable);
		executeResourceTableQuery(query);
		return createTopLevelList();
	}

	protected String createResourceTableQuery(Table resourceTable) {
		StringBuffer query = new StringBuffer();
		query.append("SELECT ");
		query.append(addQuotes(DBMapBuilder.TABLE_COLUMN_NAME));
		query.append(", ");
		query.append(addQuotes(DBMapBuilder.ID_COLUMN_NAME));
		query.append(" FROM ");
		query.append(addQuotes(resourceTable.getName()));
		query.append(" WHERE ");
		query.append(addQuotes(DBMapBuilder.URI_COLUMN_NAME));
		query.append(" = '");
		query.append(uri.toString());
		query.append("'");
		return query.toString();
	}

	protected void executeResourceTableQuery(String query) throws Exception {
		Statement statement = helper.createStatement();
		ResultSet rs = helper.executeQuery(statement, query);
		classesToIds = new HashMap();

		while (rs.next()) {
			String table = rs.getString(1);
			String id = rs.getString(2);
			addToClassesToIds(table, id);
		}

		rs.close();
		statement.close();

		if (classesToIds.isEmpty())
			throw new ResourceNotFoundException(uri);
	}

	protected void addToClassesToIds(String tableName, String id) throws Exception {
		Table table = getTable(tableName);
		EClass eClass = dbMap.getClass(table);

		List ids = (List) classesToIds.get(eClass);

		if (ids == null) {
			ids = new ArrayList();
			classesToIds.put(eClass, ids);
		}

		ids.add(new Integer(Integer.parseInt(id)));
	}

	protected Table getTable(String name) {
		List tables = dbMap.getClassTables();

		for (int i = 0, l = tables.size(); i < l; i++) {
			Table table = (Table) tables.get(i);

			if (table.getName().equals(name))
				return table;
		}

		return null;
	}

	protected List createTopLevelList() {
		List topLevel = new ArrayList();
		Iterator entries = classesToIds.entrySet().iterator();

		while (entries.hasNext()) {
			Map.Entry entry = (Map.Entry) entries.next();
			topLevel.add(entry.getKey());
			topLevel.add(entry.getValue());
		}

		return topLevel;
	}
} // GetTopLevelObjects
