/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;
import java.sql.PreparedStatement;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;
/**
 * This class represents an INSERT SQL statement that adds rows to a table.
 * Subclasses of this class add rows to a class table, reference table, and
 * attribute table.
 * <p>
 * This class creates by default a prepared SQL statement for adding a row to
 * the tables, with the given table and row.
 */
public class InsertStatement extends SQLStatement {
	protected Table table;
	protected List columns;
	protected String[] statements;
	protected Map objectsToIds;
	protected String header;
	protected List statementList;
	protected boolean rowAdded;
	protected int rowNr;
	/**
	 * Constructor for InsertStatement.
	 */
	public InsertStatement(DBMap map, DatabaseType dbType, Map objectsToIds) {
		super(dbType, map);
		this.objectsToIds = objectsToIds;
		table = null;
		columns = null;
		rowAdded = false;
	}
	public InsertStatement(DatabaseType dbType, Table table, List columns) {
		super(dbType, null);
		this.table = table;
		this.columns = columns;
	}
	/**
	 * @see org.eclipse.hyades.resources.database.internal.SQLStatement#getStatement()
	 */
	public String getStatement() {
		statement = new StringBuffer();
		statement.append("INSERT INTO ");
		statement.append(addQuotes(table.getName()));
		statement.append(" (");
		for (int i = 0, l = columns.size(); i < l; i++) {
			Column column = (Column) columns.get(i);
			if (i != 0)
				statement.append(", ");
			statement.append(addQuotes(column.getName()));
		}
		statement.append(")");//\n");
		statement.append("  VALUES (");
		for (int i = 0, l = columns.size(); i < l; i++) {
			if (i != 0)
				statement.append(", ");
			statement.append("?");
		}
		statement.append(")");
		return statement.toString();
	}
	/**
	 * Creates an insert statement; if the statement exceeds the maximum
	 * maxSQLLength, then multiple insert statements are created and stored in
	 * the statements array.
	 */
	public String getStatement(List objects) throws Exception {
		if (header == null)
			header = createHeader();
		statement = new StringBuffer();
		statement.append(header);
		for (int i = 0, l = objects.size(); i < l; i++)
			createRows((EObject) objects.get(i), i == 0);
		if (statementList != null) {
			statementList.add(statement.toString());
			statements = (String[]) statementList.toArray(new String[statementList.size()]);
		}
		if (rowAdded) {
			if (statements == null)
				return statement.toString();
			else
				return null;
		} else {
			statements = new String[0];
			return null;
		}
	}
	protected void createRows(EObject object, boolean first) throws Exception {
		StringBuffer row = createRow(object);
		if (row != null)
			addRowToStatement(row, first);
	}

	/**
	 * Adds the row to the statement.
	 */
	protected void addRowToStatement(StringBuffer row, boolean first) {
		rowAdded = true;
		if (statementTooLong(row))
			processLongStatement(row);
		else {
			if (!first)
				statement.append(", ");//\n");
			statement.append(row);
		}
	}
	/**
	 * Add the current statement to the statement list. Set the statement buffer
	 * to be the header plus the given row.
	 */
	protected void processLongStatement(StringBuffer row) {
		if (statementList == null)
			statementList = new ArrayList();
		statementList.add(statement.toString());
		statement = new StringBuffer();
		statement.append(header);
		statement.append(row);
	}
	/**
	 * Creates the INSERT INTO table-name (columns) VALUES part of the INSERT
	 * statement.
	 */
	protected String createHeader() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("INSERT INTO ");
		buffer.append(addQuotes(getTable().getName()));
		buffer.append(" (");
		List columns = getColumns();
		for (int i = 0, l = columns.size(); i < l; i++) {
			Column column = (Column) columns.get(i);
			if (i != 0)
				buffer.append(", ");
			buffer.append(addQuotes(column.getName()));
		}
		buffer.append(") VALUES ");//\n");
		return buffer.toString();
	}
	protected String createParameterMarkers() {
		StringBuffer buffer = new StringBuffer();
		List columns = getColumns();
		buffer.append('(');
		for (int i = 0, l = columns.size(); i < l; i++) {
			if (i != 0)
				buffer.append(", ");
			buffer.append('?');
		}
		buffer.append(')');
		return buffer.toString();
	}
	protected Table getTable() {
		return table;
	}
	protected List getColumns() {
		return columns;
	}
	/**
	 * Creates a row based on the EObject.
	 */
	protected StringBuffer createRow(EObject object) throws Exception {
		return null;
	}
	/**
	 * Returns true if adding the row to the statement results in a statement
	 * that exceeds the maximum maxSQLLength.
	 */
	protected boolean statementTooLong(StringBuffer row) {
		return statement.length() + row.length() > dbType.getMaximumSQLLength();
	}
	public String[] getStatements() {
		return statements;
	}
	protected void appendValue(StringBuffer buffer, Object value, int sqlType) {
		if (value == null)
			buffer.append("NULL");
		else if (sqlType == Types.CHAR || sqlType == Types.VARCHAR || sqlType == Types.LONGVARCHAR) {
			buffer.append("'");
			String valueString = value.toString();
			for (int i = 0, l = valueString.length(); i < l; i++) {
				char c = valueString.charAt(i);
				if (c == '\'')
					buffer.append("''");
				else
					buffer.append(c);
			}
			buffer.append("'");
		} else {
			buffer.append(value.toString());
		}
	}
	protected void appendBatchedValue(PreparedStatement ps, int colPos, Object value, int sqlType) throws Exception {
		if (value == null) {
			ps.setNull(colPos, sqlType);
			if (debugPreparedStatement) {
				statement.append(',');
				statement.append("NULL");
			}
		} else if (sqlType == Types.CHAR || sqlType == Types.VARCHAR || sqlType == Types.LONGVARCHAR) {
			String valueString = value.toString();
			StringBuffer buffer = new StringBuffer();
			for (int i = 0, l = valueString.length(); i < l; i++) {
				char c = valueString.charAt(i);
				if (c == '\'')
					buffer.append("''");
				else
					buffer.append(c);
			}
			ps.setString(colPos, buffer.toString());
			if (debugPreparedStatement) {
				statement.append(',');
				statement.append(buffer.toString());
			}
		} else if (sqlType == Types.INTEGER) {
			ps.setInt(colPos, ((Integer) value).intValue());
			if (debugPreparedStatement) {
				statement.append(',');
				statement.append(value);
			}
		} else if (sqlType == Types.DOUBLE) {
			ps.setDouble(colPos, ((Double) value).doubleValue());
			if (debugPreparedStatement) {
				statement.append(',');
				statement.append(value);
			}
		} else if (sqlType == Types.SMALLINT) {
			ps.setShort(colPos, ((Short) value).shortValue());
			if (debugPreparedStatement) {
				statement.append(',');
				statement.append(value);
			}
		} else {
			ps.setObject(colPos, value);
			if (debugPreparedStatement) {
				statement.append(',');
				statement.append(value);
			}
		}
	}
	/**
	 * @param objects
	 * @return
	 */
	public PreparedStatement getPreparedStatement(JDBCHelper helper, List objects) throws Exception {
		if (header == null)
			header = createHeader();
		statement.setLength(0);
		statement.append(header);
		statement.append(createParameterMarkers());
		if (statementList == null)
			statementList = new ArrayList();
		else
			statementList.clear();
		statementList.add(statement.toString());
		PreparedStatement ps = helper.createPreparedStatement(statement.toString());
		PerfUtil p = new PerfUtil("InsertStatement.getPreparedStatement() sql=" + statement.toString() + ", size=" + objects.size(), true);
		rowNr=1;
		for (int i = 0; i < objects.size(); i++) {
			if(debugPreparedStatement)
				statement.setLength(0);
			addBatchedRows(ps, (EObject) objects.get(i));
			if(debugPreparedStatement)
				statementList.add(statement.toString());
		}
		p.stopAndPrintStatus();
		statements = (String[]) statementList.toArray(new String[statementList.size()]);
		return ps;
	}
	/**
	 * @param ps
	 */
	protected void addBatchedRow(PreparedStatement ps, EObject object) throws Exception {
		// override in subclasses
	}
	protected void addBatchedRows(PreparedStatement ps, EObject object) throws Exception {
		addBatchedRow(ps, object);
	}
} // InsertStatement
